/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.validation.internal.smlvalidators;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import org.eclipse.cosmos.rm.validation.internal.artifacts.ElementModel;
import org.eclipse.cosmos.rm.validation.internal.artifacts.ElementSchemaModel;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationOutput;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.validation.internal.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.validation.internal.core.IValidator;
import org.eclipse.cosmos.rm.validation.internal.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.validation.internal.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.ElementSourceBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.IDataBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.SMLValidatingBuilder;
import org.eclipse.cosmos.rm.validation.internal.databuilders.SchemaBindingDataBuilder;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.SAXParseException;

/**
 * Validate SML instances against their schema and report any errors found.
 * 
 * @author David Whiteman
 */
public class SchemaValidator extends AbstractSMLValidator /*implements LexicalHandler*/ {

	private static final String VALIDATION_ERROR_PREFIX = "cvc";
	private Map<String, ElementModel> schemas;
	private IValidationOutput<String, Object> validationLogger;
	private boolean smlIfInput;
	private String input;
	private IDataBuilder<?> elementSourceBuilder;
	private ElementSchemaModel currentSchema;
	private ArrayList<ElementSchemaModel> instanceSchemas;
	
	public boolean validate() 
	{
		init();
 
		if (!validateSchemas()) {
			return false;
		}

		return validateInstances();
	}


	private void init() {
		// Setup task name and fields
		
		setTaskName(SMLValidationMessages.validationSchema);		
		validationLogger = getValidationOutput();
		elementSourceBuilder = DataBuilderRegistry.instance().getDataStructureBuilder(ElementSourceBuilder.ID);
		schemas = ((ElementSourceBuilder) elementSourceBuilder).getSchemas();
		instanceSchemas = new ArrayList<ElementSchemaModel>();
	}


	/**
	 * Validate all instances against the set of schemas
	 * 
	 * @return success of the validation
	 */
	private boolean validateInstances() {
		boolean success = true;
		try 
		{			
			ElementModel currentInstance = null;
			InputStream[] schemas = buildSchemaSource();
			if (schemas.length <= 0)
			{
				return true;
			}
			SAXParser newSaxParser = SMLValidatorUtil.createSAXParser(schemas, true, null, validationLogger);
			for (Iterator<ElementModel> iter = ((ElementSourceBuilder) elementSourceBuilder).getInstances().iterator(); iter.hasNext();) 
			{				
				currentInstance = iter.next();
				StringBuffer buffer = new StringBuffer(currentInstance.getSource());
				try 
				{
					newSaxParser.parse(new InputSource(new ByteArrayInputStream(buffer.toString().getBytes())), new SMLValidatingBuilder());
				} 
				catch (SAXParseException e) 
				{
					reportParsingError(currentInstance, e);
					success = false;
					if (shouldAbortOnError()) 
					{
						return success;
					}
				}
			}
		} 
		catch (SAXException e) 
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, e.getLocalizedMessage()));
			success = false;
			if (shouldAbortOnError()) {
				return success;
			}
		} 
		catch (ParserConfigurationException e) 
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, e.getLocalizedMessage()));
			success = false;
			if (shouldAbortOnError()) {
				return success;
			}
		} 
		catch (IOException e) 
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, e.getLocalizedMessage()));
			success = false;
			if (shouldAbortOnError()) {
				return success;
			}
		}
		return success;
	}

	/**
	 * Validate all schemas against each other to ensure they are correct.
	 * 
	 * @return success of the validation
	 */
	private boolean validateSchemas() {
		// Iterate over schemas to build InputStream array expected by
		// validating parser
		//SMLValidatingBuilder validatingBuilder = new SMLValidatingBuilder();
		boolean success = true;
		currentSchema = null;
		for (Iterator<ElementModel> iterator = schemas.values().iterator(); iterator.hasNext();) {
			try {
				validateSchema((ElementSchemaModel) iterator.next());
			} catch (SAXNotRecognizedException e1) {
				// unexpected condition - should probably handle this better
				e1.printStackTrace();
			} catch (SAXNotSupportedException e1) {
				// unexpected condition - should probably handle this better
				e1.printStackTrace();
			} catch (ParserConfigurationException e1) {
				// unexpected condition - should probably handle this better
				e1.printStackTrace();
			} catch (SAXParseException e1) {
				reportParsingError(currentSchema, e1);
				success = false;
				if (shouldAbortOnError()) {
					return success;
				}
			} catch (SAXException e1) {
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, e1.getLocalizedMessage()));
				success = false;
				if (shouldAbortOnError()) {
					return success;
				}
			} catch (IOException e) {
				
				success = false;
				e.printStackTrace();
			} 
		}
		return success;
	}
	
	// Recursive method validates schema against all
	// prerequsite schemas
	private void validateSchema(ElementSchemaModel schema) throws SAXException, IOException, ParserConfigurationException 
	{
		if (schema.hasBeenValidated()) {
			// No need to validate a schema twice
			return;
		}
		ArrayList<ElementModel> schemasForValidating = new ArrayList<ElementModel>();
		instanceSchemas.add(schema);
		for (Iterator<String> iterator = schema.getPrerequisiteNamespaces().iterator(); iterator.hasNext();) {
			String prereq = (String) iterator.next();
			ElementSchemaModel prereqModel = (ElementSchemaModel) schemas.get(prereq);
			if (prereqModel != null) {
				schemasForValidating.add(prereqModel);
				try {
					// ensure prereq has been validated before we validate the current schema
					validateSchema(prereqModel);
				} catch (SAXException e) {
					throw e;
				} 
			}
		}
		currentSchema = schema;	
		InputStream xmlSchemaStream = null;
		try 
		{
			List<InputStream> schemaSourcesList = new ArrayList<InputStream>();
			xmlSchemaStream = this.getClass().getClassLoader().getResourceAsStream("validation-resources/XMLSchema.xsd");
			
			// Ensure it conforms to the XML Schema 1.0 schema
			schemaSourcesList.add(xmlSchemaStream);
			SAXParser schemaParser = SMLValidatorUtil.createSAXParser(primBuildSchemaSource(schemaSourcesList, schemasForValidating), true, null, validationLogger);
			schemaParser.parse(new InputSource(new ByteArrayInputStream(currentSchema.getSource().getBytes())), new SMLValidatingBuilder());
		} 
		catch (SAXException e) 
		{
			throw e;
		} 
		finally 
		{
			currentSchema.markValidated();
			xmlSchemaStream.close();
		}
	}


	private void reportParsingError(ElementModel currentInstance,
			SAXParseException e) {
		// TODO schemaLocation is currently being flagged as an error?
		boolean validationError = (currentInstance instanceof ElementSchemaModel) || (e.getMessage().startsWith(VALIDATION_ERROR_PREFIX));
		int lineNumber = -1;
		ElementModel errorLocation = null;
		
		// TODO line number lookup can move to the ElementSourceBuilder so
		// other validators can use it, such as the schematron validator
		if (validationError) 
		{
			errorLocation = currentInstance;
			lineNumber = currentInstance.getLineNumber() + e.getLineNumber();
		} 
		else 
		{
			// TODO this needs to be reimplemented since we haven't been able yet to
			// identify the correct location of a validation error in a set of schemas.
			// For now, we will just mark the instance that exposed the schema error.
			errorLocation = currentInstance;
			lineNumber = smlIfInput ? currentInstance.getLineNumber() : -1;
		}
		
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(errorLocation == null ? null : errorLocation.getFilePath(), lineNumber, e.getColumnNumber(), e.getLocalizedMessage()));
	}


	private InputStream[] buildSchemaSource()
	{
		Collection<ElementModel> collectionSchemas = schemas.values();
		List<ElementModel> sortedSchemas = new ArrayList<ElementModel>();
		
		for (Iterator<ElementModel> schemaElements = collectionSchemas.iterator(); schemaElements.hasNext();) 
		{
			ElementSchemaModel schemaElement = (ElementSchemaModel)schemaElements.next();
			addSchema(sortedSchemas, schemaElement);
		}
		
		return primBuildSchemaSource(null, sortedSchemas);
	}

	private void addSchema(List<ElementModel> sortedSchemas, ElementSchemaModel schemaElement) 
	{
		if (sortedSchemas.contains(schemaElement))
		{
			return;
		}
		
		List<String> importedSchemas = schemaElement.getImportedSchemas();
		for (int i = 0, importCount = importedSchemas.size(); i < importCount; i++)
		{
			ElementSchemaModel importedElement = (ElementSchemaModel)schemas.get(importedSchemas.get(i));
			if (importedElement != null)
			{
				addSchema (sortedSchemas, importedElement);
			}			
		}
		sortedSchemas.add(schemaElement);
	}


	private InputStream[] primBuildSchemaSource(List<InputStream> schemaSourcesList, Collection<ElementModel> localSchemas) {
		// Iterate over schemas to build InputStream array expected by
		// validating parser
		if (schemaSourcesList == null) {
			schemaSourcesList = new ArrayList<InputStream>();
		}
		for (Iterator<ElementModel> iter = localSchemas.iterator(); iter.hasNext();) 
		{
			ElementModel schemaModel = iter.next();
			StringBuffer buffer = new StringBuffer();
			buffer.append(schemaModel.getSource());
			
			schemaSourcesList.add(new ByteArrayInputStream(buffer.toString().getBytes()));
			
		}		
		return schemaSourcesList.toArray(new InputStream[schemaSourcesList.size()]);
	}


	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);
		
		smlIfInput = !(validationAttribute.get(IValidator.ATTRIBUTE_INPUT_TYPE) instanceof String) || !IValidator.VALUE_SML_UNIT.equals(validationAttribute.get(IValidator.ATTRIBUTE_INPUT_TYPE));		
		input = (String)validationAttribute.get(IValidator.ATTRIBUTE_INSTANCE);
		
		DataBuilderRegistry builderRegistry = DataBuilderRegistry.instance();
		builderRegistry.registerDataStructureBuilder(DocumentDOMBuilder.ID, new DocumentDOMBuilder());
		builderRegistry.registerDataStructureBuilder(ElementSourceBuilder.ID, new ElementSourceBuilder(
				IValidator.VALUE_ENV_ECLIPSE.equals(validationAttribute.get(ATTRIBUTE_ENV)),
				smlIfInput ? input : null));
		builderRegistry.registerDataStructureBuilder(SchemaBindingDataBuilder.ID, new SchemaBindingDataBuilder());
	}

}
