/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.smlif.internal.actions;

import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.operations.ISMLValidateOperation;
import org.eclipse.cosmos.rm.smlif.internal.SMLPlugin;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLRepositoryManager;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.validation.internal.core.IValidator;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PartInitException;

/**
 * Represents the validate context menu action that is applicable to the project,
 * folder, or files selected.
 * 
 * @author Ali Mehregani
 */
public class ValidateResourceAction extends AbstractAction
{	
	/**
	 * The validation attributes
	 */
	private Map<String, Object> attributes;
	
	/**
	 * Constructor
	 */
	public ValidateResourceAction()
	{
		attributes = new Hashtable<String, Object>();
		attributes.put(IValidator.ATTRIBUTE_ENV, IValidator.VALUE_ENV_ECLIPSE);		
	}
		
	
	/**
	 * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
	 */
	public void run(IAction action)
	{
		if (noValidSelection())
			return;
			
		SMLValidationJob validate = new SMLValidationJob(SMLValidationMessages.validationJob);
		validate.setUser(true);
		validate.schedule();
	}

	private boolean runValidation(ISMLValidateOperation validationOperation, String inputType, String input) throws RepositoryOperationException
	{
		attributes.put(IValidator.ATTRIBUTE_INPUT_TYPE, inputType);
		attributes.put(IValidator.ATTRIBUTE_INSTANCE, input);
		validationOperation.setArguments(new Object[]{attributes});
		validationOperation.run();
		return validationOperation.getStatus();
	}
	
	
	/**
	 * The SML validation job
	 * 
	 * @author Ali Mehregani
	 */
	private class SMLValidationJob extends Job
	{
		public SMLValidationJob(String name)
		{
			super(name);
		}

		protected IStatus run(IProgressMonitor monitor)
		{	
			try
			{
				ISMLRepository repository = SMLRepositoryManager.getRepository(getRootDirectory());
				ISMLValidateOperation validationOperation = (ISMLValidateOperation)repository.getOperation(ISMLValidateOperation.ID);
					
				List<?> smlifDocuments = getSmlifdocuments();
				attributes.put(IValidator.ATTRIBUTE_PROGRESS_MONITOR, monitor);
				attributes.put(IValidator.ATTRIBUTE_INPUT_TYPE, IValidator.VALUE_SML_IF);
				boolean status = true;
				try
				{
					int smlifDocumentsCount = smlifDocuments.size();
					monitor.beginTask(SMLValidationMessages.validationTask, IValidationConstants.VALIDATION_WORK * (smlifDocumentsCount + 1));
					for (int i = 0; i < smlifDocumentsCount; i++)
					{
						status = runValidation(validationOperation, IValidator.VALUE_SML_IF, (String)smlifDocuments.get(i)) && status;
					}
					
					if (getSmlModelUnitPaths().length() > 0)
					{
						status = runValidation(validationOperation, IValidator.VALUE_SML_UNIT, getSmlModelUnitPaths()) && status;
					}
					else
					{
						monitor.worked(IValidationConstants.VALIDATION_WORK);
					}
				}
				catch (RepositoryOperationException roe)
				{
					SMLCommonUtil.openErrorWithDetail(SMLValidationMessages.errorValidationTitle, SMLValidationMessages.errorValidation, roe);
					return Status.CANCEL_STATUS;
				}
				
				/* Display a dialog notifying the user whether or not there were errors */
				final IWorkbench workbench = SMLPlugin.getDefault().getWorkbench();
				if (status)
				{
					workbench.getDisplay().syncExec(new Runnable()
					{

						public void run()
						{
							MessageDialog.openInformation(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
									SMLValidationMessages.confirmNoErrorTitle, SMLValidationMessages.confirmNoError);
						}
					});
				}
				else
				{
					// Display the problems view										
					workbench.getDisplay().syncExec(new Runnable(){

						public void run()
						{
							try
							{
								workbench.getActiveWorkbenchWindow().getActivePage().showView(IPageLayout.ID_PROBLEM_VIEW);
							} 
							catch (PartInitException e)
							{
								MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
									SMLValidationMessages.failedValidationTitle, SMLValidationMessages.failedValidation);
							}
						}
					});							
				}		
			
				return Status.OK_STATUS;
			}
			finally
			{
				monitor.done();
			}
		}
			
	}
}
