/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.smlif.internal.editor;


import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.cosmos.rm.smlif.internal.SMLImages;
import org.eclipse.cosmos.rm.smlif.internal.SMLPlugin;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLMessages;
import org.eclipse.cosmos.rm.smlif.internal.dialogs.ResourceSelectionDialog;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.validation.internal.common.XMLInternalUtility;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.MenuAdapter;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.dialogs.FilteredTree;
import org.eclipse.ui.dialogs.PatternFilter;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Section;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class DocumentsPage extends AbstractFormPage implements SelectionListener, ISelectionChangedListener
{
	private static final int INSTANCES_CHOICE = 3;

	private static final int DEFINITIONS_CHOICE = 2;

	/**
	 * The id of this page
	 */
	public static final String ID = SMLPlugin.PLUGIN_ID + ".DocumentsPage"; //$NON-NLS-1$
	
	/**
	 * Operations
	 */
	private static final byte OPERATION_ADD_DOCUMENT = 0x00;
	private static final byte OPERATION_REMOVE_DOCUMENT = 0x01;
	private static final byte OPERATION_ADD_ALIASES = 0x02;
	private static final byte OPERATION_REMOVE_ALIASES = 0x03;
	private static final byte OPERATION_MODIFY_ALIAS = 0x04;
	
	/**
	 * The aliases table
	 */
	private Table aliasTable;
	
	/**
	 * The style text representing the raw representation of the document
	 */
	private StyledText rawStyledText;

	/**
	 * The logical representation of a document
	 */
	private TreeViewer logicalRepresentation;

	/**
	 * The node representing the current document selection
	 */
	private Node input;

	/**
	 * The add alias button
	 */
	private Button addAlias;

	/**
	 * The remove alias button
	 */
	private Button removeAlias;


	/**
	 * Keeps track of the indices for an element selected in
	 * the logical tree representation of a document.
	 * KEY = An object of type {@link String} indicating the index of an element
	 * VALUE = int[2], where int[0] indicates the starting index and int[1]
	 * indicates the end index.
	 */
	private Map<String, int[]> indices;
	
	/**
	 * The logical tree viewer
	 */
	private TreeViewer documentTreeViewer;

	
	/**
	 * The alias table viewer
	 */
	private CheckboxTableViewer tableViewer;

	private FilteredTree documentTree;

	/**
	 * Constructor
	 * 
	 * @param editor The editor containing this form page
	 */
	public DocumentsPage(FormEditor editor)
	{
		super(editor, ID, SMLMessages.editorDocumentTitle);
		indices = new Hashtable<String, int[]>();
	}
	
	protected void setWeight(SashForm sashForm)
	{
		sashForm.setWeights(new int[]{40, 60});
	}
	
	protected void createLeftColumn(Composite parent)
	{
 		getManagedForm().addPart(((SMLIFEditor)getEditor()).getSmlifFormPart());
		FormToolkit toolkit = getManagedForm().getToolkit();
		Section section = createSection(parent);
		
		Composite sectionClient = toolkit.createComposite(section, SWT.NONE);
		sectionClient.setLayout(new GridLayout());
		sectionClient.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		toolkit.createLabel(sectionClient, SMLMessages.editorFilterLabel);

		/* The tree */
		documentTree = new FilteredTree(sectionClient, SWT.BORDER | SWT.SINGLE, new PatternFilter());
		documentTreeViewer = documentTree.getViewer();		
		documentTreeViewer.setContentProvider(new DocumentContentProvider());
		documentTreeViewer.setLabelProvider(new DocumentLabelProvider());		
		documentTreeViewer.addSelectionChangedListener(this);
		documentTreeViewer.setSorter(new ViewerSorter() {
			public int compare(Viewer viewer, Object e1, Object e2) {
				if ((e1 instanceof DocumentTreeElement) && (e2 instanceof DocumentTreeElement)) {
					return ((DocumentLabelProvider)documentTreeViewer.getLabelProvider()).getText(e1).compareTo(((DocumentLabelProvider)documentTreeViewer.getLabelProvider()).getText(e2));
				}
				return super.compare(viewer, e1, e2);
			}
		});
		documentTreeViewer.setInput(getEditor());


		/* The context menu of the document tree */
		Menu documentContextMenu = new Menu(documentTreeViewer.getTree());
		documentTreeViewer.getTree().setMenu(documentContextMenu);
		
		final MenuItem addDocument = createMenuItem(documentContextMenu, SMLMessages.editorAddDocument, OPERATION_ADD_DOCUMENT);	
		final MenuItem removeDocument = createMenuItem(documentContextMenu, SMLMessages.editorRemoveDocument, OPERATION_REMOVE_DOCUMENT);	
	
		documentContextMenu.addMenuListener(new MenuAdapter()
		{

			public void menuShown(MenuEvent e)
			{
				DocumentTreeElement selection = (DocumentTreeElement)((StructuredSelection)documentTreeViewer.getSelection()).getFirstElement();
				boolean noSelection = (selection == null);
				boolean rootSelected = selection != null && selection.isRootElement();
				
				addDocument.setEnabled(rootSelected || noSelection);
				addDocument.setImage((rootSelected || noSelection) ? SMLImages.INSTANCE.getImage("c", SMLImages.ADD) : SMLImages.INSTANCE.getImage("d", SMLImages.ADD)); //$NON-NLS-1$ //$NON-NLS-2$
					
				removeDocument.setEnabled(!rootSelected);
				removeDocument.setImage(!rootSelected ? SMLImages.INSTANCE.getImage("c", SMLImages.REMOVE) : SMLImages.INSTANCE.getImage("d", SMLImages.REMOVE)); //$NON-NLS-1$ //$NON-NLS-2$
			}
		});
		
		makeSectionReady (section, sectionClient, SMLMessages.editorDocuments);
	}
	
	private MenuItem createMenuItem(Menu parent, String text, byte operation)
	{
		final MenuItem menuItem = new MenuItem(parent, SWT.NONE);
		menuItem.setText(text);
		menuItem.addSelectionListener(this);
		menuItem.setData(new Byte(operation));
		
		return menuItem;
	}

	private Button createButton(Composite parent, String text, SelectionListener listener, byte operation, boolean enabled)
	{
		Button button = getManagedForm().getToolkit().createButton(parent, text, SWT.NONE);
		button.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		button.addSelectionListener(listener);
		button.setData(new Byte(operation));
		button.setEnabled(enabled);
		
		GridData layoutData = new GridData(SWT.DEFAULT, SWT.DEFAULT, true, false);
		layoutData.widthHint = 100;
		layoutData.horizontalAlignment = SWT.RIGHT;
		button.setLayoutData(layoutData);
		
		return button;
	}


	/**
	 * @see org.eclipse.cosmos.rm.smlif.internal.editor.AbstractFormPage#createRightColumn(org.eclipse.swt.widgets.Composite)
	 */
	protected void createRightColumn(Composite parent)
	{
		SMLFormToolkit toolkit = ((SMLIFEditor)getEditor()).getExtendedToolkit();		
		SashForm sashForm = toolkit.createSashForm(parent, SWT.VERTICAL);
		sashForm.setLayout(new GridLayout());
		sashForm.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		Section logicalSection = createSection(sashForm);
		
		Composite logicalClient = toolkit.createComposite(logicalSection);
		logicalClient.setLayout(new GridLayout());
		logicalClient.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		toolkit.createLabel(logicalClient, SMLMessages.editorAliases);
		
		Composite aliasComposite = toolkit.createComposite(logicalClient);
		GridLayout gridLayout = new GridLayout(2, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		aliasComposite.setLayout(gridLayout);
		aliasComposite.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		
		aliasTable = toolkit.createTable(aliasComposite, SWT.MULTI);
		aliasTable.setLinesVisible(true);
		aliasTable.setHeaderVisible(true);						
		setTableProperties();
		
		Composite buttonComposite = toolkit.createComposite(aliasComposite);
		gridLayout = new GridLayout();
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		buttonComposite.setLayout(gridLayout);
		GridData buttonCompositeGD = new GridData();
		buttonCompositeGD.verticalAlignment = SWT.TOP;
		buttonCompositeGD.horizontalAlignment = SWT.RIGHT;
		buttonComposite.setLayoutData(buttonCompositeGD);
		addAlias = createButton(buttonComposite, SMLMessages.editorAddButton, this, OPERATION_ADD_ALIASES, false);
		removeAlias = createButton(buttonComposite, SMLMessages.editorRemoveButton, this, OPERATION_REMOVE_ALIASES, false);
		aliasTable.addFocusListener(new FocusListener(){

			public void focusGained(FocusEvent e)
			{
				removeAlias.setEnabled(aliasTable.getSelection() != null && aliasTable.getSelection().length > 0);				
			}

			public void focusLost(FocusEvent e)
			{
				removeAlias.setEnabled(aliasTable.getSelection() != null && aliasTable.getSelection().length > 0);
				
			}
		});
		aliasTable.addSelectionListener(new SelectionAdapter()
		{
			public void widgetSelected(SelectionEvent e)
			{
				removeAlias.setEnabled(aliasTable.getSelection() != null && aliasTable.getSelection().length > 0);
			}
		});

			
		
		toolkit.createLabel(logicalClient, SMLMessages.editorLogicalTree);
		Tree logicalTree = toolkit.createTree(logicalClient, SWT.NONE);
		logicalTree.setLayout(new GridLayout());
		logicalTree.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		logicalRepresentation = new TreeViewer(logicalTree);
		logicalRepresentation.setContentProvider(new LogicalContentProvider());
		logicalRepresentation.setLabelProvider(new LogicalLabelProvider());
		logicalRepresentation.addSelectionChangedListener(new LogicalSelectionListener());
		makeSectionReady(logicalSection, logicalClient, SMLMessages.editorLogicalRepresentation);
		
		Section rawSection = createSection (sashForm);
		Composite rawSectionClient = toolkit.createComposite(rawSection);
		rawSectionClient.setLayout(new GridLayout());
		rawSectionClient.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		SMLIFEditor editor = (SMLIFEditor) getEditor();
		rawStyledText = editor.getExtendedToolkit().createStyledtext(rawSectionClient, null, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		rawStyledText.setLayout(new GridLayout());
		rawStyledText.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		rawStyledText.addLineStyleListener(new XMLLineStyler());
		rawStyledText.setFont(new Font(Display.getDefault(), "Courier New", 10, SWT.NORMAL)); //$NON-NLS-1$
		rawStyledText.setEditable(false);
		
		makeSectionReady(rawSection, rawSectionClient, SMLMessages.editorXMLRepresentation);
	}

	
	private void setTableProperties()
	{
		TableLayout layout = new TableLayout();
		aliasTable.setLayout(layout);
		GridData aliasTableGD = new GridData(SWT.FILL, SWT.FILL, true, true);
		aliasTableGD.heightHint = 50;
		aliasTable.setLayoutData(aliasTableGD);
				
		ColumnPixelData columnLayout = new ColumnPixelData(300, true);
		layout.addColumnData(columnLayout);
		TableColumn column = new TableColumn(aliasTable, SWT.NONE, 0);
		column.setResizable(columnLayout.resizable);
		column.setText(SMLMessages.editorName);				
		
		tableViewer = new CheckboxTableViewer(aliasTable);
		tableViewer.setContentProvider(new IStructuredContentProvider()
		{
			private Object input;
			
			public void dispose()
			{
			}

			public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
			{
				this.input = newInput;				
			}

			public Object[] getElements(Object inputElement)
			{
				if (!(input instanceof Node))
					return new Object[0];
				
				Node aliasesNode = (Node)input;
				NodeList aliases = null;
				if (aliasesNode == null || (aliases = aliasesNode.getChildNodes()) == null || aliases.getLength() <= 0)
					return new Object[0];
					
				List<String> aliasTxt = new ArrayList<String>();
				for (int i = 0, aliasCount = aliases.getLength(); i < aliasCount; i++)
				{
					Node currentAlias = aliases.item(i);
					if (currentAlias.getNodeType() == Node.TEXT_NODE)
						continue;
					
					String alias = SMLCommonUtil.retrieveTextNode(currentAlias);					
					aliasTxt.add(alias);
				}
				return aliasTxt.toArray(new String[aliasTxt.size()]);
			}
		});
		
		class AliasLabelProvider extends LabelProvider implements ITableLabelProvider
		{

			public Image getColumnImage(Object element, int columnIndex)
			{				
				return null;
			}

			public String getColumnText(Object element, int columnIndex)
			{
				return element.toString();
			}			
		}
		
		tableViewer.setLabelProvider(new AliasLabelProvider());				
		
		/* Add the cell editors */	
		tableViewer.setCellEditors(new CellEditor[]{new TextCellEditor(aliasTable)});
		tableViewer.setCellModifier(new ICellModifier()
		{
			public boolean canModify(Object element, String property)
			{
				return true;
			}

			public Object getValue(Object element, String property)
			{
				TableItem[] tableItem = aliasTable.getSelection();
				if (tableItem == null || tableItem.length != 1)
					return null;
			
				return tableItem[0].getText(0);
			}

			public void modify(Object element, String property, Object value)
			{				
				TableItem[] tableItem = aliasTable.getSelection();
				if (tableItem == null || tableItem.length != 1 || tableItem[0].getText().equals((String)value))
					return;
				
				handleOperation (OPERATION_MODIFY_ALIAS, new Object[]{value});
			}
			
		});		
		final String ALIAS_COLUMN = "ALIAS_COLUMN"; //$NON-NLS-1$
		tableViewer.setColumnProperties(new String[]{ALIAS_COLUMN});
	}


	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e)
	{
		/* Doesn't need to be implemented */
	}


	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e)
	{
		handleOperation(((Byte)e.widget.getData()).byteValue(), null);
	}	

	
	private boolean handleOperation(byte operation, Object[] arguments)
	{
		switch (operation)
		{
			case OPERATION_ADD_DOCUMENT:
			{
				/* Display the resource dialog */
				Tree documentTree = documentTreeViewer.getTree();
				TreeItem[] selection = documentTree.getSelection();
				boolean noSelection = (selection == null || selection.length <= 0);
				
				boolean definitionsSelected = false;
				if (noSelection) {
					// prompt for definitions vs. instances
					int result = promptForDocumentType();
					if (result == Window.CANCEL) {
						return false;
					}
					definitionsSelected = (result == DEFINITIONS_CHOICE);
				} else {
					definitionsSelected = documentTree.indexOf(selection[0]) == 0;
				}
				ResourceSelectionDialog resourceDialog = new ResourceSelectionDialog(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), definitionsSelected ? ISMLConstants.TYPE_DEFINITION : ISMLConstants.TYPE_INSTANCE);
				resourceDialog.setMultiSelection(true);
				IFile[] selectedFile = resourceDialog.open();
				if (selectedFile == null || selectedFile.length <= 0)
					return false;

				SMLIFEditor editor = (SMLIFEditor)getEditor();
				String parentNodeName = definitionsSelected ? ISMLConstants.DEFINITIONS_ELEMENT: ISMLConstants.INSTANCES_ELEMENT;
				Node rootNode = SMLCommonUtil.nestedNodeRetrieval(editor.getDocumentNode(), new String[][]{
							new String[] {ISMLConstants.SMLIF_URI, parentNodeName}
						}, true, false, null);
//				Node rootNode = editor.retrieveChildNode(ISMLConstants.SMLIF_URI, definitionsSelected ? ISMLConstants.DEFINITIONS_ELEMENT : ISMLConstants.INSTANCES_ELEMENT);
				if (rootNode == null)
				{
					MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
							SMLMessages.errorAddingDocumentTitle, SMLMessages.errorAddingDocument0);
					return false;
				}					
					
				IFile file = null;
				
				/* For every selected item */
				try
				{
					for (int i = 0; i < selectedFile.length; i++)
					{
						file = selectedFile[i];
						Node documentNode = SMLCommonUtil.createElement(rootNode, ISMLConstants.SMLIF_URI, ISMLConstants.DOCUMENT_ELEMENT);
						Node dataNode = SMLCommonUtil.createElement(documentNode, ISMLConstants.SMLIF_URI, ISMLConstants.DATA_ELEMENT);		
						adoptNode (dataNode.getOwnerDocument(),
								dataNode,
								XMLInternalUtility.domParseDocument(file.getContents(), false, true).getFirstChild());						
					}
				}
				catch (Exception e)
				{ 
					SMLCommonUtil.openErrorWithDetail(SMLMessages.errorAddingDocumentTitle, 
							NLS.bind(SMLMessages.errorAddingDocument1,
									file == null ? IValidationConstants.EMPTY_STRING : file.getName()), e);
					return false;
				} 
				
				endOperation(true, definitionsSelected);				
				break;
			}
			case OPERATION_REMOVE_DOCUMENT:
			{
				Tree documentTree = documentTreeViewer.getTree();
				TreeItem[] selections = documentTree.getSelection();
				if (selections == null || selections.length <= 0)
					return false;
				
				if (!MessageDialog.openQuestion(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(),
						SMLMessages.editorRemoveDocumentTitle, SMLMessages.editorRemoveDocumentQuestion))
				{
					return false;
				}
				
				TreeItem parentItem = selections[0].getParentItem();
				int documentIndex = parentItem == null ? -1 : parentItem.indexOf(selections[0]);
				SMLIFEditor editor = (SMLIFEditor)getEditor();
				Node rootNode = documentIndex < 0 ? null : editor.retrieveChildNode(ISMLConstants.SMLIF_URI, documentTree.indexOf(parentItem) == 0 ? ISMLConstants.DEFINITIONS_ELEMENT : ISMLConstants.INSTANCES_ELEMENT);
				Node[] rootChildren = null;
				if (rootNode == null || documentIndex < 0 || (rootChildren = SMLValidatorUtil.retrieveChildElements(rootNode)) == null || documentIndex >= rootChildren.length)
				{
					MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(),
						SMLMessages.errorRemovingDocumentTitle, SMLMessages.errorRemovingDocument0);
				}
				rootNode.removeChild(rootChildren[documentIndex]);
				
				endOperation(true, documentTree.indexOf(parentItem) == 0);							
				break;
			}
			case OPERATION_ADD_ALIASES:
			{
				// Retrieve the data element first so we can indicate we want the docinfo element before it
				Node dataElement = SMLCommonUtil.nestedNodeRetrieval(input, new String[][]{
						new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.DATA_ELEMENT}
					}, false, false, null);
				// Retrieve the docinfo element, indicating that it should precede the data element in sibling order.
				Node docInfoElement = SMLCommonUtil.nestedNodeRetrieval(input, new String[][]{
						new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.DOC_INFO_ELEMENT}
					}, true, false, dataElement);
				// Retrieve the aliases element as a child of docinfo
				Node aliases = SMLCommonUtil.nestedNodeRetrieval(docInfoElement, new String[][]{
						new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.ALIASES_ELEMENT}
					}, true, false, null);
				
				Document ownerDocument = aliases.getOwnerDocument();				
				Node aliasElement = SMLCommonUtil.createFormattedElement(aliases, ISMLConstants.SMLIF_URI, ISMLConstants.ALIAS_ELEMENT, true, null);
				// give it an interesting alias name based on the main element child of the data element
				aliasElement.appendChild(ownerDocument.createTextNode(getFirstChildElementName(dataElement) + SMLMessages.editorAlias));	
						
				if (aliasTable.getItemCount() == 0)
					refreshContent();
								
				endOperation(false, false);		
				break;
			}
			case OPERATION_REMOVE_ALIASES:
			{
				if (input == null)
					return false;
				Node aliases = SMLCommonUtil.nestedNodeRetrieval(input, new String[][]{
						new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.DOC_INFO_ELEMENT},
						new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.ALIASES_ELEMENT}
					});
				if (aliases == null)
				{ 
					MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
							SMLMessages.errorRemoving, SMLMessages.errorRemovingAlias1);
					return false;
				}
				TableItem[] selection = aliasTable.getSelection();
				if (selection.length <= 0)
				{
					MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
							SMLMessages.errorRemoving, SMLMessages.errorRemovingAlias2);
					return false;
				}
				
				boolean refreshContent = false;
				
				for (int i = 0; i < selection.length; i++)
				{
					int inx = aliasTable.indexOf(selection[i]);
					NodeList aliasList = aliases.getChildNodes();
					int counter = -1;
					Node currentAlias = null;
					for (int j = 0, aliasCount = aliasList.getLength(); j < aliasCount && counter != inx; j++)
					{
						currentAlias = aliasList.item(j);
						if (currentAlias.getNodeType() == Node.ELEMENT_NODE)
							counter++;
					}
					if (counter != inx)
					{
						MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
								SMLMessages.errorRemoving, SMLMessages.errorRemovingAlias1);
						return false;
					}
					currentAlias.getParentNode().removeChild(currentAlias);
					aliasTable.remove(inx);
					
					refreshContent = refreshContent || inx == 0;
				}
				
				// Clean up left over stuff from a remove (the aliases node)
				if (aliasTable.getItemCount() == 0) {
					Node parentOfAliases = aliases.getParentNode();
					parentOfAliases.removeChild(aliases);
					// Also clean up (remove) the docInfo node if there are no other element children
					if (parentOfAliases.getLocalName().equals(ISMLConstants.DOC_INFO_ELEMENT)) {
						boolean docInfoChildFound = false;
						NodeList docInfoElements = parentOfAliases.getChildNodes();
						for (int i = 0; i < docInfoElements.getLength(); i++) {
							if (docInfoElements.item(i).getNodeType() == Node.ELEMENT_NODE) {
								docInfoChildFound = true;
								break;
							}
						}
						if (!docInfoChildFound) {
							parentOfAliases.getParentNode().removeChild(parentOfAliases);
						}
					}
				}
				
				if (refreshContent)
				{
					refreshContent();
				}
				endOperation(true, false);		
				removeAlias.setEnabled(false);
				break;
			}
			case OPERATION_MODIFY_ALIAS:
			{
				Node aliases = SMLCommonUtil.nestedNodeRetrieval(input, new String[][]{
						new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.DOC_INFO_ELEMENT},
						new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.ALIASES_ELEMENT}});
				
				TableItem[] selection = aliasTable.getSelection();
				int selectionInx = selection == null || selection.length != 1 ? -1 : aliasTable.indexOf(selection[0]);
				Node[] aliasList = SMLValidatorUtil.retrieveChildElements(aliases);
				if (aliases == null || selectionInx < 0 || selectionInx >= aliasList.length)
				{
					MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
							SMLMessages.errorModifyingAliasTitle, SMLMessages.errorModifyingAlias0);
					if (selectionInx >= 0)
						aliasTable.remove(selectionInx);
					return false;
				}
				
				NodeList aliasChildNodes = aliasList[selectionInx].getChildNodes();
				for (int i = 0, aliasChildNodeCount = aliasChildNodes.getLength(); i < aliasChildNodeCount; i++)
				{
					aliasList[selectionInx].removeChild(aliasChildNodes.item(i));
				}
				aliasList[selectionInx].appendChild(aliasList[selectionInx].getOwnerDocument().createTextNode((String)arguments[0]));
				
				if (selectionInx == 0)
					refreshContent(aliasChildNodes.item(0).getNodeValue());
				
				endOperation(false, false);
				break;
			}
		}
		
		return true;
	}

	// Prompt for the type of SML document to add to the SML-IF file.
	// This prompter is invoked when the Documents or Instances
	// node is not selected.
	private int promptForDocumentType() {
		Dialog prompter = new Dialog(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell()) {
			protected void createButtonsForButtonBar(Composite parent) {
				createButton(parent, DEFINITIONS_CHOICE, SMLMessages.editorDocTypeDlgButtonDefinitions,
						true);
				createButton(parent, INSTANCES_CHOICE, SMLMessages.editorDocTypeDlgButtonInstances,
						false);
				createButton(parent, IDialogConstants.CANCEL_ID,
						IDialogConstants.CANCEL_LABEL, false);
			}
			protected void buttonPressed(int buttonId) {
				setReturnCode(buttonId);
				close();
			}
			protected void configureShell(Shell newShell) {
				super.configureShell(newShell);
				newShell.setText(SMLMessages.editorDocTypeDlgTitle);
			}
			protected Control createDialogArea(Composite parent) {
				Composite container = (Composite) super.createDialogArea(parent);
				GridLayout gridLayout = new GridLayout();
				gridLayout.numColumns = 1;
				container.setLayout(gridLayout);
				
				Label promptLabel = new Label(container, SWT.NONE);
				promptLabel.setText(SMLMessages.editorDocTypeDlgPromptText);
				return container;
			}
		};
		return prompter.open();
	}

	// Answer the name of the first child that is an element node
	private String getFirstChildElementName(Node dataElement) {
		Node child = dataElement.getFirstChild();
		while (child != null) {
			if (child.getNodeType() == Node.ELEMENT_NODE) {
				return child.getNodeName();
			}
			child = child.getNextSibling();
		}
		return ""; //$NON-NLS-1$
	}


	private void endOperation(boolean refreshDocuments, boolean expandRoot)
	{
		if (refreshDocuments)
		{
			documentTreeViewer.refresh();
			expandRootElement(expandRoot);
		}
		else
		{
			Node aliases = SMLCommonUtil.nestedNodeRetrieval(input, new String[][]{
					new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.DOC_INFO_ELEMENT},
					new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.ALIASES_ELEMENT}
				}, true, false, null);
			
			tableViewer.setInput(aliases);
		}
		
		SMLIFEditor editor = (SMLIFEditor)getEditor();
		IDocument document = editor.getSmlifSourcePage().getDocument();
		StringBuffer buffer = new StringBuffer();
		XMLInternalUtility.serializeNode(buffer, editor.getDocumentNode().getOwnerDocument());
		document.set(buffer.toString());
	}

	private void refreshContent()
	{
		refreshContent(null);
	}
	
	private void refreshContent(String nodeValue)
	{
		Tree documentTree = documentTreeViewer.getTree();
		TreeItem[] selection = documentTree.getSelection();
		int selectionInx = selection[0].getParentItem().indexOf(selection[0]);
		if (selection != null && selection.length > 0)
		{
			boolean definitionsSelected = documentTree.indexOf(selection[0].getParentItem()) == 0;
			documentTreeViewer.refresh();				
			expandRootElement(definitionsSelected);
			
			Event event = new Event();
			event.type = SWT.Selection;
			event.widget = documentTree;
			
			TreeItem rootItem = documentTree.getItem(definitionsSelected ? 0 : 1);
			int inx = -1;
			if (nodeValue != null)
			{
				TreeItem[] children = rootItem.getItems();
				for (int i = 0; i < children.length; i++)
				{
					if (nodeValue.equals(children[i].getText()))
					{
						inx = i;
						break;
					}
				}
			}
			inx = inx < 0 ? selectionInx : inx;
			event.item= rootItem.getItem(inx);
			documentTree.setSelection((TreeItem)event.item);
			documentTree.notifyListeners(event.type, event);				
		}
	}

	private void expandRootElement(boolean definitionsSelected)
	{		
		Tree documentTree = documentTreeViewer.getTree();		
		int expandedInx = definitionsSelected ? 0 : 1;
		if (expandedInx < documentTree.getItemCount())
		{
			Event event = new Event();
			event.type = SWT.Expand;
			event.widget = documentTree;
			event.item= documentTree.getItem(expandedInx);
			documentTree.getItem(expandedInx).setExpanded(true);
			documentTree.notifyListeners(event.type, event);
		}
	}

	private void adoptNode(Document document, Node parent, Node node)
	{
		switch (node.getNodeType())
		{
			case Node.TEXT_NODE:
			{
				String value = node.getNodeValue();				
				if (value == null)
					return;
				
				parent.appendChild(document.createTextNode(value));
				break;
			}			
					
			case Node.COMMENT_NODE:
			{
				String value = node.getNodeValue();				
				if (value == null)
					return;
				
				parent.appendChild(document.createComment(value));
				break;
			}			
			
			case Node.ELEMENT_NODE:
			{
				Node clonedNode = document.createElementNS(node.getNamespaceURI(), node.getNodeName());
				clonedNode.setPrefix(node.getPrefix());
				NamedNodeMap attributes = node.getAttributes();
								
				if (attributes != null)
				{
					for (int i = 0, attCount = attributes.getLength(); i < attCount; i++)
					{
						adoptNode (document, clonedNode, attributes.item(i));
					}
				}
				NodeList children = node.getChildNodes();
				if (children != null)
				{
					for (int i = 0, childCount = children.getLength(); i < childCount; i++)
					{
						adoptNode (document, clonedNode, children.item(i));
					}
				}
				
				parent.appendChild(clonedNode);				
				break;
			}	
			case Node.ATTRIBUTE_NODE:
			{
				Node clonedNode = document.createAttributeNS(node.getNamespaceURI(), node.getNodeName());
				clonedNode.setNodeValue(node.getNodeValue());
				parent.getAttributes().setNamedItemNS(clonedNode);
				
				break;
			}			
		}
	}	
	
	private boolean isDefinitionOrInstance(Object element)
	{			
		Node node = null;
		return 	element instanceof DocumentTreeElement && 
				(node = ((DocumentTreeElement)element).getNode()) != null &&
				ISMLConstants.SMLIF_URI.equals(node.getNamespaceURI()) &&
				
				(ISMLConstants.DEFINITIONS_ELEMENT.equals(node.getLocalName()) ||
				ISMLConstants.INSTANCES_ELEMENT.equals(node.getLocalName()));					
	}
	
	
	/**
	 * Invoked when an element is selected under the document
	 * tree.
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		ISelection selection = event.getSelection();
		StructuredSelection structuredSelection = null;
		if (!selection.isEmpty() && selection instanceof StructuredSelection && 
			(structuredSelection = (StructuredSelection)selection).size() > 0)
		{
			DocumentTreeElement input = (DocumentTreeElement)structuredSelection.getFirstElement();
			setControlEnabble(!input.isRootElement());
			setInput(input.isRootElement() ? null : input.getNode());
		}
		else
		{
			setInput(null);
		}
	}

	
	private void setControlEnabble(boolean enable)
	{
		addAlias.setEnabled(enable);
		rawStyledText.setEnabled(enable);
	}


	private void setInput(Object input)
	{		
		/* The alias table */
		aliasTable.removeAll();
		tableViewer.setInput(SMLCommonUtil.nestedNodeRetrieval((Node)input, new String[][] {
				new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.DOC_INFO_ELEMENT},
				new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.ALIASES_ELEMENT}
				}));
		
		
		/* The logical tree representation */
		this.input = (Node)input;
		logicalRepresentation.setInput(input);
		
		/* The raw text representation */
		indices.clear();
		Node[] node = null;
		if (input == null || (node = extractDocumentContent((Node)input)).length <= 0)
		{
			rawStyledText.setText(IValidationConstants.EMPTY_STRING);
			return;
		}		
		
		StringBuffer sb = new StringBuffer();
		Node lastChildElement = node[0].getLastChild();
		if (XMLInternalUtility.isWhitespace(lastChildElement))
		{
			String whitespace = lastChildElement.getNodeValue();
			if (whitespace != null)
			{
				sb.append(whitespace);
			}
		}
		
		XMLInternalUtility.serializeNode (sb, node[0], 0, "0", indices, false); //$NON-NLS-1$
		rawStyledText.setText(sb.toString());
	}


	private Node[] extractDocumentContent(Node documentNode)
	{
		Node dataNode = SMLCommonUtil.retrieveChildNode((Node)documentNode, 
				ISMLConstants.SMLIF_URI, ISMLConstants.DATA_ELEMENT);
		Node[] elements = SMLValidatorUtil.retrieveChildElements(dataNode);
		if (elements.length <= 0)
		{
			dataNode = SMLCommonUtil.retrieveChildNode((Node)input, 
					ISMLConstants.SMLIF_URI, ISMLConstants.LOCATOR_ELEMENT);
			return dataNode == null ? new Node[0] : new Node[]{dataNode};
		}
		
		return elements;
	}
	

	/**
	 * @see org.eclipse.cosmos.rm.smlif.internal.editor.AbstractFormPage#updateContent()
	 */
	protected void updateContent()
	{
		if (documentTreeViewer != null && !documentTreeViewer.getTree().isDisposed())			
			documentTreeViewer.refresh();
	}
	
	
	/**
	 * This is the content provider for the document tree
	 * 
	 * @author Ali Mehregani
	 */
	private class DocumentContentProvider implements ITreeContentProvider
	{
		
		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object parent)
		{
			Node node = ((DocumentTreeElement)parent).getNode();
			if (node == null)
				return new Object[0];
			
			List<DocumentTreeElement> children = new ArrayList<DocumentTreeElement>();
			NodeList nodeChildren = node.getChildNodes();
			for (int i = 0, childCount = nodeChildren.getLength(); i < childCount; i++)
			{
				Node currentChildNode = nodeChildren.item(i);
				if (currentChildNode.getNodeType() == Node.ELEMENT_NODE)
				{
					DocumentTreeElement element = new DocumentTreeElement();
					element.setNode(nodeChildren.item(i));
					children.add(element);
				}
			}				
			
			return children.toArray();
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element)
		{
			return null;
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element)
		{			
			return isDefinitionOrInstance(element);
		}

		/**
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object inputElement)
		{
			SMLIFEditor editor = (SMLIFEditor)getEditor();
			
			List<DocumentTreeElement> elements = new ArrayList<DocumentTreeElement>();
			addElement (elements, 
					editor.retrieveChildNode(ISMLConstants.SMLIF_URI, ISMLConstants.DEFINITIONS_ELEMENT), 
					SMLMessages.editorDefinitions); 
			addElement (elements, 
					editor.retrieveChildNode(ISMLConstants.SMLIF_URI, ISMLConstants.INSTANCES_ELEMENT),
					SMLMessages.editorInstances);
					

			return elements.toArray();
		}

		private void addElement(List<DocumentTreeElement> elements, Node node, String text)
		{
			if (node == null)
				return;
			
			DocumentTreeElement element = new DocumentTreeElement(true);
			element.setText(text);
			element.setNode(node);
			
			elements.add(element);
		}

		public void dispose()
		{
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{		
		}
	
	}
	
	
	/**
	 * The label provider for the document tree
	 * 
	 * @author Ali Mehregani
	 */
	private class DocumentLabelProvider implements ILabelProvider
	{
		/**
		 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
		 */
		public Image getImage(Object element)
		{			
			if (isDefinitionOrInstance(element))
			{
				return ISMLConstants.DEFINITIONS_ELEMENT.equals(((DocumentTreeElement)element).getNode().getLocalName()) ?
						SMLImages.INSTANCE.getImage(SMLImages.DEFINITIONS) :
						SMLImages.INSTANCE.getImage(SMLImages.INSTANCES);
			}
			else
			{
				return SMLImages.INSTANCE.getImage(SMLImages.DOCUMENTS);
			}
		}

		/**
		 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
		 */
		public String getText(Object element)
		{
			DocumentTreeElement treeElement = (DocumentTreeElement)element;
			if (treeElement.getText() != null)
				return treeElement.getText();
			
			return treeElement.retrieveLabel();			
		}

		/**
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#addListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void addListener(ILabelProviderListener listener)
		{
			
		}

		/**
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
		 */
		public void dispose()
		{
			
		}

		/**
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#isLabelProperty(java.lang.Object, java.lang.String)
		 */
		public boolean isLabelProperty(Object element, String property)
		{
			return false;
		}

		/**
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#removeListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void removeListener(ILabelProviderListener listener)
		{
		}		
	}
	
	
	/**
	 * Represents each element that appears under the document tree
	 * 
	 * @author Ali Mehregani
	 */
	private static class DocumentTreeElement
	{
		/**
		 * The associated node
		 */
		private Node node;
		
		/**
		 * The associated text
		 */
		private String text;

		/**
		 * Indicates whether this is a root element
		 */
		private boolean rootElement;
				
		public DocumentTreeElement()
		{
			
		}
		
		public DocumentTreeElement(boolean rootElement)
		{
			this.rootElement = rootElement;
		}
		
		/**
		 * The label of an element representing an SML document is either
		 * the first alias of the document or the first element name of
		 * the document.
		 *  
		 * @return The label for a tree element representing an SML document
		 */
		public String retrieveLabel()
		{
			if (node == null)
				return null;
			
			Node alias = SMLCommonUtil.nestedNodeRetrieval (node, new String[][]{
				new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.DOC_INFO_ELEMENT},
				new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.ALIASES_ELEMENT},
				new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.ALIAS_ELEMENT}
			});
			
			text = SMLCommonUtil.retrieveTextNode(alias);
			if (text != null && text.length() > 0)
				return text;
			
			Node childNode = SMLCommonUtil.retrieveChildNode(node, ISMLConstants.SMLIF_URI, ISMLConstants.DATA_ELEMENT);
			if (childNode == null)
			{
				childNode = SMLCommonUtil.retrieveChildNode(node, ISMLConstants.SMLIF_URI, ISMLConstants.LOCATOR_ELEMENT);
				if (childNode == null)
				{
					return SMLMessages.commonUnknown;
				}
				Node reference = childNode.getAttributes().getNamedItemNS(ISMLConstants.XLINK_URI, ISMLConstants.HREF_ATTRIBUTE);
				return reference == null ? SMLMessages.commonUnknown : SMLMessages.editorRemoteDoc + reference.getNodeValue();
			}
			else
			{
				NodeList children = childNode.getChildNodes();
				String text = retrieveElementName(children);
				return text == null ? SMLMessages.commonUnknown : text;
			}
		}

		private String retrieveElementName(NodeList children)
		{
			for (int i = 0, childCount = children.getLength(); i < childCount; i++)
			{
				Node node = children.item(i);
				if (node.getNodeType() != Node.TEXT_NODE && node.getLocalName() != null)
					return node.getLocalName();
			}
			
			return null;
		}

		/**
		 * @return the node
		 */
		public Node getNode()
		{
			return node;
		}
		
		/**
		 * @param node the node to set
		 */
		public void setNode(Node node)
		{
			this.node = node;
		}


		/**
		 * @return the text
		 */
		public String getText()
		{
			return text;
		}


		/**
		 * @param text the text to set
		 */
		public void setText(String text)
		{
			this.text = text;
		}		
		
		/**
		 * @return A flag indicating whether this is a root element
		 */
		public boolean isRootElement()
		{
			return rootElement;
		}
	}
	
	
	/**
	 * This content provider is used to provide the content for the logical
	 * tree representation of a document
	 * 
	 * @author Ali Mehregani
	 */
	private class LogicalContentProvider implements ITreeContentProvider
	{

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object parent)
		{
			return SMLValidatorUtil.retrieveChildElements((Node)parent);
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element)
		{
			return null;
		}

		
		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element)
		{
			return SMLValidatorUtil.retrieveChildElements((Node)element).length > 0;
		}

		/**
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object input)
		{			
			return extractDocumentContent((Node)input);
		}

		/**
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose()
		{
		}

		/**
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
		}		
	}
	
	
	/**
	 * The label provider for the logical tree representaion of a document.
	 * 
	 * @author Ali Mehregani
	 */
	private class LogicalLabelProvider implements ILabelProvider
	{
		/**
		 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
		 */
		public Image getImage(Object element)
		{
			return SMLImages.INSTANCE.getImage(SMLImages.ELEMENT);
		}

		/**
		 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
		 */
		public String getText(Object element)
		{
			return ((Node)element).getNodeName();
		}

		/**
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#addListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void addListener(ILabelProviderListener listener)
		{
		}

		/**
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
		 */
		public void dispose()
		{
		}

		/** 
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#isLabelProperty(java.lang.Object, java.lang.String)
		 */
		public boolean isLabelProperty(Object element, String property)
		{
			return false;
		}

		/**
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#removeListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void removeListener(ILabelProviderListener listener)
		{
		}
		
	}

	
	/**
	 * The selection listener for the logical tree representation of
	 * a document.  This is used to select the corresponding fragment under
	 * the XML representation text box
	 * 
	 * @author Ali Mehregani
	 */
	private class LogicalSelectionListener implements ISelectionChangedListener
	{
		/**
		 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
		 */
		public void selectionChanged(SelectionChangedEvent event)
		{
			ISelection selection = event.getSelection();
			if (!(selection instanceof StructuredSelection) || selection.isEmpty())
			{
				rawStyledText.setSelection(0,0);
				return;
			}
					
			String index = findSelectionIndex (null, logicalRepresentation.getTree().getSelection()[0]);
			int[] elementIndices = indices.get(index);
			if (elementIndices == null)
			{
				rawStyledText.setSelection(0,0);
				return;
			}
			
			rawStyledText.setSelection(elementIndices[0], elementIndices[1]);
		}

		
		private String findSelectionIndex(String index, TreeItem item)
		{
			TreeItem parentItem = item.getParentItem();
			if (parentItem != null)
			{
				String inx = String.valueOf(parentItem.indexOf(item));
				if (index != null && index.length() > 0)
					inx += "," + index; //$NON-NLS-1$
				
				return findSelectionIndex(inx, parentItem);
			}
			return "0" + (index == null || index.length() <= 0 ? IValidationConstants.EMPTY_STRING : "," + index); //$NON-NLS-1$ //$NON-NLS-2$
		}		
	}


	public void setActive(boolean active) {
		super.setActive(active);
		if (active) {
			documentTree.getFilterControl().setFocus();
		}
	}
}
