/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.smlif.internal.editor;

import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IFindReplaceTarget;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.swt.custom.ExtendedModifyEvent;
import org.eclipse.swt.custom.ExtendedModifyListener;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.editors.text.TextEditor;

/**
 * An extended text editor used to display the raw XML content of the SML-IF
 * document
 * 
 * @author Ali Mehregani
 */
public class SMLIFTextPage extends TextEditor
{	
	/**
	 * Find/replace target
	 */
	private FindReplaceTarget findReplaceTarget;
	

	
	public void createPartControl(Composite parent)
	{		
		super.createPartControl(parent);
		ISourceViewer sourceViewer = super.getSourceViewer();
		final StyledText styledText = sourceViewer.getTextWidget();
		styledText.addLineStyleListener(new XMLLineStyler());
		styledText.addExtendedModifyListener(new ExtendedModifyListener(){

			public void modifyText(ExtendedModifyEvent event)
			{
				styledText.redraw();
			}
		});
		sourceViewer.setEditable(true);
		getSite().setSelectionProvider(null);
	}
	
	/**
	 * @see org.eclipse.ui.part.WorkbenchPart#getAdapter(java.lang.Class)
	 */
	@SuppressWarnings("unchecked")
	public Object getAdapter(Class adapter)
	{
		if (adapter == IFindReplaceTarget.class)
		{
			if (findReplaceTarget == null && getSourceViewer() != null)
				findReplaceTarget = new FindReplaceTarget(getSourceViewer().getTextWidget());
			return findReplaceTarget;			
		}
		
		return super.getAdapter(adapter);
	}
	
	public boolean isEditable()
	{		
		getSourceViewer().setEditable(!getSourceViewer().isEditable());
		return getSourceViewer().isEditable();
	}
	
	
	public void activated()
	{
		getSourceViewer().setEditable(true);
	}

	
	public void deactivated()
	{
		getSourceViewer().setEditable(false);
	}
	
	
	/**
	 * The Macro find/replace targe.  This class is used to provide a target to the
	 * find/replace dialog.
	 * 
	 * @author Ali Mehregani
	 */
	public static class FindReplaceTarget implements IFindReplaceTarget
	{	
		/** The input */
		private StyledText input;
		
		public FindReplaceTarget(StyledText input)
		{
			this.input = input;
		}

		public boolean canPerformFind()
		{
			return input != null;
		}

		public int findAndSelect(int widgetOffset, String findString, boolean searchForward, boolean caseSensitive, boolean wholeWord)
		{						
			String macro = input.getText();
			macro = widgetOffset < 0 ? macro.substring(0) : searchForward ? macro.substring(widgetOffset) : macro.substring(0, widgetOffset);
			char[] findStringChar = findString.toCharArray();
			int firstCharacterInx = -1; 
			char currentCharacter;
					
			
			if(searchForward)
			{				
				for (int i = 0, macroLength = macro.length(); i < macroLength; i++)
				{
					currentCharacter = macro.charAt(i);
					firstCharacterInx = findFirstCharInx(currentCharacter, i, macro, findStringChar, searchForward, caseSensitive, wholeWord);
					firstCharacterInx = findSuccessiveCharacters(firstCharacterInx, findStringChar, macroLength, macro, i, 0, currentCharacter, searchForward, caseSensitive);
										
					if (firstCharacterInx != -1)
					{
						widgetOffset = widgetOffset < 0 ? 0 : widgetOffset;
						input.setSelection( i + widgetOffset, i + findStringChar.length + widgetOffset );
						return i;
					}
				}
			}
			
			else{
				
				for (int k = macro.length()-1, macroLength = macro.length(); k > 0; k--)
				{
					currentCharacter = macro.charAt(k);					
					int lastChar = findStringChar.length-1;					
					firstCharacterInx = findFirstCharInx(currentCharacter, k, macro, findStringChar, searchForward, caseSensitive, wholeWord);
					firstCharacterInx = findSuccessiveCharacters(firstCharacterInx, findStringChar, macroLength, macro, k, lastChar, currentCharacter, searchForward, caseSensitive);
					
					if (firstCharacterInx != -1)
					{
						k++;
						input.setSelection(k , k - findStringChar.length );
						return k;
					}
				}
			}	
			
			
			return -1;
		}

		private int findSuccessiveCharacters(int firstCharacterInx, char[] findStringChar, int macroLength, String macro, int macroInx, int lastChar, char currentCharacter, boolean searchForward, boolean caseSensitive)
		{
			for (int j = 1; firstCharacterInx != -1 && j < findStringChar.length; j++)
			{
				if ((searchForward ? macroInx + j : macroInx - j) >= macroLength)
				{
					firstCharacterInx = -1;
					break;
				}
				currentCharacter = macro.charAt((searchForward ? macroInx + j : macroInx - j));
				if (currentCharacter != findStringChar[searchForward ? j : lastChar - j] &&
					(caseSensitive ? true : (currentCharacter <= 90 ? currentCharacter + 32 != findStringChar[searchForward ? j : lastChar - j] : currentCharacter - 32 != findStringChar[searchForward ? j : lastChar - j])))
				{
					firstCharacterInx = -1;
					break;
				}
			}
			
			return firstCharacterInx;
		}

		private int findFirstCharInx(char currentCharacter, int i, String macro, char[] findStringChar, boolean searchForward, boolean caseSensitive, boolean wholeWord)
		{			
			if ((currentCharacter == findStringChar[searchForward ? 0 : findStringChar.length-1] || 
					(!caseSensitive && (currentCharacter <= 90 ? currentCharacter + 32 == findStringChar[0] : currentCharacter - 32 == findStringChar[0]))) && 
					(wholeWord ? macro.charAt(i - 1) == 32 && macro.charAt(i + findStringChar.length) == 32 : true))
			{
				return i;
			}
			
			return -1;
		}

		public Point getSelection()
		{
			Point selection = input.getSelection();;
			return new Point (selection.x, selection.y - selection.x);
		}

		public String getSelectionText()
		{
			Point selection = getSelection();
			return input.getText().substring(selection.x, selection.x + selection.y);
		}

		public boolean isEditable()
		{
			return true;
		}

		public void replaceSelection(String text)
		{
			if (input == null)
				return;
			
			Point selection = input.getSelection();
			input.replaceTextRange(selection.x, selection.y - selection.x, text);
		}
		
	}


	public IDocument getDocument()
	{
		return getSourceViewer().getDocument();
	}
}
