/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.smlif.internal.export.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.internal.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLIFIdentity;
import org.eclipse.cosmos.rm.repository.operations.ISMLExportOperation;
import org.eclipse.cosmos.rm.repository.operations.ISMLOperation;
import org.eclipse.cosmos.rm.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.repository.resource.ISMLIFIdentity;
import org.eclipse.cosmos.rm.smlif.internal.SMLImages;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLConstants;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLMessages;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLRepositoryManager;
import org.eclipse.cosmos.rm.validation.internal.artifacts.RuleBinding;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ContainerCheckedTreeViewer;
import org.eclipse.ui.dialogs.ListDialog;
import org.eclipse.ui.dialogs.SaveAsDialog;

public class ExportToSMLIFWizardPage extends WizardPage {
	
	private static final String DEFAULT_SMLIF_FILE_NAME = "smlif.xml"; //$NON-NLS-1$
	protected Text projectText;
	protected Text smlifText;
	protected IStructuredSelection selection;
	protected ContainerCheckedTreeViewer fileTree;

	public ExportToSMLIFWizardPage(String pageName, IStructuredSelection selection) {
		super(pageName);
		setTitle(pageName);
		
		setSelection(selection);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.dialogs.WizardNewFileCreationPage#createAdvancedControls(org.eclipse.swt.widgets.Composite)
	 */	
	public void createControl(Composite parent) {
		Composite topComponent = new Composite(parent, SWT.NONE);
		GridData topComponentData = new GridData(GridData.GRAB_HORIZONTAL
				| GridData.FILL_HORIZONTAL);
		topComponent.setLayoutData(topComponentData);

		GridLayout topComponentLayout = new GridLayout();
		topComponentLayout.numColumns = 2;
		topComponentLayout.makeColumnsEqualWidth = false;
//		topComponentLayout.marginWidth = 3;
//		topComponentLayout.marginHeight = 3;
		topComponent.setLayout(topComponentLayout);
		
		createProjectPromptComposite(topComponent);
		createSelectedResourcesLabel(topComponent);
		createFileTree(topComponent);
		createSelectionButtons(topComponent);
		Group destGroup = createDestinationGroup(topComponent);
		createDestinationText(destGroup);
		createBrowseButtons(destGroup);
						
		setControl(topComponent);
		updatePageForSelection();
		updatePageComplete();
		Dialog.applyDialogFont(topComponent);
	}

	protected void updatePageForSelection() {
		if (!selection.isEmpty()) {
			projectText.setText(((IResource) selection.getFirstElement()).getProject().getName());
		}
		initCheckedItems();
	}

	protected void createProjectPromptComposite(Composite topComponent) {
		Composite projectComposite = new Composite(topComponent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		projectComposite.setLayout(layout);
		GridData compositeData = new GridData();
		compositeData.horizontalAlignment = GridData.FILL;
		compositeData.horizontalSpan = 2;
		projectComposite.setLayoutData(compositeData);
		
		Label projectLabel = new Label(projectComposite, SWT.NONE);
		projectLabel.setText(SMLMessages.exportWizardMainPageProjectLabel);
		GridData labelData = new GridData(GridData.GRAB_HORIZONTAL
				| GridData.FILL_HORIZONTAL);
		labelData.grabExcessHorizontalSpace = true;
		labelData.horizontalSpan = 2;
		projectLabel.setLayoutData(labelData);
		
		projectText = new Text(projectComposite, SWT.BORDER | SWT.SINGLE);
		projectText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent arg0) {
				updateFileTree();
			}
		});
		GridData projectTextData = new GridData(GridData.GRAB_HORIZONTAL
				| GridData.FILL_HORIZONTAL);
		projectTextData.grabExcessHorizontalSpace = true;
		projectText.setLayoutData(projectTextData);
		
		Button projectBrowseButton = new Button(projectComposite, SWT.PUSH);
		projectBrowseButton.setText(SMLMessages.exportWizardMainPageBrowseButton);
		projectBrowseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				promptForProject();
			}
		});
	}

	protected void promptForProject() {
		ListDialog dialog = new ListDialog(getShell());
		dialog.setTitle(SMLMessages.exportWizardMainPageProjectPrompterTitle);
		dialog.setMessage(SMLMessages.exportWizardMainPageProjectPrompterMessage);
		IProject[] choices = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		dialog.setLabelProvider(new LabelProvider() {
			public String getText(Object element) {
				return ((IProject) element).getName();
			}
			public Image getImage(Object element) {
				return SMLImages.INSTANCE.getImage(SMLImages.PROJECT);
			}
		});
		dialog.setContentProvider(new ArrayContentProvider());
		dialog.setInput(choices);
		if (dialog.open() == InputDialog.OK) {
			Object[] selections = dialog.getResult();
			projectText.setText(((IProject) selections[0]).getName());
			updateFileTree();
		}
	}

	protected void updateFileTree() {
		if ("".equals(projectText.getText())) { //$NON-NLS-1$
			fileTree.getControl().setEnabled(false);
			return;
		}
		IProject selectedProject = getSelectedProject();
		if ((selectedProject == null) || (!selectedProject.exists())) {
			setMessage(NLS.bind(SMLMessages.exportWizardMainPageErrorProjectNameNotValid, projectText.getText()));
			fileTree.getControl().setEnabled(false);
			return;
		}
		fileTree.setInput(selectedProject);
		fileTree.getControl().setEnabled(true);
	}

	public IProject getSelectedProject() {
		return ResourcesPlugin.getWorkspace().getRoot().getProject(projectText.getText());
	}

	protected void createSelectedResourcesLabel(Composite topComponent) {
		Label selectedResourcesLabel = new Label(topComponent, SWT.NONE);
		selectedResourcesLabel.setText(SMLMessages.exportWizardMainPageSelectedResourcesLabel);
		GridData selectedResourcesLabelData = new GridData(GridData.GRAB_HORIZONTAL
				| GridData.FILL_HORIZONTAL);
		selectedResourcesLabelData.grabExcessHorizontalSpace = true;
		selectedResourcesLabelData.horizontalSpan = 2;
		selectedResourcesLabel.setLayoutData(selectedResourcesLabelData);
	}

	protected void createBrowseButtons(Group destGroup) {
		Composite browseButtonRow = new Composite(destGroup, SWT.NONE);
		GridLayout browseButtonRowLayout = new GridLayout();
		browseButtonRowLayout.numColumns = 2;
		browseButtonRowLayout.marginHeight = 0;
		browseButtonRowLayout.marginWidth = 0;
		browseButtonRowLayout.makeColumnsEqualWidth = true;
		GridData gd = new GridData(GridData.HORIZONTAL_ALIGN_END);
		gd.horizontalSpan = GridData.FILL;
		browseButtonRow.setLayoutData(gd);
		browseButtonRow.setLayout(browseButtonRowLayout);
		
		Button browseWorkspaceButton = new Button(browseButtonRow, SWT.PUSH);
		browseWorkspaceButton.setText(SMLMessages.exportWizardMainPageBrowseWorkspaceButton);
		browseWorkspaceButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				promptForWorkspaceDestinationFile();
			}
		});
		
		Button browseFileSystemButton = new Button(browseButtonRow, SWT.PUSH);
		browseFileSystemButton.setText(SMLMessages.exportWizardMainPageBrowseFileSystemButton);
		browseFileSystemButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				promptForExternalDestinationFile();
			}
		});
	}

	protected void createDestinationText(Group destGroup) {
		smlifText = new Text(destGroup, SWT.BORDER);
		smlifText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				if (smlifText.getText().length() > 0) {
					
				}
				updatePageComplete();
			}
		});
		GridData smlifGridData = new GridData(GridData.FILL_HORIZONTAL);
		smlifGridData.grabExcessHorizontalSpace = true;
		smlifGridData.horizontalSpan = GridData.FILL;
		smlifText.setLayoutData(smlifGridData);
		IPath fileSuggestion = getTargetFileSuggestion();
		if (fileSuggestion != null) {
			smlifText.setText(fileSuggestion.toString());
		}
	}

	protected void promptForWorkspaceDestinationFile() {
		SaveAsDialog dialog = new SaveAsDialog(getShell());
		dialog.setTitle(SMLMessages.exportWizardMainPageSaveAsDialogTitle);
		IPath path = getTargetFileSuggestion();
		if ((path == null) || path.equals("")) {
			dialog.setOriginalName(DEFAULT_SMLIF_FILE_NAME); //$NON-NLS-1$
		} else {
			dialog.setOriginalFile(ResourcesPlugin.getWorkspace().getRoot().getFile(path));
		}

		if (dialog.open() == Window.CANCEL) {
			return;
		}
		IPath name = (IPath) dialog.getResult();
		String relativePathToFile = name.toString();
		if (relativePathToFile != null) {
			smlifText.setText(relativePathToFile);
			updatePageComplete();
		}
	}

	private IPath getTargetFileSuggestion() {
		IResource firstSelection = (IResource) selection.getFirstElement();
		if (firstSelection == null) {
			return null;
		}
		if (firstSelection instanceof IContainer) {
			return firstSelection.getFullPath().append(DEFAULT_SMLIF_FILE_NAME);
		}
		return firstSelection.getParent().getFullPath().append(DEFAULT_SMLIF_FILE_NAME);
	}

	protected void createSelectionButtons(Composite topComponent) {
		Composite buttonComposite = new Composite(topComponent, SWT.NONE);
		GridLayout buttonLayout = new GridLayout();
		buttonLayout.numColumns = 1;
		buttonLayout.marginHeight = 0;
		buttonLayout.marginWidth = 0;
		buttonComposite.setLayout(buttonLayout);
		GridData buttonCompositeData = new GridData();
		buttonCompositeData.verticalAlignment = GridData.FILL;
		buttonComposite.setLayoutData(buttonCompositeData);
		
		Button selectAllButton = new Button(buttonComposite, SWT.NONE);
		selectAllButton.setText(SMLMessages.exportWizardMainPageSelectAllButton);
		selectAllButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				getFileTree().setAllChecked(true);
				getExportWizard().allFilesChecked(true);
			}
		});
		GridData buttonData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		selectAllButton.setLayoutData(buttonData);

		Button deselectAllButton = new Button(buttonComposite, SWT.NONE);
		deselectAllButton.setText(SMLMessages.exportWizardMainPageDeselectAllButton);
		deselectAllButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				getFileTree().setAllChecked(false);
				getExportWizard().allFilesChecked(false);
			}
		});
		deselectAllButton.setLayoutData(buttonData);
	}

	protected void createFileTree(Composite topComponent) 
	{
		setFileTree(new ContainerCheckedTreeViewer(topComponent));
		ExportResourceTreeContentProvider contentProvider = new ExportResourceTreeContentProvider();
		getFileTree().setContentProvider(contentProvider);
		getFileTree().setLabelProvider(new ExportResourceTreeLabelProvider());
		getFileTree().addCheckStateListener((ExportToSMLIFWizard) getWizard());
		GridData fileTreeData = new GridData(GridData.FILL_BOTH);
		fileTreeData.verticalSpan = 2;
		fileTreeData.grabExcessVerticalSpace = true;
		getFileTree().getControl().setLayoutData(fileTreeData);
		
		initCheckedItems();
	}
	
	protected void initCheckedItems() {
		for (Iterator<?> iter = selection.iterator(); iter.hasNext();) {
			IResource element = (IResource) iter.next();
			
			if (element instanceof IProject) {
				// Project has been selected in workspace, but does not exist
				// in tree, so we need to go select its children
				Object[] kids = ((ExportResourceTreeContentProvider)getFileTree().getContentProvider()).getChildren(element);
				for (int i = 0; i < kids.length; i++) {
					getFileTree().setChecked((IResource) kids[i], true);
				}
			} else {
				getFileTree().setChecked(element, true);
			}
			// ensure selection cache is updated
			getExportWizard().addFileToSelected(element);
		}
	}

	public boolean doExport() {
		getPage2().initFromMetadata();
		String smlifFileName = smlifText.getText();
		if (fileIsWorkspacePath(smlifFileName)) {
			// Convert to absolute filesystem path
			IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(smlifFileName));
			if (file != null) {
				IPath location = file.getLocation();
				if (location == null) {
					// project that does not exist
					String projectName = SMLRepositoryUtil.firstSegment(smlifFileName, 1);
					IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
					if (!project.exists()) {
						// this should always be true for this code path
						try	{
							project.create(new NullProgressMonitor());
							project.open(new NullProgressMonitor());
						} catch (CoreException e) {
							SMLCommonUtil.openErrorWithDetail(
									SMLMessages.errorCreatingProjectTitle, 
									SMLMessages.errorCreatingProject, 
									e);
							return false;
						}
					}
					file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(smlifFileName));
					smlifFileName = file.getLocation().toString();
				} else {
					smlifFileName = location.toString();
				}
			}
		}
		
		try 
		{			
			ISMLOperation operation = SMLRepositoryManager.getRepository().getOperation(ISMLExportOperation.ID);
			ISMLIFIdentity identity = new SMLIFIdentity(
					smlifFileName, 
					getPage2().getSMLIFName(), 
					getPage2().getSMLIFDescription(), 
					getPage2().getDisplayName(), 
					getPage2().getVersion(), 
					getPage2().getBaseURI()); 
			operation.setArguments(new Object[]
			{
					getSelectedModelUnits(),
					getExportWizard().getPage2().getAliases(),
					getRuleBindings().toArray(new RuleBinding[getRuleBindings().size()]),
					identity
			});
			operation.run();
			
			// Refresh the project
			IFile workspaceFile = ResourcesPlugin.getWorkspace().getRoot().getFileForLocation(new Path(smlifFileName));
			if (workspaceFile != null) {
				workspaceFile.getProject().refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
			}
			
			return true;
		} 
		catch (RepositoryOperationException ex) 
		{
			SMLCommonUtil.openErrorWithDetail(SMLMessages.exportErrorTitle, ex.getMessage(), ex);
			return false;
		} 
		catch (CoreException e)
		{
			// Ignore this exception if the project could not be refreshed
			e.printStackTrace();
			return true;
		}

	}

	protected List<RuleBinding> getRuleBindings() {
		List<?> bindings = (List<?>) getPage3().getRuleBindings();
		List<RuleBinding> result = new ArrayList<RuleBinding>();
		for (Iterator<?> iter = bindings.iterator(); iter.hasNext();) {
			RuleBinding element = (RuleBinding) iter.next();
			if (element.isValid()) {
				result.add(new RuleBinding((SMLConstants.ANY_DOCUMENT_ALIAS.equals(element.getAlias()) ? null : element.getAlias()), element.getRule()));
			}
		}
		return result;
	}

	private ExportToSMLIFWizardPage3 getPage3() {
		return ((ExportToSMLIFWizard) getWizard()).page3;
	}

	protected ExportToSMLIFWizardPage2 getPage2() {
		return ((ExportToSMLIFWizard) getWizard()).page2;
	}

	protected boolean fileIsWorkspacePath(String smlifFileName) {
		if (Platform.getOS().equals("win32")) {
			return new Path(smlifFileName).getDevice() == null;
		}
		// No good way to tell on a non-Windows platform
		// unless we assume that beginning with a slash is absolute
		// and without a slash is relative to workspace root?
		return true;
	}

	protected ISMLDocument[] getSelectedModelUnits() 
	{
		Collection<?> selectedFiles = getExportWizard().getSelectedFiles().values();
		return (ISMLDocument[])selectedFiles.toArray(new ISMLDocument[selectedFiles.size()]);
	}

	protected ExportToSMLIFWizard getExportWizard() {
		return (ExportToSMLIFWizard) getWizard();
	}

	public void setSelection(IStructuredSelection selection) {
		this.selection = selection;
	}

	protected Group createDestinationGroup(Composite topComposite) {
		Group destinationGroup = new Group(topComposite, SWT.NONE);
		destinationGroup.setText(SMLMessages.exportWizardDestinationGroupTitle);
		GridData destinationGroupData = new GridData(GridData.GRAB_HORIZONTAL
				| GridData.FILL_HORIZONTAL);
		destinationGroupData.horizontalSpan = 2;
		destinationGroup.setLayoutData(destinationGroupData);
	
		GridLayout destinationGroupLayout = new GridLayout();
		destinationGroupLayout.numColumns = 2;
		destinationGroupLayout.makeColumnsEqualWidth = false;
		destinationGroupLayout.marginWidth = 5;
		destinationGroupLayout.marginHeight = 5;
		//destinationGroupLayout.marginLeft = 3;
		//destinationGroupLayout.marginRight = 3;
		destinationGroup.setLayout(destinationGroupLayout);
		return destinationGroup;
	}

	protected void updatePageComplete() {
		setPageComplete((smlifText.getText().length() > 0) && (getCheckedCount() > 0));
	}

	protected int getCheckedCount() {
		return ((ExportToSMLIFWizard) getWizard()).getSelectedFiles().size();
	}

	protected void promptForExternalDestinationFile() {
		FileDialog dialog = new FileDialog(getShell());
		
		dialog.setText(SMLMessages.exportWizardMainPageSaveAsDialogTitle);
		dialog.setFileName(DEFAULT_SMLIF_FILE_NAME); //$NON-NLS-1$
		//dialog.setFilterPath(smlResourcePath);
		dialog.setFilterExtensions(new String[] { "*.xml", "*" }); //$NON-NLS-1$ //$NON-NLS-2$

		String name = dialog.open();
		if (name != null) {
			smlifText.setText(name);
			updatePageComplete();
		}
	}

	public void setFileTree(ContainerCheckedTreeViewer fileTree) {
		this.fileTree = fileTree;
	}

	public ContainerCheckedTreeViewer getFileTree() {
		return fileTree;
	}

	public String getProjectName() {
		return projectText.getText();
	}
}
