/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.smlif.internal.export.ui;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.cosmos.rm.repository.internal.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.repository.internal.core.FileSystemSMLRepository;
import org.eclipse.cosmos.rm.repository.internal.core.MetadataFileHandler;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLFileMetadata;
import org.eclipse.cosmos.rm.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.repository.resource.ISMLMetadata;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLMessages;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLPage;
import org.eclipse.cosmos.rm.smlif.internal.common.SMLRepositoryManager;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

/**
 * Second page of export wizard, with document alias and identity information
 * 
 * @author David Whiteman
 */
public class ExportToSMLIFWizardPage2 extends WizardPage {

	protected IStructuredSelection selection;
	protected ListViewer modelFileListViewer;
	protected ListViewer aliasListViewer;
	protected Combo typeCombo;
	protected Composite documentDetailComposite;
	protected Text nameText;
	protected Text baseURIText;
	protected Text displayNameText;
	protected Text versionText;
	protected Text descriptionText;
	protected boolean uninitializedIdentityFields = true;
	private Map<String, String[]> aliases;

	public ExportToSMLIFWizardPage2(String pageName,
			IStructuredSelection selection) {
		super(pageName);
		setTitle(pageName);

		setSelection(selection);
		aliases = new Hashtable<String, String[]>();
	}

	protected void setSelection(IStructuredSelection selection) {
		this.selection = selection;
	}

	protected void updateDocumentList() 
	{
		modelFileListViewer.setInput(getModel());
		resetDocumentDetailComposite();
	}

	public void createControl(Composite parent) {
		Composite topComponent = new Composite(parent, SWT.NONE);
		GridData topComponentData = new GridData(GridData.GRAB_HORIZONTAL
				| GridData.FILL_HORIZONTAL);
		topComponent.setLayoutData(topComponentData);

		GridLayout topComponentLayout = new GridLayout();
		topComponentLayout.numColumns = 1;
		topComponentLayout.makeColumnsEqualWidth = false;
		topComponent.setLayout(topComponentLayout);

		createIdentityInformationGroup(topComponent);
		createDocumentInformationGroup(topComponent);
		
		setControl(topComponent);
		setPageComplete(true);
		Dialog.applyDialogFont(topComponent);
	}

	protected void createIdentityInformationGroup(Composite topComponent) {
		Group identityInformationGroup = new Group(topComponent, SWT.NONE);
		identityInformationGroup
				.setText(SMLMessages.exportWizardIdentityInformationGroupTitle);
		GridData identityInformationGroupData = new GridData(
				GridData.GRAB_HORIZONTAL | GridData.FILL_BOTH);
		identityInformationGroupData.horizontalSpan = 1;
		identityInformationGroup.setLayoutData(identityInformationGroupData);

		GridLayout documentInformationGroupLayout = new GridLayout();
		documentInformationGroupLayout.numColumns = 2;
		documentInformationGroupLayout.makeColumnsEqualWidth = true;
		identityInformationGroup.setLayout(documentInformationGroupLayout);

		createNameAndURIComposite(identityInformationGroup);
		createDescriptionComposite(identityInformationGroup);
	}

	protected void createDescriptionComposite(Group identityInformationGroup) {
		Composite descriptionComposite = new Composite(
				identityInformationGroup, SWT.NONE);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.grabExcessVerticalSpace = true;
		descriptionComposite.setLayoutData(gd);

		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.makeColumnsEqualWidth = false;
		descriptionComposite.setLayout(layout);

		createFormLabel(descriptionComposite,
				SMLMessages.exportWizardDescriptionLabel);

		gd = new GridData();
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.FILL;
		gd.verticalAlignment = SWT.FILL;
		gd.grabExcessHorizontalSpace = true;
		gd.grabExcessVerticalSpace = true;

		descriptionText = new Text(descriptionComposite, SWT.MULTI | SWT.BORDER | SWT.V_SCROLL | SWT.WRAP);
		descriptionText.setLayoutData(gd);
	}

	protected void createNameAndURIComposite(Group identityInformationGroup) {
		Composite nameAndURIComposite = new Composite(identityInformationGroup,
				SWT.NONE);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.grabExcessVerticalSpace = true;
		gd.verticalAlignment = SWT.FILL;
		nameAndURIComposite.setLayoutData(gd);

		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.makeColumnsEqualWidth = false;
		nameAndURIComposite.setLayout(layout);

		createFormLabel(nameAndURIComposite, SMLMessages.exportWizardNameLabel);

		gd = new GridData();
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.FILL;
		gd.grabExcessHorizontalSpace = true;

		nameText = new Text(nameAndURIComposite, SWT.SINGLE | SWT.BORDER);
		nameText.setLayoutData(gd);

		createFormLabel(nameAndURIComposite, SMLMessages.exportWizardVersionLabel);

		versionText = new Text(nameAndURIComposite, SWT.SINGLE | SWT.BORDER);
		versionText.setLayoutData(gd);

		createFormLabel(nameAndURIComposite, SMLMessages.exportWizardDisplayNameLabel);

		displayNameText = new Text(nameAndURIComposite, SWT.SINGLE | SWT.BORDER);
		displayNameText.setLayoutData(gd);

		createFormLabel(nameAndURIComposite,
				SMLMessages.exportWizardBaseURILabel);

		baseURIText = new Text(nameAndURIComposite, SWT.SINGLE | SWT.BORDER);
		baseURIText.setLayoutData(gd);
		SMLCommonUtil.addURIValidator(baseURIText, new SMLPage(this));
	}

	protected void initDocumentIdentityFields() 
	{
		String projectName = ((ExportToSMLIFWizard)getWizard()).getMainPage().getProjectName();
		FileSystemSMLRepository repository = projectName != null ? 
			(FileSystemSMLRepository)SMLRepositoryManager.getRepository(ResourcesPlugin.getWorkspace().getRoot().getProject(projectName).getLocation().toString()) :
			(FileSystemSMLRepository)SMLRepositoryManager.getRepository();
		
		MetadataFileHandler metadataHandler = repository.getMetadataProcessor().getMetadataHandler();		
		if (nameText.getText().equals("")) { //$NON-NLS-1$
			String nameFromMetadata = metadataHandler.getIdentity().getName();
			nameText.setText((nameFromMetadata == null) ? "" : nameFromMetadata); //$NON-NLS-1$
		}
		if (baseURIText.getText().equals("")) { //$NON-NLS-1$
			String baseURI = metadataHandler.getIdentity().getBaseURI();
			baseURIText.setText((baseURI == null) ? "" : baseURI); //$NON-NLS-1$
		}
		if (displayNameText.getText().equals("")) { //$NON-NLS-1$
			String displayName = metadataHandler.getIdentity().getDisplayName();
			displayNameText.setText((displayName == null) ? "" : displayName); //$NON-NLS-1$
		}
		if (versionText.getText().equals("")) { //$NON-NLS-1$
			String version = metadataHandler.getIdentity().getVersion();
			versionText.setText((version == null) ? "" : version); //$NON-NLS-1$
		}
		if (descriptionText.getText().equals("")) { //$NON-NLS-1$
			String descriptionFromMetadata = metadataHandler.getIdentity().getDescription();
			descriptionText.setText((descriptionFromMetadata == null) ? "" : descriptionFromMetadata); //$NON-NLS-1$
		}
		
		// Copy the aliases to the alias map
		Object[] documents = getModel();
		for (int i = 0; i < documents.length; i++)
		{
			ISMLMetadata metadata = ((ISMLDocument)documents[i]).getMetadata();
			String[] documentAliases = metadata.getAliases();
			
			if (documentAliases != null && documentAliases.length > 0)
				aliases.put(metadata.getId(), metadata.getAliases());
		}
	}

	protected void createDetectedDocumentsLabel(Composite topComponent) {
		Label detectedDocumentsLabel = new Label(topComponent, SWT.NONE);
		detectedDocumentsLabel
				.setText(SMLMessages.exportWizardDetectedDocumentsLabel);
		GridData detectedDocumentsLabelData = new GridData(
				GridData.GRAB_HORIZONTAL | GridData.FILL_HORIZONTAL);
		detectedDocumentsLabelData.grabExcessHorizontalSpace = true;
		detectedDocumentsLabelData.horizontalSpan = 2;
		detectedDocumentsLabel.setLayoutData(detectedDocumentsLabelData);
	}

	protected void createDocumentInformationGroup(Composite topComponent) {
		Group documentInformationGroup = new Group(topComponent, SWT.NONE);
		documentInformationGroup
				.setText(SMLMessages.exportWizardDocumentInformationGroupLabel);
		GridData documentInformationGroupData = new GridData(
				GridData.GRAB_HORIZONTAL | GridData.FILL_BOTH);
		documentInformationGroupData.horizontalSpan = 1;
		documentInformationGroup.setLayoutData(documentInformationGroupData);

		GridLayout documentInformationGroupLayout = new GridLayout();
		documentInformationGroupLayout.numColumns = 2;
		documentInformationGroupLayout.makeColumnsEqualWidth = true;
		documentInformationGroup.setLayout(documentInformationGroupLayout);

		createDetectedDocumentsLabel(documentInformationGroup);
		createDocumentList(documentInformationGroup);
		createDocumentDetailComposite(documentInformationGroup);
	}

	protected void createDocumentDetailComposite(Group documentInformationGroup) {
		documentDetailComposite = new Composite(documentInformationGroup,
				SWT.NONE);
		GridData gd = new GridData(GridData.FILL_BOTH);
		// gd.horizontalSpan = GridData.FILL;
		documentDetailComposite.setLayoutData(gd);
		documentDetailComposite.setEnabled(false);

		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.makeColumnsEqualWidth = false;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		documentDetailComposite.setLayout(layout);

		createTypeControls(documentDetailComposite);
		createAliasControls(documentDetailComposite);
	}

	protected void createAliasControls(Composite parent) {
		createFormLabel(parent, SMLMessages.exportWizardAliasesLabel);

		aliasListViewer = createStringListViewer(parent);

		createAliasAddRemoveButtons(parent);
	}

	protected void createFormLabel(Composite parent, String labelText) {
		Label aliasLabel = new Label(parent, SWT.RIGHT);
		aliasLabel.setText(labelText);
		GridData gd = new GridData();
		gd.horizontalSpan = 1;
		gd.verticalAlignment = SWT.BEGINNING;
		gd.horizontalAlignment = SWT.FILL;
		aliasLabel.setLayoutData(gd);
	}

	protected ListViewer createStringListViewer(Composite parent) {
		GridData gd;
		ListViewer listViewer = new ListViewer(parent);
		gd = new GridData();
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.FILL;
		gd.verticalAlignment = SWT.FILL;
		gd.grabExcessHorizontalSpace = true;
		gd.grabExcessVerticalSpace = true;
		listViewer.getControl().setLayoutData(gd);
		listViewer.setLabelProvider(new LabelProvider() {
			public String getText(Object element) {
				return (String) element;
			}
		});
		listViewer.setContentProvider(new ArrayContentProvider());
		return listViewer;
	}

	protected void createAliasAddRemoveButtons(Composite parent) {
		Composite buttonComposite = ExportToSMLIFWizard.createVerticalButtonPanel(parent);

		Button addButton = ExportToSMLIFWizard.createVerticalButtonPanelButton(buttonComposite,
				SMLMessages.exportWizardAddButtonLabel);
		addButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				IInputValidator validator = new IInputValidator() {
					public String isValid(String newText) {
						if ("".equals(newText.trim())) { //$NON-NLS-1$
							return null;
						}
						boolean valid = false;
						try {
							valid = new URI(newText).getPath().equals(newText);
						} catch (URISyntaxException e) {
						}
						if (valid) {
							return null;
						}
						return NLS.bind(
								SMLMessages.exportWizardErrorURINotValid,
								newText);
					}
				};
				InputDialog input = new InputDialog(getShell(),
						SMLMessages.exportWizardNewAliasTitle,
						SMLMessages.exportWizardNewAliasPrompt, "", validator); //$NON-NLS-3$ //$NON-NLS-1$ //$NON-NLS-1$ //$NON-NLS-1$ //$NON-NLS-1$ //$NON-NLS-1$ //$NON-NLS-1$
				if (input.open() == InputDialog.OK) 
				{
					String[] documentAliases = aliases.get(getSelectedModelUnitFile().getMetadata().getId());
					if (documentAliases == null)
					{
						aliases.put(getSelectedModelUnitFile().getMetadata().getId(), new String[]{input.getValue().trim()});
					}
					else
					{
						String newAliasEntry = input.getValue().trim();
						for (int i = 0; i < documentAliases.length; i++)
						{
							// Don't add this alias if it's a repeat
							if (documentAliases[i].equals(newAliasEntry))
								return;
						}
						
						String[] newAliases = new String[documentAliases.length + 1];
						System.arraycopy(documentAliases, 0, newAliases, 0, documentAliases.length);														
						newAliases[documentAliases.length] = input.getValue().trim();
						aliases.put(getSelectedModelUnitFile().getMetadata().getId(), newAliases);
					}
										
					aliasListViewer.refresh();
					modelFileListViewer.setSelection(modelFileListViewer.getSelection());
					getExportWizard().page3.updateComboChoices();
				}
			}
		});

		Button removeButton = ExportToSMLIFWizard.createVerticalButtonPanelButton(buttonComposite, SMLMessages.exportWizardRemoveButtonText);
		removeButton.addSelectionListener(new SelectionAdapter() 
		{
			public void widgetSelected(SelectionEvent arg0) 
			{
				String[] documentAliases = aliases.get(getSelectedModelUnitFile().getMetadata().getId());
				String[] selectedAliases = aliasListViewer.getList().getSelection();
				List<String> updatedList = new ArrayList<String>();
				boolean foundEntry;
				for (int i = 0; i < documentAliases.length; i++)
				{
					foundEntry = false;
					for (int j = 0; j < selectedAliases.length; j++)
					{
						if (documentAliases[i].equals(selectedAliases[j]))
						{
							foundEntry = true;
							break;
						}
					}
					
					if (!foundEntry)
						updatedList.add(documentAliases[i]);
				}
				
				aliases.put(getSelectedModelUnitFile().getMetadata().getId(), updatedList.toArray(new String[updatedList.size()]));					
				aliasListViewer.refresh();
				modelFileListViewer.setSelection(modelFileListViewer.getSelection());
				getExportWizard().page3.updateComboChoices();
			}
		});
	}

	protected Composite createVerticalButtonPanel(Composite parent) {
		Composite buttonComposite = new Composite(parent, SWT.NONE);
		GridLayout buttonLayout = new GridLayout();
		buttonLayout.numColumns = 1;
		buttonLayout.marginHeight = 0;
		buttonLayout.marginWidth = 0;
		buttonComposite.setLayout(buttonLayout);
		GridData buttonCompositeData = new GridData();
		buttonCompositeData.verticalAlignment = GridData.FILL;
		buttonComposite.setLayoutData(buttonCompositeData);
		return buttonComposite;
	}

	protected void createTypeControls(Composite parent) 
	{
		Label typeLabel = new Label(parent, SWT.RIGHT);
		typeLabel.setText(SMLMessages.exportWizardTypeLabel);
		GridData gd = new GridData();
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.FILL;
		typeLabel.setLayoutData(gd);

		typeCombo = new Combo(parent, SWT.READ_ONLY);
		typeCombo.add(SMLMessages.exportWizardTypeComboDefinition);
		typeCombo.add(SMLMessages.exportWizardTypeComboInstance);
		typeCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				if (typeCombo.getText().equals(SMLMessages.exportWizardTypeComboDefinition)) 
				{
					((SMLFileMetadata)getSelectedModelUnitFile().getMetadata()).setDocumentType(ISMLMetadata.DOCUMENT_TYPE_DEFINITION);
				}
				if (typeCombo.getText().equals(SMLMessages.exportWizardTypeComboInstance)) 
				{
					((SMLFileMetadata)getSelectedModelUnitFile().getMetadata()).setDocumentType(ISMLMetadata.DOCUMENT_TYPE_INSTANCE);
				}
			}
		});
		gd = new GridData();
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.FILL;
		gd.grabExcessHorizontalSpace = true;
		typeCombo.setLayoutData(gd);

		Label spacer = new Label(parent, SWT.NONE);
		gd = new GridData();
		gd.horizontalSpan = 1;
		gd.horizontalAlignment = SWT.FILL;
		spacer.setLayoutData(gd);
	}

	protected void createDocumentList(Group documentInformationGroup) {
		modelFileListViewer = new ListViewer(documentInformationGroup,
				SWT.SINGLE | SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
		modelFileListViewer.setContentProvider(new ArrayContentProvider() {
			public void inputChanged(Viewer viewer, Object oldInput,
					Object newInput) {
				super.inputChanged(viewer, oldInput, newInput);
			}
		});
		modelFileListViewer.setLabelProvider(new LabelProvider() 
		{
			public String getText(Object element) 
			{
				return SMLRepositoryUtil.getDocumentName((ISMLDocument) element);
			}
		});
		modelFileListViewer.setSorter(new ViewerSorter() {
			public int compare(Viewer viewer, Object resource1, Object resource2) {
				return ((ISMLDocument) resource1).getMetadata().getId().compareTo(
						((ISMLDocument) resource2).getMetadata().getId());
			}
		});
		
		modelFileListViewer
				.addSelectionChangedListener(new ISelectionChangedListener() {
					public void selectionChanged(SelectionChangedEvent event) {
						if (event.getSelection().isEmpty()) {
							resetDocumentDetailComposite();
						} else {
							documentDetailComposite.setEnabled(true);
							ISMLDocument selectedModelUnitFile = (ISMLDocument) ((IStructuredSelection) event
									.getSelection()).getFirstElement();
							
							// Populate the alias list						
							aliasListViewer.setInput(aliases.get(selectedModelUnitFile.getMetadata().getId()));
							
							typeCombo.select(selectedModelUnitFile.getMetadata().getDocumentType() == ISMLMetadata.DOCUMENT_TYPE_DEFINITION ? 
											typeCombo.indexOf(SMLMessages.exportWizardTypeComboDefinition) : 
											typeCombo.indexOf(SMLMessages.exportWizardTypeComboInstance));
						}
					}
				});

		GridData listViewerData = new GridData(SWT.FILL, SWT.FILL, true, true);
		modelFileListViewer.getControl().setLayoutData(listViewerData);
	}

	/*
	 * Clean up document detail composite when no document is selected
	 */
	protected void resetDocumentDetailComposite() {
		documentDetailComposite.setEnabled(false);
		aliasListViewer.setInput(new Object[] {});
		typeCombo.deselectAll();
	}

	public String getSMLIFName() {
		return nameText.getText();
	}

	public String getBaseURI() {
		return baseURIText.getText();
	}

	public String getVersion() {
		return versionText.getText();
	}

	public String getDisplayName() {
		return displayNameText.getText();
	}

	public String getSMLIFDescription() {
		return descriptionText.getText();
	}

	protected ISMLDocument getSelectedModelUnitFile() {
		return (ISMLDocument) ((IStructuredSelection) modelFileListViewer
				.getSelection()).getFirstElement();
	}

	protected ExportToSMLIFWizard getExportWizard() {
		return (ExportToSMLIFWizard) getWizard();
	}

	protected ExportToSMLIFWizardPage getMainPage() {
		return (ExportToSMLIFWizardPage) getWizard().getStartingPage();
	}

	protected Object[] getModel() 
	{
		Collection<?> selectedFiles = getExportWizard().getSelectedFiles().values();
		return selectedFiles.size() == 0 ? new ISMLDocument[0] : (Object[]) selectedFiles.toArray();
	}

	public void setVisible(boolean visible) {
		super.setVisible(visible);
		initFromMetadata();
		
		if (visible)
			modelFileListViewer.setInput(getModel());
	}

	protected void initFromMetadata() {
		if (uninitializedIdentityFields) {
			initDocumentIdentityFields();
			getExportWizard().page3.ruleBindingsTableViewer.refresh();
			uninitializedIdentityFields = false;
		}
	}

	public Map<String, String[]> getAliases()
	{
		return aliases;
	}

}
