/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.smlif.internal.validate.ui;

import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.cosmos.rm.smlif.internal.SMLPlugin;
import org.eclipse.cosmos.rm.validation.internal.SMLActivator;
import org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput;
import org.eclipse.cosmos.rm.validation.internal.common.FileOutput;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationConstants;
import org.eclipse.cosmos.rm.validation.internal.common.IValidationMessage;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.validation.internal.core.IValidator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.NLS;

/**
 * A marker output is used in an Eclipse environment to report validation
 * messages to the problem view 
 * 
 * @author Ali Mehregani
 */
public class MarkerOutput extends AbstractValidationOutput
{
	/**
	 * The ID of the Service Modeling Language marker
	 */
	public static final String SML_MARKER_ID = SMLPlugin.PLUGIN_ID + ".smlMarker";
	
	/**
	 * The workspace root
	 */
	private IWorkspaceRoot workspaceRoot;
	
	/**
	 * The attributes describing the settings
	 */
	private Map<String, Object> settings;
	
	/**
	 * Used as the backup validation output
	 */
	private FileOutput backupOutput;
		
	/**
	 * Indicates if an error message has already been reported
	 */
	private boolean errorReported;

	
	public MarkerOutput()
	{
		workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();		
		errorReported = false;
	}	
	
	/** 
	 * @see org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> attributes)
	{
		super.initialize(attributes);
		this.settings = attributes;
		
		try
		{
			String files = (String)attributes.get(IValidator.ATTRIBUTE_INSTANCE);			
			if(files == null)
			{
				return;
			}
			
			StringTokenizer tokens = new StringTokenizer(files, ",");
			while(tokens.hasMoreTokens())
			{
				IPath path = new Path(tokens.nextToken());
				IFile res = ResourcesPlugin.getWorkspace().getRoot().getFileForLocation(path);
				if ((res == null) || (!res.exists())) {
					// must be workspace relative
					res = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
				}
				if (res != null && res.exists())
				{
					IMarker[] markers = res.findMarkers(SML_MARKER_ID, false, IResource.DEPTH_ZERO);
					ResourcesPlugin.getWorkspace().deleteMarkers(markers);
				}
				
			}			
		}
		catch (Exception e)
		{
			SMLActivator.logError(e);
			e.printStackTrace();
		}
	}	
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput#reportMessage(org.eclipse.cosmos.rm.validation.internal.common.AbstractValidationOutput.ValidationMessage)
	 */
	public void writeMessageToSource(IValidationMessage validationMessage)
	{
		IMarker marker;
		try
		{						
			/* The attribute takes priority */
			Object resourceAttribute = validationMessage.getAttribute(IValidationMessage.ATTRIBUTE_RESOURCE);
			IFile file = null;
			if (resourceAttribute instanceof String)
			{
				IPath path = new Path((String)resourceAttribute);
				IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
				file = root.getFileForLocation(path);
				if (file == null || !file.exists())
					file = root.getFile(path);
				
				if (file == null || !file.exists())
					file = null;
			}
			
			IResource resource = file == null ? (getAssociatedResource() == null || !(getAssociatedResource().getResource() instanceof IResource)? workspaceRoot : (IResource)getAssociatedResource().getResource()) : file;
			int severity = validationMessage.getAttribute(ValidationMessage.ATTRIBUTE_SEVERITY, ValidationMessage.SEVERITY_INFO);
			marker = resource.createMarker(SML_MARKER_ID);
			marker.setAttribute(IMarker.MESSAGE, validationMessage.getAttribute(ValidationMessage.ATTRIBUTE_MESSAGE, IValidationConstants.EMPTY_STRING));
			marker.setAttribute(IMarker.SEVERITY, severity == ValidationMessage.SEVERITY_ERROR ? IMarker.SEVERITY_ERROR :
												 (severity == ValidationMessage.SEVERITY_WARNING ? IMarker.SEVERITY_WARNING : IMarker.SEVERITY_INFO));
			
			int lineNumber = validationMessage.getAttribute(ValidationMessage.ATTRIBUTE_LINE_NUMBER, ValidationMessage.NO_LINE_NUMBER);
			if (lineNumber > ValidationMessage.NO_LINE_NUMBER)
			{
				marker.setAttribute(IMarker.LINE_NUMBER, lineNumber);
			}			
			int columnNumber = validationMessage.getAttribute(ValidationMessage.ATTRIBUTE_COLUMN_NUMBER, ValidationMessage.NO_LINE_NUMBER);
			if (columnNumber > ValidationMessage.NO_LINE_NUMBER)
			{
				marker.setAttribute(IMarker.CHAR_START, columnNumber);
			}			
		} 
		catch (CoreException e)
		{
			if (backupOutput == null)
			{
				backupOutput = new FileOutput();
				backupOutput.initialize(settings);
			}
			
			if (!errorReported)
			{
				errorReported = true;
				MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
						SMLValidationMessages.commonError, NLS.bind(SMLValidationMessages.errorMarkerOutput, backupOutput.getFilePath()));
			}
			
			backupOutput.reportMessage(validationMessage);
		}
	}
}
