/*******************************************************************************
 * Copyright (c) 2008 CA Inc. and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     CA - initial API and implementation
 *     IBM - provisional API change
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.metadata.transform;

import java.io.InputStream;
import java.io.Reader;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IMetadataArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IQueryServiceMetadata;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.MetadataArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.TransformerException;
import org.xml.sax.InputSource;

/**
 * Conversions between XML query structures and Java object graphs
 * 
 * @provisonal
 * @author Bill Muldoon
 */
public class QueryMetadataTransformer {

	/**
	 * Convert an IQueryServiceMetadata into the XML syntax for queries described in the CMBDf spec
	 * 
	 * @param queryServiceMetadata
	 * @return an input stream on the XML containing the <queryServiceMetadata> structure
	 */
	public static InputStream transform(IQueryServiceMetadata queryServiceMetadata) {
		return CMDBfServicesUtil.transformImpl((IXMLWritable) queryServiceMetadata);
	}

	/**
	 * Convert input stream containing an XML structure for a CMDBf queryServiceMetadata into a
	 * Java object graph
	 * 
	 * @param xmlInputStream
	 * @return IQueryServiceMetadata object structure representing a query
	 * @throws TransformerException
	 */
	public static IQueryServiceMetadata transform(InputStream xmlInputStream) throws TransformerException {
		return transform (xmlInputStream, MetadataArtifactFactory.getInstance(), null);	
	}
	
	/**
	 * Convert input stream containing an XML structure for a CMDBf queryServiceMetadata into a
	 * Java object graph.  Uses the provided artifact factory to create custom POJO artifacts
	 * 
	 * @param xmlInputStream
	 * @param artifactFactory
	 * @return IQueryServiceMetadata object structure representing a query
	 * @throws TransformerException
	 */
	public static IQueryServiceMetadata transform(InputStream xmlInputStream, IMetadataArtifactFactory artifactFactory) throws TransformerException {
		return transform (xmlInputStream, artifactFactory, null);	
	}
	
	/**
	 * Convert input stream containing an XML structure for a CMDBf queryServiceMetadata into a
	 * Java object graph
	 * 
	 * @param reader
	 * @return IQueryServiceMetadata object structure representing a query
	 * @throws TransformerException
	 */
	public static IQueryServiceMetadata transform(Reader reader) throws TransformerException {
		return transform (reader, MetadataArtifactFactory.getInstance(), null);	
	}

	public static IQueryServiceMetadata transform(Reader reader, IMetadataArtifactFactory artifactFactory) throws TransformerException {
		return transform (reader, artifactFactory, null);	
	}

	/**
	 * Convert input stream containing an XML structure for a CMDBf queryServiceMetadata into a
	 * Java object graph.  Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param xmlInputStream
	 * @param transformOptions a Map of options that affect default behavior of the transformer
	 * @return IQueryServiceMetadata object structure representing a query
	 * @throws TransformerException
	 */
	public static IQueryServiceMetadata transform(InputStream xmlInputStream, IMetadataArtifactFactory artifactFactory, Map<String, Object> transformOptions) throws TransformerException
	{
		return transformOptions == null ? 
				(IQueryServiceMetadata) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new ServiceMetadataInputStreamHandler(true, artifactFactory)):
				(IQueryServiceMetadata) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new ServiceMetadataInputStreamHandler(true, artifactFactory), transformOptions);	
	}
	
	/**
	 * Convert input stream containing an XML structure for a CMDBf queryServiceMetadata into a
	 * Java object graph.  Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param reader
	 * @param transformOptions a Map of options that affect default behavior of the transformer
	 * @return IQuery object structure representing a query
	 * @throws TransformerException
	 */
	public static IQueryServiceMetadata transform(Reader reader, IMetadataArtifactFactory artifactFactory, Map<String, Object> transformOptions) throws TransformerException 
	{
		return transformOptions == null ? 
				(IQueryServiceMetadata) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new ServiceMetadataInputStreamHandler(true, artifactFactory)) :
				(IQueryServiceMetadata) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new ServiceMetadataInputStreamHandler(true, artifactFactory), transformOptions);	
	}

}
