/*******************************************************************************
 * Copyright (c) 2008 CA Inc. and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     CA - initial API and implementation
 *     IBM - provisional API change
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.metadata.transform;

import org.eclipse.cosmos.dc.cmdbf.services.common.IRootElement;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IContentSelectorSupport;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IDialect;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IMetadataArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IPropertyValueOperators;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IQueryCapabilities;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IQueryServiceMetadata;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IRecordConstraintSupport;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IRecordType;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IRecordTypeList;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IRecordTypes;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IRegistrationServiceMetadata;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IRelationshipTemplateSupport;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IServiceDescription;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.IXpathSupport;
import org.eclipse.cosmos.dc.cmdbf.services.metadata.transform.input.artifacts.MetadataArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;


/**
 * SAX parser that builds QueryServiceMetadata object from XML input
 * 
 * @provisional
 * @author Bill Muldoon
 */
public class ServiceMetadataInputStreamHandler extends DefaultHandler implements ITransformerHandler {

	private String currentElementData;
	private Boolean bQueryMetadata = true; // true for query metadata, false for registration metadata 
	private IQueryServiceMetadata queryServiceMetadata = null;
	private IRegistrationServiceMetadata registrationServiceMetadata = null;
	private IQueryCapabilities queryCapabilities = null;
	private IRecordConstraintSupport recordConstraintSupport = null;
	private IXpathSupport xpathSupport = null;
	private IRecordTypeList recordTypeList = null;
	private IRecordTypes recordTypes = null;
	private IMetadataArtifactFactory artifactFactory = null;
	
	public ServiceMetadataInputStreamHandler(boolean queryMetadata, IMetadataArtifactFactory artifactFactory) {
		super();
	
		this.artifactFactory = artifactFactory;
		bQueryMetadata = queryMetadata;
	}


	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException 
	{
		if (!ServiceMetadataConstants.SERVICEMETADATA_NAMESPACE.equals(uri))
		{
			return;
		}
		
		// <queryServiceMetadata> element
		if ( bQueryMetadata && ServiceMetadataConstants.QUERYSERVICEMETADATA_ELEMENT.equals(localName)) 
		{
			queryServiceMetadata = artifactFactory.createQueryServiceMetadata();
		}
		
		// <registrationServiceMetadata> element
		else if ( !bQueryMetadata && ServiceMetadataConstants.REGISTRATIONSERVICEMETADATA_ELEMENT.equals(localName)) 
		{
			registrationServiceMetadata = artifactFactory.createRegistrationServiceMetadata();
		}
		
		// <serviceMetadata> element
		else if (ServiceMetadataConstants.SERVICEDESCRIPTION_ELEMENT.equals(localName)) 
		{
			IServiceDescription serviceDescription = artifactFactory.createServiceDescription();
			
			if ( bQueryMetadata ){
				if ( queryServiceMetadata != null ){
					queryServiceMetadata.setServiceDescription( serviceDescription );
				}
			} else {
				if ( registrationServiceMetadata != null ){
					registrationServiceMetadata.setServiceDescription( serviceDescription );
				}				
			}
		}
		
		// <queryCapabilities> element
		else if (bQueryMetadata && ServiceMetadataConstants.QUERYCAPABILITIES_ELEMENT.equals(localName)) 
		{
			queryCapabilities = artifactFactory.createQueryCapabilities();
			
			if ( queryServiceMetadata != null ){
				queryServiceMetadata.setQueryCapabilities(queryCapabilities);
			}
		}
		
		// <relationshipTemplateSupport> element
		else if (bQueryMetadata && ServiceMetadataConstants.RELATIONSHIPTEMPLATESUPPORT_ELEMENT.equals(localName)) 
		{
			IRelationshipTemplateSupport relationshipTemplateSupport = artifactFactory.createRelationshipTemplateSupport();
			
			String depthLimit = attributes.getValue(ServiceMetadataConstants.DEPTHLIMIT_ATTRIBUTE);
			if ( depthLimit != null ){
				relationshipTemplateSupport.setDepthLimit(Boolean.parseBoolean(depthLimit));
			}
			
			String minimumMaximum = attributes.getValue(ServiceMetadataConstants.MINIMUMMAXIMUM_ATTRIBUTE);
			if ( minimumMaximum != null ){
				relationshipTemplateSupport.setMinimumMaximum(Boolean.parseBoolean(minimumMaximum));
			}

			if ( queryCapabilities != null ){
				queryCapabilities.setRelationshipTemplateSupport(relationshipTemplateSupport);
			}
		}
		
		// <contentSelectorSupport> element
		else if (bQueryMetadata && ServiceMetadataConstants.CONTENTSELECTORSUPPORT_ELEMENT.equals(localName)) 
		{
			IContentSelectorSupport contentSelectorSupport = artifactFactory.createContentSelectorSupport();
			
			String recordTypeSelector = attributes.getValue(ServiceMetadataConstants.RECORDTYPESELECTOR_ATTRIBUTE);
			if ( recordTypeSelector != null ){
				contentSelectorSupport.setRecordTypeSelector(Boolean.parseBoolean(recordTypeSelector));
			}
			
			String propertySelector = attributes.getValue(ServiceMetadataConstants.PROPERTYSELECTOR_ATTRIBUTE);
			if ( propertySelector != null ){
				contentSelectorSupport.setPropertySelector(Boolean.parseBoolean(propertySelector));
			}

			if ( queryCapabilities != null ){
				queryCapabilities.setContentSelectorSupport(contentSelectorSupport);
			}
		}
		
		// <recordConstraintSupport> element
		else if (bQueryMetadata && ServiceMetadataConstants.RECORDCONSTRAINTSUPPORT_ELEMENT.equals(localName)) 
		{
			recordConstraintSupport = artifactFactory.createRecordConstraintSupport();
			
			String recordTypeConstraint = attributes.getValue(ServiceMetadataConstants.RECORDTYPECONSTRAINT_ATTRIBUTE);
			if ( recordTypeConstraint != null ){
				recordConstraintSupport.setRecordTypeConstraint(Boolean.parseBoolean(recordTypeConstraint));
			}
			
			String propertyValueConstraint = attributes.getValue(ServiceMetadataConstants.PROPERTYVALUECONSTRAINT_ATTRIBUTE);
			if ( propertyValueConstraint != null ){
				recordConstraintSupport.setPropertyValueConstraint(Boolean.parseBoolean(propertyValueConstraint));
			}

			if ( queryCapabilities != null ){
				queryCapabilities.setRecordConstraintSupport(recordConstraintSupport);
			}
		}
		
		// <propertyValueOperators> element
		else if (bQueryMetadata && ServiceMetadataConstants.PROPERTYVALUEOPERATORS_ELEMENT.equals(localName)) 
		{
			IPropertyValueOperators propertyValueOperators = artifactFactory.createPropertyValueOperators();
			
			String equal = attributes.getValue(ServiceMetadataConstants.EQUAL_ATTRIBUTE);
			if ( equal != null ){
				propertyValueOperators.setEqual(Boolean.parseBoolean(equal));
			}
			
			String less = attributes.getValue(ServiceMetadataConstants.LESS_ATTRIBUTE);
			if ( less != null ){
				propertyValueOperators.setLess(Boolean.parseBoolean(less));
			}
			
			String lessOrEqual = attributes.getValue(ServiceMetadataConstants.LESSOREQUAL_ATTRIBUTE);
			if ( lessOrEqual != null ){
				propertyValueOperators.setLessOrEqual(Boolean.parseBoolean(lessOrEqual));
			}
			
			String greater = attributes.getValue(ServiceMetadataConstants.GREATER_ATTRIBUTE);
			if ( greater != null ){
				propertyValueOperators.setGreater(Boolean.parseBoolean(greater));
			}
			
			String greaterOrEqual = attributes.getValue(ServiceMetadataConstants.GREATEROREQUAL_ATTRIBUTE);
			if ( greaterOrEqual != null ){
				propertyValueOperators.setGreaterOrEqual(Boolean.parseBoolean(greaterOrEqual));
			}
			
			String contains = attributes.getValue(ServiceMetadataConstants.CONTAINS_ATTRIBUTE);
			if ( contains != null ){
				propertyValueOperators.setContains(Boolean.parseBoolean(contains));
			}
			
			String like = attributes.getValue(ServiceMetadataConstants.LIKE_ATTRIBUTE);
			if ( like != null ){
				propertyValueOperators.setLike(Boolean.parseBoolean(like));
			}
			
			String isNull = attributes.getValue(ServiceMetadataConstants.ISNULL_ATTRIBUTE);
			if ( isNull != null ){
				propertyValueOperators.setNull(Boolean.parseBoolean(isNull));
			}

			if ( recordConstraintSupport != null ){
				recordConstraintSupport.setPropertyValueOperators(propertyValueOperators);
			}
		}
		
		// <xpathSupport> element
		else if (bQueryMetadata && ServiceMetadataConstants.XPATHSUPPORT_ELEMENT.equals(localName)) 
		{
			xpathSupport = artifactFactory.createXpathSupport();
			
			if ( queryCapabilities != null ){
				queryCapabilities.setXpathSupport(xpathSupport);
			}
		}
		
		// <recordTypeList> element
		else if (ServiceMetadataConstants.RECORDTYPELIST_ELEMENT.equals(localName)) 
		{
			recordTypeList = artifactFactory.createRecordTypeList();
			
			if ( bQueryMetadata ){
				if ( queryServiceMetadata != null ){
					queryServiceMetadata.setRecordTypeList(recordTypeList);
				}
			} else {
				if ( registrationServiceMetadata != null ){
					registrationServiceMetadata.setRecordTypeList(recordTypeList);
				}				
			}
		}
		
		// <recordTypes> element
		else if (ServiceMetadataConstants.RECORDTYPES_ELEMENT.equals(localName)) 
		{
			recordTypes = artifactFactory.createRecordTypes();
			
			String namespace = attributes.getValue(ServiceMetadataConstants.NAMESPACE_ATTRIBUTE);
			if ( namespace != null ){
				recordTypes.setNamespace(namespace);
			}
			
			String schemaLocation = attributes.getValue(ServiceMetadataConstants.SCHEMALOCATION_ATTRIBUTE);
			if ( schemaLocation != null ){
				recordTypes.setSchemaLocation(schemaLocation);
			}

			if ( recordTypeList != null ){
				recordTypeList.addRecordTypes(recordTypes);
			}
		}
		
		// <recordType> element
		else if (ServiceMetadataConstants.RECORDTYPE_ELEMENT.equals(localName)) 
		{
			IRecordType recordType = artifactFactory.createRecordType();
			
			String lName = attributes.getValue(ServiceMetadataConstants.LOCALNAME_ATTRIBUTE);
			if ( lName != null ){
				recordType.setLocalName(lName);
			}
			
			String appliesTo = attributes.getValue(ServiceMetadataConstants.APPLIESTO_ATTRIBUTE);
			if ( appliesTo != null ){
				recordType.setAppliesTo(appliesTo);
			}

			if ( recordTypes != null ){
				recordTypes.addRecordType(recordType);
			}
		}

	}

	
	public void endElement(String uri, String localName, String name) throws SAXException 
	{
		if (!ServiceMetadataConstants.SERVICEMETADATA_NAMESPACE.equals(uri))
		{
			return;
		}
		
		// <mdrId> element
		if (ServiceMetadataConstants.MDRID_ELEMENT.equals(localName)) 
		{			
			if ( bQueryMetadata ){
				if ( queryServiceMetadata != null && queryServiceMetadata.getServiceDescription() != null ){
					queryServiceMetadata.getServiceDescription().setMdrId( currentElementData );
				}
			} else {
				if ( registrationServiceMetadata != null && registrationServiceMetadata.getServiceDescription() != null ){
					registrationServiceMetadata.getServiceDescription().setMdrId( currentElementData );
				}
			}
		}
		
		// <description> element
		else if (ServiceMetadataConstants.DESCRIPTION_ELEMENT.equals(localName)) 
		{			
			if ( bQueryMetadata ){
				if ( queryServiceMetadata != null && queryServiceMetadata.getServiceDescription() != null ){
					queryServiceMetadata.getServiceDescription().setDescription( currentElementData );
				}
			} else {
				if ( registrationServiceMetadata != null && registrationServiceMetadata.getServiceDescription() != null ){
					registrationServiceMetadata.getServiceDescription().setDescription( currentElementData );
				}				
			}
		}
		
		// <dialect> element
		else if ( bQueryMetadata && ServiceMetadataConstants.DIALECT_ELEMENT.equals(localName)) 
		{
			IDialect dialect = artifactFactory.createDialect();
			dialect.setDialect( currentElementData );
			
			if ( xpathSupport != null ){
				xpathSupport.addDialect(dialect);
			}
		}

		currentElementData = null;

	}
	

	public void characters(char[] ch, int start, int length) throws SAXException 
	{
		String tempData = new String(ch, start, length).trim();
		if (currentElementData == null) {
			currentElementData = tempData;
		} else {
			currentElementData += tempData;
		}
	}

	public IRootElement getResult() {
		if ( bQueryMetadata ){
			return queryServiceMetadata;
		}
		
		return registrationServiceMetadata;
	}

	protected IMetadataArtifactFactory getArtifactFactory() {
		if (artifactFactory == null) {
			artifactFactory = MetadataArtifactFactory.getInstance();
		}
		return artifactFactory;
	}
}
