/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.application.impl;

import java.io.Serializable;

import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.internal.ISMLRepositoryConstants;
import org.eclipse.cosmos.rm.repository.internal.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceFacet;
import org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceInstance;
import org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceProperty;
import org.eclipse.cosmos.rm.repository.internal.reference.SMLQualifiedReference;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.w3c.dom.Node;

/**
 * An implementation of a resource facet.  The {@link #load(Node)} method should
 * be used to instantiate an instance of this class.
 * 
 * @author Ali Mehregani
 */
public class SMLFileResourceFacet extends AbstractSMLFragment implements ISMLResourceFacet, Serializable
{
	/**
	 * Serial version UID
	 */
	private static final long serialVersionUID = 3726281149432601361L;

	/**
	 * The repository
	 */
	private ISMLRepository repository;
	
	/**
	 * The root node
	 */
	private Node root;
	
	/**
	 * Indicates if this is a reference
	 */
	private boolean reference;	
	
	/**
	 * Instantiate an instance of this class based on the DOM
	 * node passed in.
	 * 
	 * @param node The facet node
	 * @return An instance of this class if node represents a facet node;
	 * null otherwise
	 */
	public static SMLFileResourceFacet load(ISMLRepository repository, Node node)
	{					
		if (node == null || node.getNodeType() != Node.ELEMENT_NODE)
			return null;
		
		boolean reference = ISMLRepositoryConstants.URI_ITEM.equals(node.getNamespaceURI()) && ISMLRepositoryConstants.FACET_REF_ELEMENT.equals(node.getLocalName());
		return  new SMLFileResourceFacet(repository, node, reference);				
	}
	
	/**
	 * The constructor
	 * 
	 * @param repository The repository
	 * @param root The root node
	 * @param reference true if this is a reference; false otherwise
	 */
	private SMLFileResourceFacet(ISMLRepository repository, Node root, boolean reference)
	{
		this.repository = repository;
		this.root = root;
		this.reference = reference;
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceFacet#resolveReference()
	 */
	public ISMLResourceFacet resolveReference()
	{
		if (!reference)
			return this;
	
		return load(repository, new SMLQualifiedReference(repository, SMLValidatorUtil.extractReference(root)).resolveReference());
	}

	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceFacet#retrieveFacets()
	 */
	public ISMLResourceFacet[] retrieveFacets()
	{
		return SMLRepositoryUtil.retrieveFacets(((ISMLResourceInstance)getParentDocument()), root, repository);
	}

	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceFacet#retrieveProperties()
	 */
	public ISMLResourceProperty[] retrieveProperties()
	{
		return SMLRepositoryUtil.retrieveProperties(repository, ((ISMLResourceInstance)getParentDocument()), root);
	}

	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceFacet#retrieveResources()
	 */
	public ISMLResourceInstance[] retrieveResources()
	{
		return SMLRepositoryUtil.retrieveResources(repository, ((ISMLResourceInstance)getParentDocument()), root);
	}

	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceBase#isReference()
	 */
	public boolean isReference()
	{
		return reference;
	}

	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.impl.AbstractSMLFragment#getRootNode()
	 */
	protected Node getRootNode()
	{
		return root;
	}
}
