/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.application.impl;

import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.internal.ISMLRepositoryConstants;
import org.eclipse.cosmos.rm.repository.internal.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceFacet;
import org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceInstance;
import org.eclipse.cosmos.rm.repository.internal.reference.SMLQualifiedReference;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLFileDocument;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLFileInstanceDocument;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLFileMetadata;
import org.eclipse.cosmos.rm.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.repository.resource.ISMLMetadata;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

/**
 * The implementation of an SML document representing a resource
 * 
 * @author Ali Mehregani
 */
public class SMLFileResourceInstance extends SMLFileInstanceDocument implements ISMLResourceInstance
{
	/**
	 * Serial version UID
	 */
	private static final long serialVersionUID = 4657641861840507364L;

	
	/**
	 * The reference string if this is a reference
	 */
	private String reference;
	
	/**
	 * Delegates to this document
	 */
	private ISMLDocument delegate;
	
	/**
	 * The root document representing a nested resource
	 */
	private Node root;
	
	/**
	 * The parent document
	 */
	private ISMLDocument parentDocument;

	/**
	 * Instantiate an instance of this class based on the DOM
	 * node passed in.
	 * 
	 * @param node The resource instance node
	 * @return An instance of this class if node represents an instance node;
	 * null otherwise
	 */
	public static SMLFileResourceInstance load(ISMLRepository repository, Node root)
	{		
		if (root == null || root.getNodeType() != Node.ELEMENT_NODE)
			return null;
		
		String reference = null;
		if (ISMLRepositoryConstants.URI_ITEM.equals(root.getNamespaceURI()) && 
			ISMLRepositoryConstants.ITEM_REF_ELEMENT.equals(root.getLocalName()))
		{
			reference = SMLValidatorUtil.extractReference(root);
		}
		
		SMLFileResourceInstance instance = new SMLFileResourceInstance(repository, reference);
		if (reference == null)
		{
			instance.root = root;
		}
		
		return instance;
	}
	
	
	/**
	 * The constructor
	 * 
	 * @param repository The repository
	 * @param reference A flag indicating whether this resource is a reference
	 */
	public SMLFileResourceInstance(ISMLRepository repository, String reference)
	{
		super(repository);
		this.reference = reference;
	}

	
	/**
	 * Constructor
	 * 
	 * @param repository The repository
	 * @param delegate The delegate
	 */
	public SMLFileResourceInstance(ISMLRepository repository, ISMLDocument delegate)
	{
		super(repository);
		this.delegate = delegate;
	}
		
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceInstance#resolveReference()
	 */
	public ISMLResourceInstance resolveReference()
	{
		if (reference == null)
			return this;
	
		try
		{
			SMLQualifiedReference qualifiedReference = new SMLQualifiedReference(getRepository(), reference);
			ISMLDocument[] document = getRepository().fetchDocuments(new SMLFileMetadata(null, ISMLMetadata.DOCUMENT_TYPE_NONE, null, new String[]{qualifiedReference.getDocumentReference()}, null));
			if (document == null || document.length <= 0)
				document = getRepository().fetchDocuments(new SMLFileMetadata(qualifiedReference.getDocumentReference(), ISMLMetadata.DOCUMENT_TYPE_NONE, null, null, null));
			return document != null && document.length > 0 ? new SMLFileResourceInstance(getRepository(), document[0]) : null;
		} 
		catch (RepositoryOperationException e)
		{
			e.printStackTrace();
		}
		
		return null;
		
	}
	

	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceInstance#retrieveFacets()
	 */
	public ISMLResourceFacet[] retrieveFacets() throws RepositoryOperationException
	{
		return SMLRepositoryUtil.retrieveFacets(parentDocument == null ? this : (ISMLResourceInstance)parentDocument, root == null ? (getDOMDocument() == null ? null : getDOMDocument().getFirstChild()) : root, getRepository());
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceBase#isReference()
	 */
	public boolean isReference()
	{
		return reference != null;
	}

	
	/** 
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceBase#retrieveName()
	 */
	public String retrieveName()
	{		
		try
		{
			return retrieveRootAttribute(null, ISMLRepositoryConstants.NAME_ATTRIBUTE);
		} 
		catch (RepositoryOperationException e)
		{
			return null;
		}
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceBase#retrieveDisplayName()
	 */
	public String retrieveDisplayName()
	{
		try
		{
			return retrieveRootAttribute(null, ISMLRepositoryConstants.DISPLAY_NAME_ATTRIBUTE);
		} 
		catch (RepositoryOperationException e)
		{
			return null;
		}
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.application.ISMLResourceBase#retrieveDescription()
	 */
	public String retrieveDescription()
	{
		try
		{
			return retrieveRootAttribute(null, ISMLRepositoryConstants.DESCRIPTION_ATTRIBUTE);
		} 
		catch (RepositoryOperationException e)
		{
			return null;
		}
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.resource.SMLFileDocument#getDOMDocument()
	 */
	public Document getDOMDocument() throws RepositoryOperationException
	{		
		return delegate instanceof SMLFileDocument ? ((SMLFileDocument)delegate).getDOMDocument() : super.getDOMDocument();
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.resource.SMLFileDocument#getRepository()
	 */
	public ISMLRepository getRepository()
	{
		return delegate instanceof SMLFileDocument ? ((SMLFileDocument)delegate).getRepository() : super.getRepository();
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.resource.SMLFileDocument#retrieveRootAttribute(java.lang.String, java.lang.String)
	 */
	public String retrieveRootAttribute(String uri, String localName) throws RepositoryOperationException
	{
		return delegate instanceof SMLFileDocument ? ((SMLFileDocument)delegate).retrieveRootAttribute(uri, localName) : super.retrieveRootAttribute(uri, localName);
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.resource.SMLFileDocument#getMetadata()
	 */
	public ISMLMetadata getMetadata()
	{
		return delegate == null ? super.getMetadata() : delegate.getMetadata(); 
	}
	
	
	public void setParentDocument(ISMLDocument parentDocument)
	{
		this.parentDocument = parentDocument;
	}
	
	
	public ISMLDocument getParentDocument()
	{
		return parentDocument;
	}
}
