/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.core;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.internal.CMDBfInternalUtility;
import org.eclipse.cosmos.rm.repository.internal.ISMLRepositoryConstants;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLIFIdentity;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * SAX Handler for parsing a metadata file to determined saved alias, rule binding,
 * and document identity information.  This allows no loss of information when a set of
 * documents that have just been imported is exported.
 * 
 * @author David Whiteman
 * @author Ali Mehregani
 */
public class MetadataFileHandler extends DefaultHandler 
{	
	/**
	 * Elements
	 */
	private static final String ELEMENT_FILE = "file";
	private static final String ELEMENT_METADATA = "metadata";
	
	/**
	 * Attributes
	 */
	private static final String ATTRIBUTE_PATH = "path";

	
	/**
	 * An array of flags used to identify the current position that is
	 * being parsed in the meta-data document.
	 * The ordering is as follows:
	 * inAlias, inBaseURI, inDescription, inName, inDisplayName, 
	 * inVersion, inDocumentAlias, inRuleAlias
	 */
	private final boolean[] POSITION_INDICATOR = new boolean[8];
	
	/**
	 * Keeps an (element name, index) map.
	 * KEY = Element name
	 * Value = An integer storing the index that it corresponds to
	 */
	private static final Map<String, Integer> elementIndexMap;
	
	/**
	 * Map holding a list of file paths for each alias
	 * KEY = alias
	 * VALUE = A list of file paths
	 */
	private Map<String, List<String>> aliasFileMap;
	
	/**
	 * Map holding a list of aliases for each file
	 * KEY = File path
	 * VALUE = A list of aliases
	 */
	private Map<String, List<String>> fileAliasMap;
	
	/**
	 * Buffer to hold a text node
	 */
	private StringBuffer buffer;
	
	/**
	 * Most recent file element processed.  used for setting up aliases for the file.
	 */
	private String lastFile;

	/**
	 * The SML-IF identity
	 */
	private SMLIFIdentity identity;
	
	/**
	 * The rule bindings indexed by document aliases
	 */	
	private Map<String, List<String>> aliasRuleBinding;
	
	/**
	 * The rule bindings indexed by rule prefix
	 */
	private Map<String, List<String>> ruleAliasBinding;
	
	/**
	 * Rule alias buffer
	 */
	private StringBuffer currentRuleAliasBuffer;
	
	/**
	 * The dirty status of this handler
	 */
	private boolean dirty;
	
	static
	{
		final String[] KEYS = new String[]{
				ISMLConstants.ALIAS_ELEMENT, ISMLConstants.BASE_URI_ELEMENT, ISMLConstants.DESCRIPTION_ELEMENT,
				ISMLConstants.NAME_ELEMENT, ISMLConstants.DISPLAY_NAME_ELEMENT, ISMLConstants.VERSION_ELEMENT,
				ISMLConstants.DOCUMENTALIAS_ELEMENT, ISMLConstants.RULEALIAS_ELEMENT};	
		
		elementIndexMap = new Hashtable<String, Integer>();
		for (int i = 0; i < KEYS.length; i++)
		{
			elementIndexMap.put(KEYS[i], new Integer(i));
		}
	}
	
	public MetadataFileHandler()
	{
		aliasRuleBinding = new Hashtable<String, List<String>>();
		ruleAliasBinding = new Hashtable<String, List<String>>();
		aliasFileMap = new Hashtable<String, List<String>>();
		fileAliasMap = new Hashtable<String, List<String>>();
		buffer = new StringBuffer();
		currentRuleAliasBuffer = new StringBuffer();		
		identity = new SMLIFIdentity();
	}		
	

	public void characters(char[] ch, int start, int length) throws SAXException 
	{
		String elementData = new String(ch, start, length);
		String trimmedElementData = elementData.trim();
		
		if (POSITION_INDICATOR[0] || POSITION_INDICATOR[1] || POSITION_INDICATOR[3] || POSITION_INDICATOR[4] || 
			POSITION_INDICATOR[5] || POSITION_INDICATOR[6])
		{
			buffer.append(trimmedElementData);
		}
		else if (POSITION_INDICATOR[2])
		{
			buffer.append(elementData);
		}
		else if (POSITION_INDICATOR[7])
		{
			currentRuleAliasBuffer.append(trimmedElementData);
		}
				
		super.characters(ch, start, length);
	}

	public void endElement(String uri, String localName, String qName) throws SAXException 
	{
		if (ISMLConstants.ALIAS_ELEMENT.equals(qName)) 
		{			
			String alias = buffer.toString().trim();
			addToList(aliasFileMap, alias, lastFile);
			addToList(fileAliasMap, lastFile, alias);
		}
		else if (ISMLConstants.BASE_URI_ELEMENT.equals(qName)) 
		{
			identity.setBaseURI(buffer.toString().trim());
		}
		else if (ISMLConstants.NAME_ELEMENT.equals(qName)) 
		{
			identity.setName(buffer.toString().trim());
		}
		else if (ISMLConstants.VERSION_ELEMENT.equals(qName)) 
		{
			identity.setVersion(buffer.toString().trim());
		}
		else if (ISMLConstants.DISPLAY_NAME_ELEMENT.equals(qName)) 
		{		
			identity.setDisplayName(buffer.toString().trim());
		}
		else if (ISMLConstants.DESCRIPTION_ELEMENT.equals(qName)) 
		{		
			identity.setDescription(SMLValidatorUtil.removeLineBreaks(buffer.toString()));
		}
		else if (ISMLConstants.RULEBINDING_ELEMENT.equals(qName)) 
		{
			if (currentRuleAliasBuffer.length() > 0) 
			{
				String docAlias = null;
				if (buffer.length() > 0) {
					docAlias = buffer.toString().trim();;
				}
				String ruleAlias = currentRuleAliasBuffer.toString().trim();
				
				addToList (aliasRuleBinding, docAlias, ruleAlias);
				addToList (ruleAliasBinding, ruleAlias, docAlias);		
			}
			buffer = new StringBuffer();
			currentRuleAliasBuffer = new StringBuffer();
		} 
		
		int inx = getIndex(qName);
		if (inx >= 0)
		{
			POSITION_INDICATOR[inx] = false;
			if (inx <= 5)
				buffer = new StringBuffer();
		}		
		
		super.endElement(uri, localName, qName);
	}

	private void addToList(Map<String, List<String>> storage, String key, String value)
	{
		List<String> list = storage.get(key);
		if (list == null)
		{
			list = new ArrayList<String>();
			storage.put(key, list);
		}
		
		if (!list.contains(value))
			list.add(value);
	}

	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException 
	{		
		int inx = getIndex(qName); 
		if (inx >= 0)
		{
			POSITION_INDICATOR[inx] = true;
		}
		else if (ELEMENT_FILE.equals(qName)) 
		{
			lastFile = attributes.getValue(ATTRIBUTE_PATH);
		}
		
		
		super.startElement(uri, localName, qName, attributes);
	}

	private int getIndex(String key)
	{
		Integer inx = (Integer)elementIndexMap.get(key);		

		if (inx != null)
			return inx.intValue();
		return -1;
	}


	public Map<String, List<String>> getAliasFileMap() 
	{
		return aliasFileMap;
	}
	
	public Map<String, List<String>> getFileAliasMap() 
	{
		return fileAliasMap;
	}

	public Map<String, List<String>> getRuleAliasBinding()
	{
		return ruleAliasBinding; 
	}
	
	public Map<String, List<String>> getAliasRuleBinding()
	{
		return aliasRuleBinding; 
	}
	
	public SMLIFIdentity getIdentity() 
	{
		return identity;
	}


	public void addAlias(String id, String[] aliases)
	{
		dirty = true;
		for (int i = 0; i < aliases.length; i++)
		{
			addToList(fileAliasMap, id, aliases[i]);
			addToList(aliasFileMap, aliases[i], id);
		}
	}


	public void addRuleBinding(String alias, String[] rules)
	{
		dirty = true;
		for (int i = 0; i < rules.length; i++)
		{
			addToList(ruleAliasBinding, rules[i], alias);
			addToList(aliasRuleBinding, alias, rules[i]);
		}
	}


	/**
	 * Writes the meta information to the print stream passed in
	 * 
	 * @param printStream The print stream used to write the meta information
	 */
	public void write(PrintStream printStream)
	{
		printStream.println(ISMLRepositoryConstants.XML_DECLARATION);
		printStream.println(SMLValidatorUtil.beginTagFor(ELEMENT_METADATA));
		
		writeFields (printStream, new String[][] {
				new String[]{ISMLConstants.NAME_ELEMENT, identity.getName()}, 
				new String[]{ISMLConstants.DISPLAY_NAME_ELEMENT, identity.getDisplayName()}, 
				new String[]{ISMLConstants.VERSION_ELEMENT, identity.getVersion()},
				new String[]{ISMLConstants.BASE_URI_ELEMENT, identity.getBaseURI()}
				});
		
		if (identity.getDescription() != null)
		{
			printStream.println(ISMLConstants.tab+ SMLValidatorUtil.beginTagFor(ISMLConstants.DESCRIPTION_ELEMENT) + identity.getDescription() + 
								ISMLConstants.nl + ISMLConstants.tab + SMLValidatorUtil.endTagFor(ISMLConstants.DESCRIPTION_ELEMENT));
		}
		
		// Write the rule bindings
		for (Iterator<String> rules = ruleAliasBinding.keySet().iterator(); rules.hasNext();) 
		{
			String currentRule = rules.next();
			List<String> aliases = ruleAliasBinding.get(currentRule);
			
			for (int i = 0, aliasCount = aliases.size(); i < aliasCount; i++)
			{
				// Write the rule binding begin tag
				printStream.println(ISMLConstants.tab + SMLValidatorUtil.beginTagFor(ISMLConstants.RULEBINDING_ELEMENT));
				String docAlias = (String)aliases.get(i);
				if (docAlias != null) 
				{
					// Write the doc alias tag
					printStream.println(ISMLConstants.tab + ISMLConstants.tab
							+ SMLValidatorUtil.createElementTag(ISMLConstants.DOCUMENTALIAS_ELEMENT, docAlias));
				}
				
				// Write the rule alias tag
				printStream.println(ISMLConstants.tab + ISMLConstants.tab + SMLValidatorUtil.createElementTag(ISMLConstants.RULEALIAS_ELEMENT, currentRule));
				
				// Write the rule binding end tag
				printStream.println(ISMLConstants.tab + SMLValidatorUtil.endTagFor(ISMLConstants.RULEBINDING_ELEMENT));				
			}
		}
		
		// Write the document aliases	
		for (Iterator<String> files = fileAliasMap.keySet().iterator(); files.hasNext();) 
		{
			String currentFile = (String) files.next();
		
			// Write name of document file to meta-data
			StringBuffer aliasesBuffer = new StringBuffer();	
			CMDBfInternalUtility.addElement(aliasesBuffer, 1, ELEMENT_FILE, false, false);
			aliasesBuffer.append(ISMLConstants.space + ATTRIBUTE_PATH + ICMDBfServicesConstants.EQUAL_SIGN + ICMDBfServicesConstants.DOUBLE_QUOTE + currentFile + 
					ICMDBfServicesConstants.DOUBLE_QUOTE + ICMDBfServicesConstants.CLOSE_ANGLE_BRACKET);
			
			printStream.println(aliasesBuffer.toString()); 			
			List<String> aliases = fileAliasMap.get(currentFile);
						
			// Write aliases to meta-data
			for (int i = 0, aliasCount = aliases.size(); i < aliasCount; i++)
			{
				printStream.println(ISMLConstants.tab + ISMLConstants.tab + SMLValidatorUtil.createElementTag(ISMLConstants.ALIAS_ELEMENT, (String)aliases.get(i)));
			}						
			printStream.println(ISMLConstants.tab+ SMLValidatorUtil.endTagFor(ELEMENT_FILE));
		}
		
		printStream.println(SMLValidatorUtil.endTagFor(ELEMENT_METADATA));
	}

	public boolean isDirty()
	{
		return dirty;
	}
	private void writeFields(PrintStream printStream, String[][] fields)
	{
		for (int i = 0; i < fields.length; i++)
		{
			if (fields[i][1] == null)
				continue;
			
			printStream.println(ISMLConstants.tab + SMLValidatorUtil.createElementTag(fields[i][0], fields[i][1]));
		}
	}


	public void setField(byte field, String value)
	{
		boolean originalDirtyState = dirty;
		dirty = true;
		switch (field)
		{
			case MetadataProcessor.FIELD_NAME:
				identity.setName(value);
				break;
			case MetadataProcessor.FIELD_VERSION:
				identity.setVersion(value);
				break;
			case MetadataProcessor.FIELD_BASE_URI:
				identity.setBaseURI(value);
				break;
			case MetadataProcessor.FIELD_DESCRIPTION:
				identity.setDescription(value);
				break;
			case MetadataProcessor.FIELD_DISPLAY_NAME:
				identity.setDisplayName(value);
				break;			
			default:
				dirty = originalDirtyState;
		}		
	}
}
