/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.core;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.internal.ISMLRepositoryConstants;
import org.eclipse.cosmos.rm.repository.internal.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLFileMetadata;
import org.eclipse.cosmos.rm.repository.resource.ISMLDocument;
import org.xml.sax.SAXException;

/**
 * The meta-data processor loads in the meta-information of the repository.
 * The meta information includes aliases, rule bindings, and SML-IF fields that
 * are lost when importing an SML-IF document.
 * 
 * @author Ali Mehregani
 */
public class MetadataProcessor
{
	/**
	 * The fields
	 */
	public final static byte 	FIELD_NAME = 0x00,				// Name
								FIELD_DISPLAY_NAME = 0x01,		// Display name
								FIELD_VERSION = 0x02,			// Version
								FIELD_DESCRIPTION = 0x03,		// Description
								FIELD_BASE_URI = 0x04;			// Base URI
	
	
	/**
	 * The repository
	 */
	private FileSystemSMLRepository repository;
			
	/**
	 * Meta-data handler
	 */
	private MetadataFileHandler metadataHandler;

	
	/**
	 * Constructor
	 * 
	 * @param contextDirectory The context directory of the repository
	 */	
	public MetadataProcessor(FileSystemSMLRepository repository)
	{
		this.repository = repository;
		this.metadataHandler = new MetadataFileHandler();		
		load();
	}
	
	private void load()
	{
		/* The meta-information is expected to reside in the context root directory */
		File metadataFile = new File (repository.getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, "") + "/" + ISMLRepositoryConstants.METADATA_REPOSITORY);
		if (!metadataFile.exists())
			return;
			
		SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
		
			
		try
		{
			SAXParser parser = saxParserFactory.newSAXParser();
			parser.parse(metadataFile, metadataHandler);
			return;
		} 
		catch (ParserConfigurationException e)
		{
			e.printStackTrace();
		} 
		catch (SAXException e)
		{
			e.printStackTrace();
		} 
		catch (IOException e)
		{
			e.printStackTrace();
		}
		
		metadataHandler = new MetadataFileHandler();
	}

	
	public List<ISMLDocument> retrieveDocuments(Map<String, Boolean> data, boolean isAlias)
	{		
		Map<String, List<String>> aliasMap = metadataHandler.getAliasFileMap();
		Map<String, List<String>> ruleBindings = metadataHandler.getRuleAliasBinding();
		List<String> finalListOfFiles = new ArrayList<String>();
		for (Iterator<String> keys = data.keySet().iterator(); keys.hasNext();)
		{
			String currentKey = (String) keys.next();
			List<String> listOfData = isAlias ? aliasMap.get(currentKey) : ruleBindings.get(currentKey);
			if (!isAlias)
			{
				List<String> listOfFiles = new ArrayList<String>();
				if (listOfData != null)
				{
					for (int i = 0, fileCount = listOfData.size(); i < fileCount; i++)
					{
						List<String> o = (List<String>)aliasMap.get(listOfData.get(i));
						if (o != null)
							listOfFiles.addAll(o);
					}
				}
				listOfData = listOfFiles;
			}
			
			if (listOfData != null)
			{
				finalListOfFiles.addAll(listOfData);
			}
		}
		
		SMLFileMetadata metadata = new SMLFileMetadata();
		List<ISMLDocument> listOfSMLDocuments = new ArrayList<ISMLDocument>();
		for (int i = 0, fileCount = finalListOfFiles.size(); i < fileCount; i++)
		{
			metadata.setId((String)finalListOfFiles.get(i));
			ISMLDocument[] documents = null;
			try
			{
				documents = repository.fetchDocuments(metadata);
			}
			catch (RepositoryOperationException e)
			{
				e.printStackTrace();
			}
			if (documents != null)
			{
				for (int j = 0; j < documents.length; j++)
				{
					if (!listOfSMLDocuments.contains(documents[j]))
						listOfSMLDocuments.add(documents[j]);	
				}
			}
		}
		
		return listOfSMLDocuments;
	}

	/**
	 * Retrieves and returns the aliases of the document with the
	 * id passed in.
	 * 
	 * @param id The id of the document (aka the file name)
	 * @return The aliases of the document
	 */
	public String[] retrieveAliases(String id)
	{
		List<String> aliases = new ArrayList<String>();
		
		addAliases(aliases, id);
		addAliases(aliases, SMLRepositoryUtil.getAlternativeId(repository, id));		
		return aliases.toArray(new String[aliases.size()]);
	}

	
	private void addAliases(List<String> aliases, String id)
	{
		List<String> al = (List<String>)metadataHandler.getFileAliasMap().get(id);
		if (al != null)
			aliases.addAll(al);
	}

	/**
	 * Retrieve the bound rule of the document with the ID passed
	 * in.
	 * 
	 * @param id The id of the document (aka the path to the file name)
	 * @return The bound rules
	 */
	public Map<String, List<String>> retrieveBoundRules(String id)
	{		
		Map<String, List<String>> rules = new Hashtable<String, List<String>>();
		copyMap(retrieveRules(id), rules);
		copyMap(retrieveRules(SMLRepositoryUtil.getAlternativeId(repository, id)), rules);
		return rules;
	}

	private void copyMap(Map<String, List<String>> source, Map<String, List<String>> destination)
	{
		Set<String> keys = source.keySet();
		for (Iterator<String> keyIterator = keys.iterator(); keyIterator.hasNext();)
		{
			String currentKey = keyIterator.next();
			List<String> destValue = destination.get(currentKey);
			List<String> srcValue = source.get(currentKey);
			if (destValue == null)
				destination.put(currentKey, srcValue);
			else
			{
				for (int i = 0, srcValueCount = srcValue.size(); i < srcValueCount; i++)
				{
					if (!destValue.contains(srcValue.get(i)))
						destValue.add(srcValue.get(i));
				}
			}
				
		}
	}

	private Map<String, List<String>> retrieveRules(String id)
	{
		String[] aliases = retrieveAliases(id);
		Map<String, List<String>> rules = new Hashtable<String, List<String>>();
		
		if (aliases != null)
		{
			for (int i = 0; i < aliases.length; i++)
			{
				List<String> ruleAliases = metadataHandler.getAliasRuleBinding().get(aliases[i]);
				if (ruleAliases != null)
					rules.put(aliases[i], ruleAliases);
			}
		}
		
		return rules;
	}
	
	/**
	 * Binds the set of aliases with the documents represented by the
	 * id argument.
	 * 
	 * @param id The id of the document
	 * @param aliases The aliases of the document
	 */
	public void addAliases (String id, String[] aliases)
	{
		metadataHandler.addAlias (id, aliases);
	}
	
	
	/**
	 * Binds a rule with a set of aliases passed in
	 * 
	 * @param rule The rule alias
	 * @param aliases The aliases that the rule should be bound to
	 */
	public void addRuleBinding (String alias, String[] rules)
	{
		metadataHandler.addRuleBinding(alias, rules);		
	}
	
	
	/**
	 * Sets the value of a field
	 * 
	 * @param field The field.  See FIELD_* constants.
	 * @param value The value of the field.
	 */
	public void setField (byte field, String value)
	{
		metadataHandler.setField(field, value);
	}
	
	
	/**
	 * Equivalent to setField(fields[i], values[i]);
	 */
	public void setFields (byte[] fields, String[] values)
	{
		int fieldCount = Math.min(fields.length, values.length);
		for (int i = 0; i < fieldCount; i++)
		{
			metadataHandler.setField(fields[i], values[i]);
		}
	}
	
	/**
	 * Serializes the meta information into XML and writes it to
	 * a file that resides in the context root directory.
	 */
	public void write()
	{
		try
		{
			if (!metadataHandler.isDirty())
				return;
			
			File metadataFile = new File(repository.getProperty(IFileSystemSMLProperties.ROOT_DIRECTORY, "") + "/" + ISMLRepositoryConstants.METADATA_REPOSITORY);
			if (!metadataFile.exists())				
				metadataFile.createNewFile();
	
			PrintStream metadataPrintStream = new PrintStream(metadataFile);
			metadataHandler.write(metadataPrintStream);
			metadataPrintStream.close();
		} 
		catch (IOException e)
		{			
			e.printStackTrace();
		}
	}

	/**
	 * @return the metadataHandler
	 */
	public MetadataFileHandler getMetadataHandler()
	{
		return metadataHandler;
	}
		
	/**
	 * Returns the file alias mapping
	 * 
	 * @return The file alias mapping
	 */
	public Map<String, List<String>> getFileAliasMapping()
	{
		return metadataHandler.getFileAliasMap();
	}
		
	/**
	 * Returns the alias rule mapping
	 * 
	 * @return The alias rule mapping
	 */
	public Map<String, List<String>> getAliasRuleMapping()
	{
		return metadataHandler.getAliasRuleBinding();
	}
}
