/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.operations;

import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.operations.ISMLOperation;

/**
 * An abstract class used to implement common functionality between
 * repository operations. 
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractSMLOperation extends AbstractListenerManager implements ISMLOperation
{
	/**
	 * The repository associated with this operation
	 */
	private ISMLRepository repository;
	
	/**
	 * The arguments of this operation
	 */
	private Object[] arguments;

	
	/**
	 * Constructor
	 * 
	 * @param expectedArgType The expected argument type
	 */
	public AbstractSMLOperation(ISMLRepository repository)
	{
		this.repository = repository;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.operations.ISMLOperation#setArguments(java.lang.Object[])
	 */
	public void setArguments(Object[] arguments) throws IllegalArgumentException
	{
		this.arguments = arguments;  
	}
	

	/**
	 * @see org.eclipse.cosmos.rm.repository.operations.ISMLOperation#addArgument(java.lang.Object)
	 */
	public void addArgument(Object argument) throws IllegalArgumentException
	{
		Object[] newArguments = new Object[arguments == null ? 1 : arguments.length + 1];
		
		if (arguments != null)
		{
			System.arraycopy(arguments, 0, newArguments, 0, arguments.length);
		}
		
		newArguments[newArguments.length - 1] = argument;
		this.arguments = newArguments;
	}

	public Object[] getArguments()
	{
		return arguments;
	}
	
	/**
	 * @return the repository
	 */
	protected ISMLRepository getRepository()
	{
		return repository;
	}


	/**
	 * @param repository the repository to set
	 */
	protected void setRepository(ISMLRepository repository)
	{
		this.repository = repository;
	}	
	
	/**
	 * Checks that the expected class is equivalent to actual
	 * 
	 * @param expected The expected class name
	 * @param actual The actual class name
	 * @return true if the expected class is of type actual; false otherwise
	 */
	protected boolean checkType(Class<?> expected, Class<?> actual)
	{		
		if (expected == actual)
			return true;
		
		/* Check the interfaces */
		Class<?>[] interfaces = actual.getInterfaces();
		for (int i = 0; i < interfaces.length; i++)
		{
			if (interfaces[i] == expected)
				return true;
		}
		
		/* Check the super class */
		Class<?> superClass = actual.getSuperclass();
		if (superClass == null)
			return false;
		return checkType(expected, actual.getSuperclass());		
	}
	
	
	/**
	 * Checks that each argument's class is of type expected
	 * 
	 * @param arguments The arguments
	 * @param expected The expected class
	 * @return True if all argument's classes are of type expected;
	 * false otherwise
	 */
	protected boolean checkType(Object[] arguments, Class<?> expected)
	{
		for (int i = 0; i < arguments.length; i++)
		{
			if (!checkType(expected, arguments[i].getClass()))
				return false;
		}
		
		return true;
	}
	
	
	/**
	 * Checks to ensure that argument[i]'s class is equivalent to 
	 * expectedClasses[i]. 
	 * 
	 * @return true if arguments[i]'s class is equivalent to expectedClasses[i];
	 * false otherwise.
	 */
	protected boolean checkType (Object[] arguments, Class<?>[] expectedClasses)
	{
		if (arguments.length < expectedClasses.length)
			return false;
		
		for (int i = 0; i < expectedClasses.length; i++)
		{
			boolean status = checkType(expectedClasses[i], arguments[i].getClass());
			if (!status)
				return false;
		}
		
		return true;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.operations.ISMLOperation#getOutput()
	 */
	public Object getOutput()
	{
		// Return null by default (i.e. operations will not produce an output
		// by default)
		return null;
	}
}
