/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.operations;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.rmi.server.ExportException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.internal.ISMLRepositoryConstants;
import org.eclipse.cosmos.rm.repository.internal.RepositoryMessages;
import org.eclipse.cosmos.rm.repository.internal.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLIFIdentity;
import org.eclipse.cosmos.rm.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.repository.resource.ISMLIFIdentity;
import org.eclipse.cosmos.rm.repository.resource.ISMLMetadata;
import org.eclipse.cosmos.rm.validation.internal.artifacts.RuleBinding;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.osgi.util.NLS;

/**
 * Export a set of SML model documents to an SML-IF file
 * 
 * @author David Whiteman
 * @modified Ali Mehregani
 */
public class ExportOperation
{
	private StringBuffer smlifFileBuffer = new StringBuffer();
	private String rootFileName = ""; //$NON-NLS-1$z
	private StringBuffer definitions = new StringBuffer();
	private StringBuffer instances = new StringBuffer();
	private StringBuffer ruleBindingsBuffer = new StringBuffer();	
	private ISMLRepository repository;
	
	
	public ExportOperation (ISMLRepository repository)
	{
		this.repository = repository;
	}
	
	protected void createDirectoryStructureFor(File smlifFile) throws RepositoryOperationException
	{
		boolean mkdirsResult = false;
		if (!smlifFile.getParentFile().exists()) {
			mkdirsResult = smlifFile.getParentFile().mkdirs();
			if (!mkdirsResult) {
				throw new RepositoryOperationException(NLS.bind(RepositoryMessages.exportErrorCouldNotCreateDirectoryStructure, smlifFile.getName()));
			}
		}
	}

	/**
	 * Answer whether a file exists and is readable
	 * 
	 * @param file
	 * @return whether the file exists and is readable
	 */
	protected boolean fileExistsAndIsReadable(File file) throws ExportException {
		String name = file.getName();

		if (!file.exists()) {
			throw new ExportException(NLS.bind(RepositoryMessages.exportErrorFileNotFound, name));
		}
		if (!file.canRead()) {
			throw new ExportException(NLS.bind(RepositoryMessages.exportErrorFileNotReadable, name));
		}
		
		return true;
	}

	/**
	 * Add indicated file to SML-IF document buffers, depending on type of file
	 * @param aliases 
	 * 
	 * @param file
	 */
	protected void addFileToDefinitionsAndInstances(Map<String, String[]> aliases, ISMLDocument document, boolean useMetadata) 
	{
		ISMLMetadata metadata = document.getMetadata();
		String name = metadata.getId().trim();
		if (name == null) 
		{
			return;
		}

		if (metadata.getDocumentType() == ISMLMetadata.DOCUMENT_TYPE_DEFINITION) 
		{
			if (definitions.length() == 0) 
			{
				definitions.append(ISMLRepositoryConstants.definitionsStart);
			}
			addFileToSection(document, definitions, aliases, useMetadata);
		} 
		else if (metadata.getDocumentType() == ISMLMetadata.DOCUMENT_TYPE_INSTANCE) 
		{
			if (instances.length() == 0) {
				instances.append(ISMLRepositoryConstants.instancesStart);
			}
			addFileToSection(document, instances, aliases, useMetadata);
		}

	}
	
	/**
	 * Add the file to the indicated section of the document.  The document
	 * sections are maintained as StringBuffer instances.
	 * 
	 * @param file the file being added to the SML-IF document
	 * @param sectionBuffer the string buffer for either the definitions or instances section
	 * @param aliases 
	 */
	protected void addFileToSection(ISMLDocument file, StringBuffer sectionBuffer, Map<String, String[]> aliases, boolean useMetadata) {
		sectionBuffer.append(getFileAlias(file, aliases, useMetadata));
		sectionBuffer.append(getFileContent(file));
		sectionBuffer.append(ISMLRepositoryConstants.docDataEnd);
	}

	/**
	 * Construct alias entity for the indicated file
	 * 
	 * @param file file being added to SML-IF document
	 * @param aliases 
	 * @return StringBuffer containing alias markup
	 */
	protected StringBuffer getFileAlias(ISMLDocument file, Map<String, String[]> aliases, boolean useMetadata) 
	{	
		StringBuffer buffer = new StringBuffer();	
		buffer.append(ISMLRepositoryConstants.docAliasesStart);
				
		String id = file.getMetadata().getId();
		String[] documentAliases = aliases.get(id);
		documentAliases = documentAliases == null ? aliases.get(SMLRepositoryUtil.getAlternativeId(repository, id)) : documentAliases;
				
		String[] finalAliases = null;
		int finalAliasSize = 0;
		
		String[] metaDataAliases = useMetadata ? file.getMetadata().getAliases() : null;
		finalAliasSize = metaDataAliases == null ? 0 : metaDataAliases.length;
		finalAliasSize += documentAliases == null ? 0 : documentAliases.length;
		
		finalAliases = new String[finalAliasSize];		
		if (metaDataAliases != null)
		{
			System.arraycopy(metaDataAliases, 0, finalAliases, 0, metaDataAliases.length);
		}
		if (documentAliases != null)
		{
			System.arraycopy(documentAliases, 0, finalAliases, metaDataAliases == null ? 0 : metaDataAliases.length, documentAliases.length);
		}		
		
		if (finalAliases.length <= 0) 
		{				
			appendAliasToBuffer(SMLRepositoryUtil.getDocumentName(file), buffer);
		} 
		else 
		{			
			for (int i = 0; i < finalAliases.length; i++)
			{				
				appendAliasToBuffer(finalAliases[i], buffer);
			}			
		}

		buffer.append(ISMLRepositoryConstants.docAliasesEnd);
		buffer.append(ISMLConstants.nl);
		return buffer;
	}

	protected void appendAliasToBuffer(String alias, StringBuffer buffer) 
	{
		buffer.append(ISMLRepositoryConstants.docAliasStart).append(alias)
		.append(ISMLRepositoryConstants.docAliasEnd);
	}

	/**
	 * Read definition or instance file and return contents as a string buffer
	 * 
	 * @param modelUnitFile
	 * @return
	 */
	protected StringBuffer getFileContent(ISMLDocument modelUnitFile) {
		StringBuffer result = new StringBuffer();
		//result.append(SMLConstants.nl).append(SMLConstants.tab);
		
		LineNumberReader reader = null;
		try {
			File file = new File(modelUnitFile.getMetadata().getId());
			reader = new LineNumberReader(new FileReader(file));

			String line = reader.readLine();
			while (line != null) {
				int idx = line.indexOf("?>"); //$NON-NLS-1$
				if (idx != -1) {
					result.append(line.substring(idx + 2));
				} else {
					result.append(line).append(ISMLConstants.nl);
				}

				line = reader.readLine();
			}
		} catch (IOException exc) {
			exc.printStackTrace();
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
					/* Not handled */
				}
			}
		}

		//result.append(SMLConstants.tab).append(SMLConstants.nl);

		return result;
	}

	/**
	 * Build an SML-IF file from a collection of individual files representing
	 * definition and instance model units.
	 * 
	 * @param modelUnits The model units representing the instances and definitions
	 * @param aliases The aliases of the documents
	 * @param ruleBindings The rule bindings
	 * @param identity The SML-IF identity
	 * @param useMetadata Indicates whether the meta-data of the documents should be considered
	 * @param replaceTarget Indicates whether the operation is replace or append of the SML-IF file
	 * @throws RepositoryOperationException In case of an unexpected error
	 */
	public void exportFromFiles(ISMLDocument[] modelUnits, Map<String, String[]> aliases, RuleBinding[] ruleBindings, ISMLIFIdentity identity, boolean useMetadata, boolean replaceTarget) throws RepositoryOperationException 
	{
		File smlifFile = new File(identity.getId());
		
		if (!replaceTarget && !smlifFile.exists()) {
			// If this is an append, the file must exist
			throw new RepositoryOperationException(NLS.bind("File {0} to be appended to does not exist", identity.getId()), null);
		}

		rootFileName = smlifFile.getName();

		String modelName = SMLRepositoryUtil.isNullOrEmpty(identity.getName()) ? rootFileName : identity.getName();
		((SMLIFIdentity)identity).setName(modelName);

		SMLRepositoryUtil.appendSMLIFIdentityInformation(smlifFileBuffer, identity);
		buildRuleBindings(modelUnits, ruleBindings, useMetadata);

		try
		{
			for (int i = 0; i < modelUnits.length; i++)
			{			
				String filePath = modelUnits[i].getMetadata().getId();
				File file = new File(filePath);				
				if (!fileExistsAndIsReadable(file)) {
					continue; // skip over bogus files???
					//return false;
				}
								
				if (file.isDirectory()) 
				{
					throw new RepositoryOperationException(NLS.bind(RepositoryMessages.exportErrorNameShouldNotBeDirectory, filePath), null);
				}
				addFileToDefinitionsAndInstances(aliases, modelUnits[i], useMetadata);			
			}
		}
		catch (ExportException e)
		{
			throw new RepositoryOperationException(e.getLocalizedMessage(), e);
		}

		appendDefinitionsAndInstances();		
		if (smlifFile.exists()) {
			smlifFile.delete();
		}
		createDirectoryStructureFor(smlifFile);

		try 
		{
			// Need to explicitly indicate UTF-8 so the output of export matches the input from an import
			PrintWriter writer = new PrintWriter(new OutputStreamWriter(new FileOutputStream(smlifFile), ICMDBfServicesConstants.UTF_8));
			writer.print(smlifFileBuffer.toString());
			writer.close();

		} 
		catch (IOException exc) 
		{
			throw new RepositoryOperationException(exc.getLocalizedMessage(), exc);
		}		
	}

	protected void buildRuleBindings(ISMLDocument[] modelUnits, RuleBinding[] ruleBindings, boolean useMetadata) 
	{
		if (ruleBindings == null) 
			return;

		for (int i = 0; i < ruleBindings.length; i++)
		{					
			writeRuleBinding(ruleBindings[i].getRule(), ruleBindings[i].getAlias());			
		}
		
		if (useMetadata)
		{
			for (int i = 0; i < modelUnits.length; i++)
			{
				Map<String, List<String>> bindings = modelUnits[i].getMetadata().getBoundRules();
				if (bindings == null)
					continue;
				
				for (Iterator<String> aliases = bindings.keySet().iterator(); aliases.hasNext();)
				{
					String alias = aliases.next();
					List<String> rules = bindings.get(alias);
					
					for (int j = 0, ruleCount = rules.size(); j < ruleCount; j++)
					{
						writeRuleBinding((String)rules.get(j), alias);		
					}						
				}					
			}
		}

		if (ruleBindingsBuffer.length() > 0) 
		{
			// Don't add begin and end tags unless there really are rule bindings
			ruleBindingsBuffer.insert(0, ISMLRepositoryConstants.ruleBindingsStart);
			ruleBindingsBuffer.append(ISMLRepositoryConstants.ruleBindingsEnd);
		}
	}


	private void writeRuleBinding(String ruleAlias, String docAlias)
	{
		ruleBindingsBuffer.append(ISMLConstants.tab + ISMLConstants.tab + SMLValidatorUtil.beginTagFor(ISMLRepositoryConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.RULEBINDING_ELEMENT) + ISMLConstants.nl);
		if (docAlias != null) 
		{
			// write the doc alias tag
			ruleBindingsBuffer.append(ISMLConstants.tab
					+ ISMLConstants.tab + ISMLConstants.tab 
					+ SMLValidatorUtil.createElementTag(
							ISMLRepositoryConstants.SMLIF_NAMESPACE_PREFIX,
							ISMLConstants.DOCUMENTALIAS_ELEMENT,
							docAlias) + ISMLConstants.nl);
		}	
		
		// write the rule alias tag
		ruleBindingsBuffer.append(ISMLConstants.tab + ISMLConstants.tab + ISMLConstants.tab + SMLValidatorUtil.createElementTag(ISMLRepositoryConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.RULEALIAS_ELEMENT, ruleAlias) + ISMLConstants.nl);
		// write the rule binding end tag
		ruleBindingsBuffer.append(ISMLConstants.tab + ISMLConstants.tab + SMLValidatorUtil.endTagFor(ISMLRepositoryConstants.SMLIF_NAMESPACE_PREFIX, ISMLConstants.RULEBINDING_ELEMENT) + ISMLConstants.nl);
	}

	/**
	 * Copy definition and instance buffers to SML-IF buffer before
	 * writing file to system.
	 */
	protected void appendDefinitionsAndInstances() {
		if (hasRuleBindings()) {
			smlifFileBuffer.append(ISMLConstants.nl).append(ruleBindingsBuffer);
		}
		
		if (hasDefinitions()) {
			definitions.append(ISMLRepositoryConstants.definitionsEnd);
			smlifFileBuffer.append(ISMLConstants.nl).append(definitions);
		}
		if (hasInstances()) {
			instances.append(ISMLRepositoryConstants.instancesEnd);
			smlifFileBuffer.append(ISMLConstants.nl).append(instances);
		}
		smlifFileBuffer.append(ISMLRepositoryConstants.smlifEnd);
	}

	protected boolean hasRuleBindings() {
		return ruleBindingsBuffer.length() > 0;
	}
	
	protected boolean hasDefinitions() {
		return definitions.length() > 0;
	}

	protected boolean hasInstances() {
		return instances.length() > 0;
	}

}
