/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.operations;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.Iterator;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.internal.RepositoryMessages;
import org.eclipse.cosmos.rm.repository.internal.SMLRepositoryUtil;
import org.eclipse.cosmos.rm.repository.internal.core.FileSystemSMLRepository;
import org.eclipse.cosmos.rm.repository.internal.core.MetadataProcessor;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLFileMetadata;
import org.eclipse.cosmos.rm.validation.internal.artifacts.RuleBinding;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;
import org.xml.sax.SAXException;

/**
 * Import model structure from an SML-IF file.
 * 
 * @author David Whiteman
 * @author Ali Mehregani
 */
public class ImportOperation 
{	
	/**
	 * The repository
	 */
	private FileSystemSMLRepository repository;

	
	/**
	 * Constructor
	 * 
	 * @param repository The repository
	 */
	public ImportOperation(FileSystemSMLRepository repository)
	{
		this.repository = repository;
	}

	
	
	/**
	 * Import a SML-IF file into the work space, creating files and model structure for definition and instance documents,
	 * and also capturing extra meta-data in a separate file (hidden from user, but preserved for future exports).
	 * 
	 * @param resource Name of an SML-IF file to be imported.  Expected to be absolute path on file system.
	 * @param destination Name of output directory.  Expected to be relative to the workspace.
	 * @throws ImportException
	 */
	public void importFromFileSystemFile(String resource, String destination) throws RepositoryOperationException 
	{
		if (SMLRepositoryUtil.isNullOrEmpty(resource)) 
		{
			throw new RepositoryOperationException(RepositoryMessages.importErrorNullSmlifFileName);
		}
		if (destination == null) 
		{
			throw new RepositoryOperationException(RepositoryMessages.importErrorNullTargetDirName);
		}
		
		
		SMLIFFileHandler handler = new SMLIFFileHandler();
		try {
			SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
			saxParserFactory.setNamespaceAware(true);

			saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACES, true);
			saxParserFactory.setFeature(ISMLConstants.SAX_PARSER_FEATURE_NAMESPACE_PREFIXES, true);
	
			SAXParser parser = saxParserFactory.newSAXParser();
			parser.parse(resource, handler);
			
			// Set the fields of the meta-data processor
			MetadataProcessor processor = repository.getMetadataProcessor();
			processor.setFields(
					new byte[]{
						MetadataProcessor.FIELD_NAME,
						MetadataProcessor.FIELD_DISPLAY_NAME,
						MetadataProcessor.FIELD_BASE_URI,
						MetadataProcessor.FIELD_DESCRIPTION,
						MetadataProcessor.FIELD_VERSION},
					new String[]{
						handler.getName(),
						handler.getDisplayName(),
						handler.getBaseURI(),
						handler.getDescription(),
						handler.getVersion()});
			
			// For each rule binding
			for (Iterator<RuleBinding> ruleBindingsIterator = handler.getRuleBindingMappings().iterator(); ruleBindingsIterator.hasNext();) 
			{				
				RuleBinding ruleBinding = ruleBindingsIterator.next();
				processor.addRuleBinding(ruleBinding.getAlias(), new String[]{ruleBinding.getRule()});
			}
			
			// For each model unit
			for (Iterator<ModelUnit> i = handler.getModelUnits().iterator(); i.hasNext();) 
			{
				ModelUnit modelUnit = i.next();
				String targetFile = destination + "/" + modelUnit.getFileName();
								
				// Write the model unit to the repository			
				repository.addDocument(	
						new SMLFileMetadata(targetFile, 1, null, 
							modelUnit.getAliases().toArray(new String[modelUnit.getAliases().size()]), 
							null), 									
						new ByteArrayInputStream(modelUnit.getFileContents().toString().getBytes()));
			}

		} 
		catch (ParserConfigurationException e) 
		{
			throw new RepositoryOperationException(e.getLocalizedMessage(), e);
		} 
		catch (SAXException e) 
		{
			throw new RepositoryOperationException(e.getLocalizedMessage(), e);
		} 
		catch (IOException e) 
		{
			throw new RepositoryOperationException(e.getLocalizedMessage(), e);
		}
	}
}
