/**********************************************************************
 * Copyright (c) 2007 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.operations;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;

import org.eclipse.cosmos.rm.repository.internal.ISMLRepositoryConstants;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;

/**
 * This represents an individual definition or instance document
 * 
 * @author David Whiteman
 */
public class ModelUnit {
	private static final String URL_PREFIX = "http://";
	private static final String XML_SUFFIX = ".xml";
	private static final String XSD_SUFFIX = ".xsd";
	private static final String INSTANCE_PREFIX = "instance";
	private static final String DEFINITION_PREFIX = "definition";
	
	private String fileName = null;
	private StringBuffer contents = new StringBuffer();
	private ArrayList<String> aliases = new ArrayList<String>();
	private int type = -1;
	private String filePrefix;

	public ModelUnit(int type) {
		this.type = type;
	}

	public void start() {
		contents.append(ISMLRepositoryConstants.XML_DECLARATION).append(ISMLConstants.nl);
	}

	public void stop() {
		// Currently a no-op.  Can be used for any cleanup necessary or debug info
	}

	public StringBuffer getFileContents() {
		return contents;
	}

	public String getFileName() {
		//String fileName = this.fileName;
		if ((fileName == null) || ("".equals(fileName))) {
			if (aliases.size() > 0) {
				fileName = convertAliasToValidFileName((String) aliases.get(0));
			} else {
				File tempFile = null;
				try {
					// Generate random name for file if no alias provided
					tempFile = File.createTempFile(getFilePrefix(), getFileSuffix());
					tempFile.delete();
				} catch (IOException e) {
				}
				fileName = tempFile.getName();
			}
		}
		return fileName;
	}
	
	private String convertAliasToValidFileName(String string) {
		if (!string.startsWith(URL_PREFIX)) {
			return string;
		}
		return string.substring(URL_PREFIX.length());
	}

	/*
	 * If a prefix has been supplied, use it, else
	 * use a standard generic prefix.
	 */
	private String getFilePrefix() {
		if (filePrefix != null) {
			return filePrefix;
		}
		return isDefinition() ? DEFINITION_PREFIX : INSTANCE_PREFIX;
	}

	private String getFileSuffix() {
		return isDefinition() ? XSD_SUFFIX : XML_SUFFIX;
	}

	protected boolean isDefinition() {
		return (type == ISMLConstants.TYPE_DEFINITION);
	}

	public void setFileName(String fileName) {
		this.fileName = fileName;
	}

	public void appendToContents(String string) {
		contents.append(string);
	}

	public void addAlias(String string) {
		aliases.add(string);
	}

	public ArrayList<String> getAliases() {
		return aliases;
	}

	/**
	 * Set a prefix to be used for generated file names.
	 * For instance documents, this is usually the root element
	 * of the model unit.  For schema definitions, this is typically
	 * the rightmost token of the targetNamespace.
	 * 
	 * @param localName
	 */
	public void setFilePrefix(String localName) {
		filePrefix = localName;
	}
}
