/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.operations.cmdbf;

import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.cmdbf.services.internal.CMDBfInternalUtility;
import org.eclipse.cosmos.dc.cmdbf.services.internal.CMDBfMessages;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.impl.AbstractItemConstraintHandler;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IConstraint;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.INodes;
import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.internal.RepositoryMessages;

/**
 * An abstract extension of {@link AbstractItemConstraintHandler} that provides
 * some common methods for the item constraint handlers of the SML repository.
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractSMLConstraintHandler extends AbstractItemConstraintHandler
{	
	@Override
	public void initialize(Map<String, Object> init) throws CMDBfServiceException
	{
		// Make sure the SML repository is included in the initialization data 
		if (!(init.get(CMDBfConstants.DATA_PROVIDER) instanceof ISMLRepository))
		{
			throw new CMDBfServiceException(
					CMDBfServiceException.RECIEVER,
					CMDBfServiceException.QUERY_ERROR,
					CMDBfMessages.faultsQueryError,
					CMDBfInternalUtility.createTextNode(RepositoryMessages.errorBadInitData));
		}
		super.initialize(init);
	}
	
	
	/**
	 * Returns the repository associated with this SML item
	 * constraint handler
	 * 
	 * @return The associated SML repository
	 */
	protected ISMLRepository getRepository()
	{
		return (ISMLRepository)getValue(CMDBfConstants.DATA_PROVIDER);
	}
	
	
	/**
	 * Retrieves the context.  If the argument passed in happens to be a starting
	 * context, then all the documents of the repository are retrieved and passed
	 * in as context.
	 * 
	 * @param context The context
	 * @return All documents of the repository if context.iStartingContext() returns
	 * true; context if it returns false
	 * @throws RepositoryOperationException In case of an error
	 */
	protected INodes retrieveContext(INodes context) throws RepositoryOperationException
	{
		return context.isStartingContext() ? CMDBfUtil.retrieveRepositoryDocs(getRepository(), context.getId()) :
				context;
	}
	
	
	/**
	 * Should be overwritten if subclass is an item constraint handler
	 * 
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.service.impl.AbstractItemConstraintHandler#handle(org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.INodes, org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IConstraint)
	 */
	protected INodes handle(INodes context, IConstraint constraint) throws CMDBfServiceException
	{
		return null;
	}
}
