/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.operations.cmdbf;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.internal.CMDBfInternalUtility;
import org.eclipse.cosmos.dc.cmdbf.services.internal.CMDBfMessages;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IConstraint;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IOperator;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IProperty;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IPropertyValue;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.INodes;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * The implementation of the property value constraint used under
 * an itemTemplate
 * 
 * @author Ali Mehregani
 */
public class ItemPropertyHandler extends AbstractSMLConstraintHandler
{
	/** 
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.service.impl.AbstractItemConstraintHandler#handle(org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.INodes, org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IConstraint)
	 */
	public INodes handle(INodes context, IConstraint constraint) throws CMDBfServiceException
	{				
		try
		{
			INodes result = retrieveContext(context);
			IPropertyValue propertyConstraint = (IPropertyValue)constraint;
			IGraphElement[] elements = result.getElements();
			IOperator[] operators = propertyConstraint.getOperators();
			if (operators == null)
			{
				return context;
			}
			
			ISMLRepository repository = getRepository();
			List<IGraphElement> revisedList = new ArrayList<IGraphElement>();
			boolean matchAny = propertyConstraint.isMatchAny();
			
			// For each entry in the data set
			for (int i = 0; i < elements.length; i++)
			{						
				Node rootNode = CMDBfUtil.retrieveDocumentNode(repository, (IItem)elements[i]);
				if (rootNode == null)
				{
					continue;
				}
							
				boolean overallResult = !matchAny; // Initial value if false if composition is OR and true if it's AND
				
				List<Object> matches = new ArrayList<Object>();
				retrieveProperty (rootNode, propertyConstraint.getNamespace(), propertyConstraint.getLocalName(), matches, true); 
				if (matches.isEmpty())
				{
					matches.add(new NullSMLProperty());
				}					
				
				// For each operator
				for (int j = 0; j < operators.length; j++)
				{				
					boolean operationResult = false;
					Object value = operators[j].getValue();
					boolean caseSensitive = operators[j].isCaseSensitive();
					
					for (int k = 0, propertyCount = matches.size(); !operationResult && k < propertyCount; k++)
					{
						IProperty property = (IProperty)matches.get(k);
						switch (operators[j].getType())
						{
							case IOperator.CONTAINS:
								if (!(value instanceof String))
								{
									break;
								}
								operationResult = property.contains((String)value, caseSensitive);
								break;
							
							case IOperator.EQUALS:
								operationResult = property.equals(value, caseSensitive);
								break;
							
							case IOperator.GREATER_THAN:
								operationResult = property.greater(value);
								break;
							
							case IOperator.GREATER_THAN_OR_EQUAL:
								operationResult = property.greaterOrEqual(value);
								break;
								
							case IOperator.LESS_THAN:
								operationResult = property.less(value);
								break;
								
							case IOperator.LESS_THAN_OR_EQUAL:
								operationResult = property.lessOrEqual(value);
								break;
								
							case IOperator.LIKE:
								if (!(value instanceof String))
								{
									break;
								}
								operationResult = property.like((String)value, caseSensitive);
								break;
								
							case IOperator.IS_NULL:
								operationResult = property.isNull();
								break;						
						}
						
						// Negate the result if necessary
						if (!(property instanceof NullSMLProperty) && operators[j].isNegated())
						{
							operationResult = !operationResult;
						}				
					}
					
					
					// OR case
					if (matchAny)
					{
						// Break as soon as one operation passes
						if (operationResult)
						{	
							overallResult = true;
							break;
						}						
					}
					
					// AND case
					else
					{
						// Break as soon as operation fails
						if (!operationResult)
						{
							overallResult = false;
							break;
						}					
					}
				}
				
				// Remove the item if the property constraint constraint has failed
				if (overallResult)
				{
					revisedList.add(elements[i]);
				}						
			}
			
			result.setElements(revisedList.toArray(new IGraphElement[revisedList.size()]));
			return result;
		} 
		catch (RepositoryOperationException e)
		{
			throw new CMDBfServiceException(
					CMDBfServiceException.RECIEVER,
					CMDBfServiceException.QUERY_ERROR,
					CMDBfMessages.faultsQueryError,
					CMDBfInternalUtility.createTextNode(e.getLocalizedMessage()),
					e);
		}			
	}

	
	/**
	 * Recursively looks for the property with the namespace and localname
	 * passed in.
	 * 
	 * @param node The current node
	 * @param namespace The name space
	 * @param localName The local name
	 * @param matches The list of matches
	 * @param storageType Used to indicate the item types that are added to the matches list
	 * @return The property node
	 */
	public static void retrieveProperty(Node node, URI namespace, String localName, List<Object> matches, boolean storageType)
	{					
		String namespaceStr = CMDBfServicesUtil.toString(namespace);
		if (namespaceStr.equals(node.getNamespaceURI()) && localName.equals(node.getLocalName()))
		{
			matches.add(storageType ? SMLPropertyFactory.create(node) : node);
			if (!storageType)
			{
				return;
			}
		}
		
		// Check the attributes if current node is an element
		boolean namespaceMatch = namespaceStr.equals(node.getNamespaceURI());
		if (node.getNodeType() == Node.ELEMENT_NODE)
		{
			NamedNodeMap attributes = node.getAttributes();
			for (int i = 0, attCount = attributes.getLength(); i < attCount; i++)
			{								
				if ((namespaceMatch || namespaceStr.equals(attributes.item(i).getNamespaceURI())) && localName.equals(attributes.item(i).getLocalName()))
				{
					if (storageType)
					{
						matches.add (SMLPropertyFactory.create(attributes.item(i)));
					}
					else
					{
						matches.add (node);
						return;
					}
				}				
			}
		}
		
		NodeList children = node.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			retrieveProperty (children.item(i), namespace, localName, matches, storageType);			
		}
	}

}
