/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.operations.cmdbf;

import java.io.InputStream;
import java.util.Hashtable;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.impl.CMDBfQueryOperation;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.QueryOutputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IQueryResult;
import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.internal.operations.AbstractSMLOperation;
import org.eclipse.cosmos.rm.repository.operations.cmdbf.ICMDBfQueryOperation;

/**
 * Represents the CMDBf query operation.  This operation accepts the following
 * arguments in the order listed below:
 * <ol>
 * 	<li>An {@link InputStream} containing the XML query string (required) </li>
 *  <li>A {@link Map} containing options for the query operation (optional) </li>
 * </ol>
 * The output of this operation is a query response that conforms to CMDBf's 
 * response dialect.  See http://cosmos.org for more details. 
 * 
 * @author Ali Mehregani
 */
public class SMLQueryOperation extends AbstractSMLOperation implements ICMDBfQueryOperation
{
	/**
	 * The CMDBf query operation
	 */
	private CMDBfQueryOperation queryOperation;
	
	/**
	 * The output of this operation
	 */
	private InputStream output;
		
	
	/**
	 * Constructor
	 * 
	 * @param repository The repository
	 */
	public SMLQueryOperation(ISMLRepository repository)
	{
		super(repository);
		
		// Initialize the default CMDBf query operation provided
		queryOperation = new CMDBfQueryOperation(SMLConstraintHandlerFactory.getInstance(), null);
		Map<String, Object> init = new Hashtable<String, Object>();
		init.put(CMDBfConstants.DATA_PROVIDER, repository);
		queryOperation.initialize(init);
	}

	
	/** 
	 * @see org.eclipse.cosmos.rm.repository.operations.ISMLOperation#getId()
	 */
	public String getId()
	{		
		return ICMDBfQueryOperation.ID;
	}


	/**
	 * @see org.eclipse.cosmos.rm.repository.operations.ISMLOperation#run()
	 */
	@SuppressWarnings("unchecked")
	public void run() throws RepositoryOperationException
	{
		// Read in the argument 
		Object[] arguments = getArguments();
		
		if (arguments == null || arguments.length < 1 || !(arguments[0] instanceof InputStream))
		{
			throw new RepositoryOperationException("The CMDBf query operation accepts exactly one argument -\nAn InputStream representing the query input string");			
		}		
		
		try
		{
			if (arguments.length >= 2 && arguments[1] instanceof Map)
			{
				queryOperation.setRuntimeOptions((Map<String,Object>)arguments[1]);
			}
			
			IQueryResult result = queryOperation.execute((InputStream)arguments[0]);
			this.output = QueryOutputTransformer.transform(result);
		} 
		catch (CMDBfServiceException e)
		{
			throw new RepositoryOperationException(e);
		}		
		
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.internal.operations.AbstractSMLOperation#getOutput()
	 */
	public Object getOutput()
	{
		return output;
	}
}
