/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.reference;

import java.net.URI;
import java.net.URISyntaxException;

import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLFileDocument;
import org.eclipse.cosmos.rm.repository.internal.resource.SMLFileMetadata;
import org.eclipse.cosmos.rm.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.repository.resource.ISMLMetadata;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil.RemoteRetrievalException;
import org.eclipse.cosmos.rm.validation.internal.reference.URIReference;
import org.w3c.dom.Node;

/**
 * An SML document reference.  This class can be used to retrieve the
 * root DOM node of the referenced SML document.  
 * 
 * @author Ali Mehregani
 */
public class SMLDocumentReference extends URIReference
{

	private ISMLRepository repository;

	/**
	 * The constructor
	 * 
	 * @param repositor The repository
	 * @param reference The referenced document
	 * @throws URISyntaxException If the reference is incorrect
	 */
	public SMLDocumentReference(ISMLRepository repositor, String reference) throws URISyntaxException
	{		
		super(reference);
		this.repository = repositor;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.validation.internal.reference.URIReference#retrieveDocumentDOM()
	 */
	public Node retrieveDocumentDOM() throws URISyntaxException
	{
		String document = getDocumentReference();
		
		/* Return what's stored in the repository */
		String scheme = getReference().getScheme();
		if (scheme == null)
		{
			try
			{
				ISMLDocument[] documents = repository.fetchDocuments(new SMLFileMetadata(null, ISMLMetadata.DOCUMENT_TYPE_NONE, null, new String[]{document}, null));
				if (documents == null || documents.length <= 0)
					documents = repository.fetchDocuments(new SMLFileMetadata(document, ISMLMetadata.DOCUMENT_TYPE_NONE, null, null, null));
				
				return documents.length == 1 && documents[0] instanceof SMLFileDocument ? ((SMLFileDocument)documents[0]).getDOMDocument() : null;
			} 
			catch (RepositoryOperationException e)
			{				
				e.printStackTrace();
				return null;
			}			
		}
		
		/* Otherwise we'll need to download the reference document */ 
		try
		{
			return SMLValidatorUtil.retrieveRemoteDocument(getReference().toString());
		} 
		catch (RemoteRetrievalException e)
		{
			return null;
		}	
	}
	
	/**
	 * @return the base
	 */
	public URI getBase()
	{
		return super.getBase();
	}


	/**
	 * @param base the base to set
	 */
	public void setBase(URI base)
	{
		super.setBase(base);
	}	
}
