/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.repository.internal.resource;

import java.io.File;
import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;

import org.eclipse.cosmos.dc.cmdbf.services.internal.CMDBfInternalUtility;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IItemConvertible;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.INodes;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.QueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IInstanceId;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.rm.repository.core.ISMLRepository;
import org.eclipse.cosmos.rm.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.repository.internal.operations.cmdbf.CMDBfConstants;
import org.eclipse.cosmos.rm.repository.resource.ISMLDefinitionDocument;
import org.eclipse.cosmos.rm.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.repository.resource.ISMLInstanceDocument;
import org.eclipse.cosmos.rm.repository.resource.ISMLMetadata;
import org.eclipse.cosmos.rm.validation.internal.artifacts.ReferenceDescriptor;
import org.eclipse.cosmos.rm.validation.internal.common.ISMLConstants;
import org.eclipse.cosmos.rm.validation.internal.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.validation.internal.databuilders.ReferenceExtractor;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * The file system implementation of an instance SML document
 * 
 * @author Ali Mehregani
 */
public class SMLFileInstanceDocument extends SMLFileDocument implements ISMLInstanceDocument, Comparable<SMLFileDocument>, IItemConvertible
{

	/**
	 * The list of references for this document.
	 */
	private List<ReferenceDescriptor> references;

	
	public SMLFileInstanceDocument(ISMLRepository repository)
	{
		super(repository);		
	}
	
	/**
	 * @see org.eclipse.cosmos.rm.repository.resource.ISMLDocument#retrieveReferences(org.eclipse.cosmos.rm.repository.resource.ISMLMetadata)
	 */
	public ISMLDocument[] retrieveReferences(ISMLMetadata metadata) throws RepositoryOperationException
	{
		if (this.getMetadata() == null || this.getMetadata().getId() == null)
			return new ISMLDocument[0];
		
		File document = new File (this.getMetadata().getId());
		if (references == null || super.getLastModifiedDate() != document.lastModified())
		{
			if (!document.exists())
				return new ISMLDocument[0];
			
			ReferenceExtractor referenceExtractor = new ReferenceExtractor();
			try
			{
				SMLValidatorUtil.saxParseDocument(new FileInputStream(document), referenceExtractor);
			} 
			catch (Exception e)
			{
				throw new RepositoryOperationException(e.getLocalizedMessage(), e);
			}
			references = (List<ReferenceDescriptor>)referenceExtractor.getDataStructure();
		}

		String[] smlURIs = new String[references.size()];
		for (int i = 0; i < smlURIs.length; i++)
		{
			smlURIs[i] = references.get(i).getReference();
		}		
		for (int i = 0; i < smlURIs.length; i++)
		{
			int anchor = smlURIs[i].indexOf('#');
			if (anchor > 0)
				smlURIs[i] = smlURIs[i].substring(0, anchor);	
			
			smlURIs[i] = smlURIs[i].trim();		
		}
		SMLFileMetadata queryMetadata = new SMLFileMetadata();
		
		queryMetadata.setAliases(smlURIs);
		Set<ISMLDocument> references = new TreeSet<ISMLDocument>();
		ISMLDocument[] referencedDocuments = getRepository().fetchDocuments(queryMetadata);
		
		for (int i = 0; i < referencedDocuments.length; i++)
		{
			references.add(referencedDocuments[i]);	
		}
		queryMetadata.setAliases(null);
		for (int i = 0; i < smlURIs.length; i++)
		{
			queryMetadata.setId(smlURIs[i]);
			referencedDocuments = getRepository().fetchDocuments(queryMetadata);
			
			/* There can only be one reference fetched if id is being used */
			if (referencedDocuments != null && referencedDocuments.length == 1)
				references.add(referencedDocuments[0]);
		}		
		
		if (referencedDocuments == null)
			return new ISMLDocument[0];
		
		List<ISMLDocument> finalDocumentList = new ArrayList<ISMLDocument>();
		for (Iterator<ISMLDocument> referencesIterator = references.iterator(); referencesIterator.hasNext();)
		{
			ISMLDocument currentDocument = referencesIterator.next();
			ISMLMetadata documentMetadata = currentDocument.getMetadata();
			if (isEqual(new String[]{metadata.getId(), metadata.getRootElementName()}, 
						new String[]{documentMetadata.getId(), documentMetadata.getRootElementName()}) &&
				isEqual(metadata.getAliases(), documentMetadata.getAliases()) &&
				isEqual(metadata.getBoundRules(), documentMetadata.getBoundRules()) &&
				isEqual(metadata.getDocumentType(), documentMetadata.getDocumentType()))
			{
				finalDocumentList.add(currentDocument);
			}				
		}
		
		return finalDocumentList.toArray(new ISMLDocument[finalDocumentList.size()]);
	}

	private boolean isEqual(Map<String, ?> expected, Map<String, ?> actual)
	{
		if (expected == null)
			return actual == null || actual.size() == 0;
		
		if (expected.size() != actual.size())
			return false;
		
		for (Iterator<String> keys = expected.keySet().iterator(); keys.hasNext();)
		{
			String currentKey = keys.next();
			if (!expected.get(currentKey).equals(actual.get(currentKey)))
				return false;
		}
		
		return true;
	}

	private boolean isEqual(int expected, int actual)
	{
		if (expected < 0)
			return true;
		return expected == actual;
	}


	private boolean isEqual(String[] expected, String[] actual)
	{
		if (expected == null || expected.length == 0)
			return true;
		
		boolean foundEntry = true;
		for (int j = 0; j < expected.length; j++)
		{
			if (expected[j] == null)
				continue;
			
			foundEntry = false;
			if (actual != null)
			{
				for (int j2 = 0; j2 < actual.length; j2++)
				{
					if (expected[j].equals(actual[j2]))
					{
						foundEntry = true;
						break;
					}
				}
			}
			
			if (!foundEntry)
				return false;
		}
		
		return true;
	}
	
		
	/**
	 * @see org.eclipse.cosmos.rm.repository.resource.ISMLInstanceDocument#retrieveDefinitions()
	 */
	public ISMLDefinitionDocument[] retrieveDefinitions() throws RepositoryOperationException
	{
		String schemaLocation = retrieveRootAttribute (ISMLConstants.SCHEMA_INSTANCE_URI, ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE);		
		if (schemaLocation == null) {
			return new ISMLDefinitionDocument[] {};
		}
		
		List<ISMLDocument> definitionDocumentsList = new ArrayList<ISMLDocument>();
		StringTokenizer tokenizer = new StringTokenizer(schemaLocation);
		while (tokenizer.hasMoreTokens()) {
			tokenizer.nextToken(); // do nothing with this token?
			String definition = null;
			try {
				definition = tokenizer.nextToken();
			} catch (NoSuchElementException e) {
				break; // odd number of tokens.  report error?
			}
			if (!definition.startsWith("/"))
			{
				String filePath = this.getMetadata().getId();
				if (filePath != null)
				{
					int inx = filePath.lastIndexOf('/');
					if (inx >= 0)
						filePath = filePath.substring(0, inx + 1);
					definition = filePath + definition;	
				}			
			}
			ISMLDocument[] definitionDocument = getRepository().fetchDocuments(new SMLFileMetadata(definition, ISMLMetadata.DOCUMENT_TYPE_NONE, null, null, null));
			if ((definitionDocument != null) && (definitionDocument.length == 1)) {
				if (definitionDocument[0] instanceof ISMLDefinitionDocument) {
					definitionDocumentsList.add(definitionDocument[0]);
				}
			}
		}
		
		return definitionDocumentsList.toArray(new ISMLDefinitionDocument[definitionDocumentsList.size()]);
	}

	
	/**
	 * @see java.lang.Comparable#compareTo(T)
	 */
	public int compareTo(SMLFileDocument arg0)
	{
		ISMLMetadata metadata = ((SMLFileDocument)arg0).getMetadata();
		String id = getMetadata() == null ? null : getMetadata().getId();
		return id == null ? -1 : id.compareTo(metadata.getId());
	}

	
	public String toString()
	{		
		return super.toString();
	}

		
	public IItem toItem(INodes parent)
	{
		IItem item = QueryOutputArtifactFactory.getInstance().createItem();
		
		try
		{
			final Node node = getDOMDocument().getFirstChild();
			
			// Create the record
			IRecord record = QueryOutputArtifactFactory.getInstance().createRecord(item, getMetadata().getId());
			StringBuffer buffer = new StringBuffer();
			CMDBfInternalUtility.serializeNode(buffer, node);
			
			record.setValueFromString(buffer.toString());			
			
			// Find the name spaces
			NamedNodeMap attributes = node.getAttributes();		
			int colonInx;
			if (attributes != null)
			{
				for (int i = 0, attCount = attributes.getLength(); i < attCount; i++)
				{
					Node currentAttribute = attributes.item(i);
					String attLocalName = currentAttribute.getLocalName(); 
					if (attLocalName.startsWith(ISMLConstants.XML_NS_ATTRIBUTE))
					{
						String namespaceURI = currentAttribute.getNodeValue();
						String prefix = (colonInx = attLocalName.indexOf(':')) < 0 ? null : attLocalName.substring(colonInx + 1);
						
						if (namespaceURI != null && prefix != null)
						{
							record.addNamespace(prefix, namespaceURI);
						}
					}
				}
			}
						
			item.addRecord(record);		

			IInstanceId instanceId = QueryOutputArtifactFactory.getInstance().createInstanceId(CMDBfConstants.REPOSITORY_MDRID, getMetadata().getId());
			item.addInstanceId(instanceId);
			
			return item;
		} 
		catch (RepositoryOperationException e)
		{			
			return null;
		}		
	}		
}
