/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform;

import java.io.InputStream;
import java.io.Reader;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.output.artifacts.IDeregisterOutputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.output.artifacts.IDeregisterResponse;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.TransformerException;
import org.xml.sax.InputSource;

/**
 * Convert input stream containing an XML-based deregistration response into a data structure
 * 
 * @provisional
 * @author David Whiteman
 */
public class DeregistrationOutputTransformer {

	/**
	 * Transform an {@link IDeregisterResponse} into an InputStream containing the XML
	 * output for that object
	 * 
	 * @param response
	 * @return
	 */
	public static InputStream transform(IDeregisterResponse response) {
		return CMDBfServicesUtil.transformImpl((IXMLWritable) response);
	}

	/**
	 * Transform a stream of XML input containing a <deregisterResponse> structure to an IDeregisterResponse.
	 * 
	 * @param xmlInputStream
	 * @return
	 * @throws TransformerException
	 */
	public static IDeregisterResponse transform(InputStream xmlInputStream) throws TransformerException 
	{
		return transform(xmlInputStream, null, null);	
	}

	/**
	 * @param xmlInputStream
	 * @param artifactFactory
	 * @return
	 * @throws TransformerException
	 */
	public static IDeregisterResponse transform(InputStream xmlInputStream, IDeregisterOutputArtifactFactory artifactFactory) throws TransformerException 
	{
		return transform(xmlInputStream, artifactFactory, null);	
	}

	
	/**
	 * Transform a Reader containing a <deregisterResponse> structure to an IDeregisterResponse.
	 * 
	 * @param xmlInputStream
	 * @return
	 * @throws TransformerException
	 */
	public static IDeregisterResponse transform(Reader reader) throws TransformerException {
		return transform(reader, null, null);	
	}

	/**
	 * Transform a Reader containing a <deregisterResponse> structure to an IDeregisterResponse.
	 * 
	 * @param reader
	 * @param artifactFactory
	 * @return
	 * @throws TransformerException
	 */
	public static IDeregisterResponse transform(Reader reader, IDeregisterOutputArtifactFactory artifactFactory) throws TransformerException {
		return transform(reader, artifactFactory, null);	
	}

	/**
	 * Transform a stream of XML input containing a <deregisterResponse> structure to an IDeregisterResponse.
	 * Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param xmlInputStream
	 * @param transformerOptions
	 * @return
	 * @throws TransformerException
	 */
	public static IDeregisterResponse transform(InputStream xmlInputStream, IDeregisterOutputArtifactFactory artifactFactory, Map<String, Object> transformerOptions) throws TransformerException {
		return transformerOptions == null ? 
				(IDeregisterResponse) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new DeregistrationOutputStreamHandler(artifactFactory)) :
				(IDeregisterResponse) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new DeregistrationOutputStreamHandler(artifactFactory), transformerOptions);	
	}
	
	/**
	 * Transform a stream of XML input containing a <deregisterResponse> structure to an IDeregisterResponse.
	 * Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param reader
	 * @param transformerOptions
	 * @return
	 * @throws TransformerException
	 */
	public static IDeregisterResponse transform(Reader reader, IDeregisterOutputArtifactFactory artifactFactory, Map<String, Object> transformerOptions) throws TransformerException {
		return transformerOptions == null ? 
				(IDeregisterResponse) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new DeregistrationOutputStreamHandler(artifactFactory)) :
				(IDeregisterResponse) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new DeregistrationOutputStreamHandler(artifactFactory), transformerOptions);	
	}

}
