/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.registration.transform;


import java.io.StringWriter;

import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.IRootElement;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.output.artifacts.IDeregisterResponse;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IAlternateInstanceId;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IInstanceResponse;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IRegisterResponse;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IRegistrationServiceOutputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IResponse;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IInstanceId;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * SAX parser that builds {@link IResponse} object from XML input.  This is a generalized class
 * used for parsing both register response and deregister response structures, since these both
 * have identical schemas.
 * 
 * @provisional
 * @author David Whiteman
 */
public abstract class AbstractOutputStreamHandler extends DefaultHandler implements ITransformerHandler {
	private IResponse response;
	private String currentElementData;
	private IInstanceId currentInstanceId;
	private IInstanceResponse currentInstanceResponse;
	private IAlternateInstanceId currentAlternateInstanceId;
	private IRegistrationServiceOutputArtifactFactory artifactFactory;
	
	public AbstractOutputStreamHandler(
			IRegistrationServiceOutputArtifactFactory artifactFactory) {
		super();
		this.artifactFactory = artifactFactory;
	}

	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException 
	{
		if (!ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE.equals(uri))
		{
			return;
		}

		if (getResponseElementName().equals(localName)) {
			response = createResponse();
		} else if (IRegistrationTransformerConstants.INSTANCE_RESPONSE_ELEMENT.equals(localName)) {
			currentInstanceResponse = getArtifactFactory().createInstanceResponse();
			response.addInstanceResponse(currentInstanceResponse);
		} else if (IRegistrationTransformerConstants.ACCEPTED_ELEMENT.equals(localName)) {
			currentInstanceResponse.setAccepted(getArtifactFactory().createAccepted());
		} else if (IRegistrationTransformerConstants.DECLINED_ELEMENT.equals(localName)) {
			currentInstanceResponse.setDeclined(getArtifactFactory().createDeclined());
		} else if (ICMDBfServicesConstants.INSTANCE_ID_ELEMENT.equals(localName)) {
			currentInstanceId = getArtifactFactory().createInstanceId();
		} else if (IRegistrationTransformerConstants.ALTERNATE_INSTANCE_ID_ELEMENT.equals(localName)) {
			currentAlternateInstanceId = getArtifactFactory().createAlternateInstanceId();
		}

	}

	protected IResponse createResponse() {
		return getArtifactFactory().createResponse();
	}

	protected abstract String getResponseElementName();

	public void endElement(String uri, String localName, String name) throws SAXException 
	{
		if (!ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE.equals(uri))
		{
			return;
		}
		if (ICMDBfServicesConstants.MDR_ID_ELEMENT.equals(localName)) {
			// could be an </mdrId> in either an <instanceId> or <alternateInstanceId>
			if (currentInstanceId != null) {
				currentInstanceId.setMdrId(currentElementData);
			} else if (currentAlternateInstanceId != null) {
				currentAlternateInstanceId.setMdrId(currentElementData);
			}
			currentElementData = null;
		} else if (ICMDBfServicesConstants.LOCAL_ID_ELEMENT.equals(localName)) {
			// could be a </localId> in either an <instanceId> or <alternateInstanceId>
			if (currentInstanceId != null) {
				currentInstanceId.setLocalId(currentElementData);
			} else if (currentAlternateInstanceId != null) {
				currentAlternateInstanceId.setLocalId(currentElementData);
			}
			currentElementData = null;
		} else if (ICMDBfServicesConstants.INSTANCE_ID_ELEMENT.equals(localName)) {
			currentInstanceResponse.setInstanceId(currentInstanceId);
			currentInstanceId = null;
		} else if (IRegistrationTransformerConstants.ALTERNATE_INSTANCE_ID_ELEMENT.equals(localName)) {
			currentInstanceResponse.getAccepted().addAlternateInstanceId(currentAlternateInstanceId);
			currentAlternateInstanceId = null;
		} else if (IRegistrationTransformerConstants.REASON_ELEMENT.equals(localName)) {
			currentInstanceResponse.getDeclined().addReason(currentElementData);
			currentElementData = null;
		} 
				
	}

	public void characters(char[] ch, int start, int length) throws SAXException 
	{
		// General logic for building up any data between a begin and end tag
		
		String tempData = new String(ch, start, length).trim();
		if (currentElementData == null) {
			currentElementData = tempData;
		} else {
			StringWriter tempWriter = new StringWriter();
			tempWriter.write(tempData);
			currentElementData += tempWriter.toString();
		}
	}

	/**
	 * Answer either a {@link IDeregisterResponse} or {@link IRegisterResponse} created
	 * by this handler.
	 * 
	 * @return
	 */
	public IResponse getResponse() 
	{
		return response;
	}

	public void error(SAXParseException e) throws SAXException 
	{
		throw e;
	}

	/* Answer the root element created by this handler
	 * 
	 * @see org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler#getResult()
	 */
	public IRootElement getResult() {
		return (IRootElement) getResponse();
	}

	protected IRegistrationServiceOutputArtifactFactory getArtifactFactory() {
		if (artifactFactory == null) {
			artifactFactory = createArtifactFactory();
		}
		return artifactFactory;
	}

	protected abstract IRegistrationServiceOutputArtifactFactory createArtifactFactory();
}
