/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.registration.transform;

import java.io.StringWriter;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.IRootElement;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.IRegisterInputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.IRegisterRequest;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.RegisterInputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IAdditionalRecordType;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IInstanceId;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRecordMetadata;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRelationship;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * SAX parser that builds IRegisterRequest object from XML input
 * 
 * @provisional
 * @author David Whiteman
 */
public class RegistrationInputStreamHandler extends DefaultHandler implements ITransformerHandler {
	private IRegisterRequest request;
	private IGraphElement currentRegisterEntity;
	private String currentElementData;
	private IInstanceId currentInstanceId;
	private IRecord currentRecord;
	private IRecordMetadata currentRecordMetadata;
	private IRegisterInputArtifactFactory artifactFactory;
	
	public RegistrationInputStreamHandler(
			IRegisterInputArtifactFactory artifactFactory) {
		this.artifactFactory = artifactFactory;
	}

	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException 
	{
		if (!ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE.equals(uri))
		{
			if (currentRecord == null) 
			{
				return;
			}
			
			StringWriter tempWriter = new StringWriter();
			CMDBfServicesUtil.addIndent(tempWriter, 4);
			tempWriter.write(CMDBfServicesUtil.beginTagFor(name, attributes));
			tempWriter.write(ICMDBfServicesConstants.nl);
			currentElementData += tempWriter.toString();
			if (!name.equals(localName)) {
				String prefix = name.split(ICMDBfServicesConstants.COLON)[0];
				currentRecord.addNamespace(prefix, uri);
			}
			return;
		}

		if (IRegistrationTransformerConstants.REGISTER_REQUEST_ELEMENT.equals(localName)) 
		{
			request = getArtifactFactory().createRegisterRequest();
		} 
		else if (ICMDBfServicesConstants.ITEM_ELEMENT.equals(localName)) 
		{
			currentRegisterEntity = getArtifactFactory().createItem();
			request.addItem((IItem) currentRegisterEntity);
		} 
		else if (ICMDBfServicesConstants.RELATIONSHIP_ELEMENT.equals(localName)) 
		{
			currentRegisterEntity = getArtifactFactory().createRelationship();
			request.addRelationship((IRelationship) currentRegisterEntity);
		} 
		else if (ICMDBfServicesConstants.SOURCE_ELEMENT.equals(localName)) 
		{
			currentInstanceId = getArtifactFactory().createInstanceId();
		} 
		else if (ICMDBfServicesConstants.TARGET_ELEMENT.equals(localName)) 
		{
			currentInstanceId = getArtifactFactory().createInstanceId();
		} 
		
		// <record>
		else if (ICMDBfServicesConstants.RECORD_ELEMENT.equals(localName)) 
		{
			currentRecord = getArtifactFactory().createRecord((IGraphElement) currentRegisterEntity, null);
			currentRegisterEntity.addRecord(currentRecord);
			currentElementData = "";
		} 
		
		// <recordMetadata>
		else if (currentRecord != null && ICMDBfServicesConstants.RECORD_METADATA_ELEMENT.equals(localName)) 
		{
			currentRecordMetadata = getArtifactFactory().createRecordMetadata();
			currentRecord.setRecordMetadata(currentRecordMetadata);
			currentRecord.setValueFromString(currentElementData);
			currentElementData = "";
		}
		
		else if (ICMDBfServicesConstants.INSTANCE_ID_ELEMENT.equals(localName)) 
		{
			currentInstanceId = getArtifactFactory().createInstanceId();
		} 
		else if (ICMDBfServicesConstants.ADDITIONAL_RECORD_TYPE_ELEMENT.equals(localName)) {
			String namespace = attributes.getValue(ICMDBfServicesConstants.NAMESPACE_ATTRIBUTE);
			String locName = attributes.getValue(ICMDBfServicesConstants.LOCAL_NAME_ATTRIBUTE);
			IAdditionalRecordType adt = getArtifactFactory().createAdditionalRecordType(namespace, locName);
			currentRegisterEntity.addAdditionalRecordType(adt);
		} 

	}

	
	public void endElement(String uri, String localName, String name) throws SAXException 
	{
		if (!ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE.equals(uri))
		{
			if (currentRecord != null) {
				StringWriter tempWriter = new StringWriter();
				CMDBfServicesUtil.addIndent(tempWriter, 4);
				tempWriter.write(CMDBfServicesUtil.endTagFor(name));
				tempWriter.write(ICMDBfServicesConstants.nl);
				currentElementData += tempWriter.toString();
			}
			return;
		}
		
		if (ICMDBfServicesConstants.MDR_ID_ELEMENT.equals(localName)) 
		{
			if (currentInstanceId != null) {
				// This is an mdrId found inside some instanceId element
				currentInstanceId.setMdrId(currentElementData.trim());
			} else {
				// Must be the mdrId for the register request
				request.setMdrId(CMDBfServicesUtil.createURI(currentElementData));
			}
			currentElementData = null;
		} 
		else if (ICMDBfServicesConstants.LOCAL_ID_ELEMENT.equals(localName)) 
		{
			currentInstanceId.setLocalId(currentElementData);
			currentElementData = null;
		} 
		else if (ICMDBfServicesConstants.INSTANCE_ID_ELEMENT.equals(localName)) 
		{
			currentRegisterEntity.addInstanceId(currentInstanceId);
			currentInstanceId = null;
		} 
		else if (ICMDBfServicesConstants.SOURCE_ELEMENT.equals(localName)) 
		{
			((IRelationship) currentRegisterEntity).setSourceId(currentInstanceId);
			currentInstanceId = null;
		}
		else if (ICMDBfServicesConstants.TARGET_ELEMENT.equals(localName)) 
		{
			((IRelationship) currentRegisterEntity).setTargetId(currentInstanceId);
			currentInstanceId = null;
		} 
		else if (ICMDBfServicesConstants.RECORD_ELEMENT.equals(localName)) 
		{
			currentRecord = null;
		}
		
		// <recordMetadata>
		else if (ICMDBfServicesConstants.RECORD_METADATA_ELEMENT.equals(localName))
		{
			// handle xml:any data following known element types
			if ((currentElementData != null) && (!"".equals(currentElementData.trim()))) {
				currentRecordMetadata.setExtensibilityElementsFromString(currentElementData);
			}
			currentRecordMetadata = null;
			currentElementData = "";
		}
		
		// <recordId>
		else if (currentRecordMetadata != null && ICMDBfServicesConstants.RECORD_ID_ELEMENT.equals(localName))
		{
			currentRecordMetadata.setRecordId(currentElementData == null ? null : CMDBfServicesUtil.createURI(currentElementData.trim()));
			currentElementData = "";
		}		
		
		// <lastModified>
		else if (currentRecordMetadata != null && ICMDBfServicesConstants.LAST_MODIFIED_ELEMENT.equals(localName))
		{			
			currentRecordMetadata.setLastModified(currentElementData);
			currentElementData = null;
		}
		
		// <baselineId>
		else if (currentRecordMetadata != null && ICMDBfServicesConstants.BASE_LINE_ID_ELEMENT.equals(localName))
		{
			currentRecordMetadata.setBaselineId(currentElementData);
			currentElementData = null;
		}
		
		// <snapshotId>
		else if (currentRecordMetadata != null && ICMDBfServicesConstants.SNAPSHOT_ID_ELEMENT.equals(localName))
		{
			currentRecordMetadata.setSnapshotId(currentElementData);
			currentElementData = null;
		}
	}

	public void characters(char[] ch, int start, int length) throws SAXException 
	{
		String tempData = new String(ch, start, length).trim();
		if (currentElementData == null) {
			currentElementData = tempData;
		} else {
			StringWriter tempWriter = new StringWriter();
			if (currentRecord != null) {
				CMDBfServicesUtil.addIndent(tempWriter, 4);
			}
			tempWriter.write(tempData);
			if (currentRecord != null) {
				tempWriter.write('\n');
			}
			currentElementData += tempWriter.toString();
		}
	}

	public IRegisterRequest getRequest() 
	{
		return request;
	}

	public void error(SAXParseException e) throws SAXException 
	{
		throw e;
	}

	public IRootElement getResult() {
		return (IRootElement) getRequest();
	}

	protected IRegisterInputArtifactFactory getArtifactFactory() {
		if (artifactFactory == null) {
			artifactFactory = RegisterInputArtifactFactory.getInstance();
		}
		return artifactFactory;
	}
}
