/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.registration.transform;

import java.io.InputStream;
import java.io.Reader;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.IRegisterInputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.IRegisterRequest;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.TransformerException;
import org.xml.sax.InputSource;

/**
 * Convert input stream containing an XML-based registration request to and from a data structure
 * 
 * @provisional
 * @author David Whiteman
 */
public class RegistrationInputTransformer {

	/**
	 * Transform a stream of XML input containing a <registerRequest> structure to an IRegisterRequest
	 * 
	 * @param request IRegisterRequest to be converted to XML
	 * @return InputStream containing XML conversion of IRegisterRequest
	 */
	public static InputStream transform(IRegisterRequest request) {
		return CMDBfServicesUtil.transformImpl((IXMLWritable) request);
	}

	/**
	 * Transform a stream of XML input containing a <registerRequest> structure to an IRegisterRequest
	 * 
	 * @param xmlInputStream InputStream containing XML with <registerRequest> being the topmost element
	 * @return IRegisterRequest graph structure of POJOs used by MDRs
	 * @throws TransformerException
	 */
	public static IRegisterRequest transform(InputStream xmlInputStream) throws TransformerException 
	{
		return transform(xmlInputStream, null, null);	
	}

	/**
	 * @param xmlInputStream
	 * @param artifactFactory
	 * @return
	 * @throws TransformerException
	 */
	public static IRegisterRequest transform(InputStream xmlInputStream, IRegisterInputArtifactFactory artifactFactory) throws TransformerException 
	{
		return transform(xmlInputStream, artifactFactory, null);	
	}

	/**
	 * Transform a stream of XML input containing a <registerRequest> structure to an IRegisterRequest
	 * 
	 * @param reader InputStream containing XML with <registerRequest> being the topmost element
	 * @return IRegisterRequest graph structure of POJOs used by MDRs
	 * @throws TransformerException
	 */
	public static IRegisterRequest transform(Reader reader) throws TransformerException 
	{
		return transform(reader, null, null);	
	}

	/**
	 * @param reader
	 * @param artifactFactory
	 * @return
	 * @throws TransformerException
	 */
	public static IRegisterRequest transform(Reader reader, IRegisterInputArtifactFactory artifactFactory) throws TransformerException 
	{
		return transform(reader, artifactFactory, null);	
	}

	/**
	 * Transform a stream of XML input containing a <registerRequest> structure to an IRegisterRequest.
	 * Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param xmlInputStream InputStream containing XML with <registerRequest> being the topmost element
	 * @param transformOptions Map of transformer options
	 * @return IRegisterRequest graph structure of POJOs used by MDRs
	 * @throws TransformerException
	 */
	public static IRegisterRequest transform(InputStream xmlInputStream, IRegisterInputArtifactFactory artifactFactory, Map<String, Object> transformOptions) throws TransformerException 
	{
		return transformOptions == null ? 
				(IRegisterRequest) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new RegistrationInputStreamHandler(artifactFactory)) :
				(IRegisterRequest) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new RegistrationInputStreamHandler(artifactFactory), transformOptions);	
	}

	
	/**
	 * Transform a stream of XML input containing a <registerRequest> structure to an IRegisterRequest.
	 * Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param reader InputStream containing XML with <registerRequest> being the topmost element
	 * @param transformOptions Map of transformer options
	 * @return IRegisterRequest graph structure of POJOs used by MDRs
	 * @throws TransformerException
	 */
	public static IRegisterRequest transform(Reader reader, IRegisterInputArtifactFactory artifactFactory, Map<String, Object> transformOptions) throws TransformerException 
	{
		return transformOptions == null ? 
				(IRegisterRequest) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new RegistrationInputStreamHandler(artifactFactory)) :
				(IRegisterRequest) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new RegistrationInputStreamHandler(artifactFactory), transformOptions);	
	}
}
