/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.registration.transform;

import java.io.InputStream;
import java.io.Reader;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IRegisterOutputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IRegisterResponse;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.TransformerException;
import org.xml.sax.InputSource;

/**
 * Convert input stream containing an XML-based registration response to and from a data structure
 * 
 * @provisional
 * @author David Whiteman
 */
public class RegistrationOutputTransformer {

	/**
	 * Transform an {@link IRegisterResponse} to an InputStream of XML
	 * 
	 * @param response IRegisterResponse
	 * @return
	 */
	public static InputStream transform(IRegisterResponse response) {
		return CMDBfServicesUtil.transformImpl((IXMLWritable) response);
	}

	/**
	 * Transform an InputStream of XML to an {@link IRegisterResponse}
	 * 
	 * @param xmlInputStream
	 * @return
	 * @throws TransformerException
	 */
	public static IRegisterResponse transform(InputStream xmlInputStream) throws TransformerException {
		return transform(xmlInputStream, null, null);	
	}

	public static IRegisterResponse transform(InputStream xmlInputStream, IRegisterOutputArtifactFactory artifactFactory) throws TransformerException {
		return transform(xmlInputStream, artifactFactory, null);	
	}

	/**
	 * Transform an InputStream of XML to an {@link IRegisterResponse}
	 * 
	 * @param reader
	 * @return
	 * @throws TransformerException
	 */
	public static IRegisterResponse transform(Reader reader) throws TransformerException {
		return transform(reader, null, null);	
	}

	/**
	 * @param reader
	 * @param artifactFactory
	 * @return
	 * @throws TransformerException
	 */
	public static IRegisterResponse transform(Reader reader, IRegisterOutputArtifactFactory artifactFactory) throws TransformerException {
		return transform(reader, artifactFactory, null);	
	}

	/**
	 * Transform an InputStream of XML to an {@link IRegisterResponse}
	 * Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param xmlInputStream
	 * @param transformerOptions
	 * @return {@link IRegisterResponse}
	 * @throws TransformerException
	 */
	public static IRegisterResponse transform(InputStream xmlInputStream, IRegisterOutputArtifactFactory artifactFactory, Map<String, Object> transformerOptions) throws TransformerException 
	{
		return transformerOptions == null ? 
				(IRegisterResponse) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new RegistrationOutputStreamHandler(artifactFactory)) :
				(IRegisterResponse) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new RegistrationOutputStreamHandler(artifactFactory), transformerOptions);	
	}
	
	
	/**
	 * Transform an InputStream of XML to an {@link IRegisterResponse}
	 * Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param reader
	 * @param transformerOptions
	 * @return {@link IRegisterResponse}
	 * @throws TransformerException
	 */
	public static IRegisterResponse transform(Reader reader, IRegisterOutputArtifactFactory artifactFactory, Map<String, Object> transformerOptions) throws TransformerException 
	{
		return transformerOptions == null ? 
				(IRegisterResponse) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new RegistrationOutputStreamHandler(artifactFactory)) :
				(IRegisterResponse) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new RegistrationOutputStreamHandler(artifactFactory), transformerOptions);	
	}

}
