/*
 * Copyright (c) 2005-2007 Compuware Corporation and others.
 * All rights reserved. This program and the accompanying
 * materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at:
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 *
 */
package org.eclipse.cosmos.me.management.provisional.wsdm.impl;

import java.io.IOException;
import java.io.StringReader;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import javax.management.MBeanAttributeInfo;
import javax.management.MBeanParameterInfo;

import org.apache.muse.core.Capability;
import org.apache.muse.util.xml.XmlUtils;
import org.apache.muse.ws.metadata.WsxConstants;
import org.apache.muse.ws.metadata.impl.SimpleMetadataExchange;
import org.apache.muse.ws.wsdl.WsdlUtils;
import org.eclipse.cosmos.me.management.annotations.ManagedOperation;
import org.eclipse.cosmos.me.management.annotations.ManagedParameter;
import org.eclipse.cosmos.me.management.common.info.ManagedAttributeInfo;
import org.eclipse.cosmos.me.management.common.info.ManagedOperationInfo;
import org.eclipse.cosmos.me.management.common.info.ManagedResourceInfo;
import org.eclipse.cosmos.me.management.common.util.ManagementProxy;
import org.eclipse.cosmos.me.management.provisional.wsdm.PolicyAssertion;
import org.eclipse.cosmos.me.management.provisional.wsdm.PolicyAssertionProvider;
import org.eclipse.cosmos.me.management.provisional.wsdm.annotations.WSDMOperation;
import org.eclipse.cosmos.me.management.provisional.wsdm.annotations.WSDMParameterType;
import org.eclipse.cosmos.me.management.provisional.wsdm.util.SchemaHelper;
import org.eclipse.cosmos.me.management.provisional.wsdm.impl.info.WSDMInfo;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class WSDLSource extends SimpleMetadataExchange {

	private static HashMap<String, String> java2xsdType = new HashMap<String, String>();

	static {
		java2xsdType.put("java.lang.String", "xsd:string");
		java2xsdType.put("java.lang.Integer", "xsd:integer");
		java2xsdType.put("java.lang.Boolean", "xsd:boolean");
		java2xsdType.put("boolean", "xsd:boolean");
		java2xsdType.put("int", "xsd:integer");
		java2xsdType.put("[Ljava.lang.String;", "xsd:string");
	}

	private static final String WSDL_START = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
			+ "<wsdl:definitions targetNamespace=\"";

	private static final String WSDL_DEFAULT_NAMESPACES = " xmlns=\"http://schemas.xmlsoap.org/wsdl/\""
			+ " xmlns:wsa=\"http://www.w3.org/2005/08/addressing\""
			+ " xmlns:wsdl=\"http://schemas.xmlsoap.org/wsdl/\""
			+ " xmlns:wsdl-soap=\"http://schemas.xmlsoap.org/wsdl/soap/\""
			+ " xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\""
			+ " xmlns:muws1=\"http://docs.oasis-open.org/wsdm/muws1-2.xsd\""
			+ " xmlns:muws2=\"http://docs.oasis-open.org/wsdm/muws2-2.xsd\""
			+ " xmlns:wsrf-r=\"http://docs.oasis-open.org/wsrf/r-2\"" 
			+ " xmlns:wsp=\"http://www.w3.org/ns/ws-policy\""
			+ " xmlns:wsx=\"http://schemas.xmlsoap.org/ws/2004/09/mex\"";

	// " xmlns:wsrf-rl=\"http://docs.oasis-open.org/wsrf/rl-2\"" +
	// " xmlns:wsrf-bf=\"http://docs.oasis-open.org/wsrf/bf-2\"" +
	// " xmlns:muws1=\"http://docs.oasis-open.org/wsdm/muws1-2.xsd\"";

	private static final String WSDL_NOTIFICATION_NAMESPACES = " xmlns:wsnt=\"http://docs.oasis-open.org/wsn/b-2\""
			+ " xmlns:wsntw=\"http://docs.oasis-open.org/wsn/bw-2\""
			+ " xmlns:wst=\"http://docs.oasis-open.org/wsn/t-1\"";

	private static final String WSDL_RMD_NAMESPACE = " xmlns:wsrmd=\"http://docs.oasis-open.org/wsrf/rmd-1\"";

	private static final String WSDL_NOTIFICATION_WSDL_IMPORT = "<wsdl:import namespace=\"http://docs.oasis-open.org/wsn/bw-2\""
			+ " location=\"WS-BaseNotification-1_3.wsdl\"/>";

	private static final String WSDL_TYPE_START = "<wsdl:types>"
			+ "<xsd:schema" + " elementFormDefault=\"qualified\""
			+ " targetNamespace=\"http://www.w3.org/2005/08/addressing\">"
			+ "<xsd:include schemaLocation=\"WS-Addressing-2005_08.xsd\"/>"
			+ "</xsd:schema>"
			+ "<xsd:schema" 
			+ " elementFormDefault=\"qualified\""
			+ " targetNamespace=\"http://schemas.xmlsoap.org/ws/2004/09/mex\">"
			+ "<xsd:include schemaLocation=\"WS-MetadataExchange-2004_09.xsd\"/>" 
			+ "</xsd:schema>";

	private static final String WSDL_WSRF_TYPES = "<xsd:schema"
			+ " elementFormDefault=\"qualified\""
			+ " targetNamespace=\"http://docs.oasis-open.org/wsrf/rp-2\">"
			+ "<xsd:include schemaLocation=\"WS-ResourceProperties-1_2.xsd\"/>"
			+ "</xsd:schema>" + "<xsd:schema"
			+ " elementFormDefault=\"qualified\""
			+ " targetNamespace=\"http://docs.oasis-open.org/wsrf/r-2\">"
			+ "<xsd:include schemaLocation=\"WS-Resource-1_2.xsd\" />"
			+ "</xsd:schema>";
	// "<xsd:schema" +
	// " elementFormDefault=\"qualified\"" +
	// " targetNamespace=\"http://docs.oasis-open.org/wsrf/rl-2\">" +
	// "<xsd:include schemaLocation=\"WS-ResourceLifetime-1_2.xsd\"/>" +
	// "</xsd:schema>" +
	// "<xsd:schema" +
	// " elementFormDefault=\"qualified\"" +
	// " targetNamespace=\"http://docs.oasis-open.org/wsrf/rmd-1\">" +
	// "<xsd:include schemaLocation=\"WS-ResourceMetadataDescriptor-CD-01.xsd\"
	// />" +
	// "</xsd:schema>";

	private static final String WSDL_MUWS1_TYPES = "<xsd:schema"
			+ " elementFormDefault=\"qualified\""
			+ " targetNamespace=\"http://docs.oasis-open.org/wsdm/muws1-2.xsd\">"
			+ "<xsd:include schemaLocation=\"WSDM-MUWS-Part1-1_1.xsd\" />"
			+ "</xsd:schema>"
			+ "<xsd:schema"
			+ " elementFormDefault=\"qualified\""
			+ " targetNamespace=\"http://docs.oasis-open.org/wsdm/muws2-2.xsd\">"
			+ "<xsd:include schemaLocation=\"WSDM-MUWS-Part2-1_1.xsd\" />"
			+ "</xsd:schema>";

	private static final String WSDL_WSN_TYPES = "<xsd:schema"
			+ " elementFormDefault=\"qualified\""
			+ " targetNamespace=\"http://docs.oasis-open.org/wsn/b-2\">"
			+ "<xsd:include schemaLocation=\"WS-BaseNotification-1_3.xsd\" />"
			+ "</xsd:schema>" + "<xsd:schema"
			+ " elementFormDefault=\"qualified\""
			+ " targetNamespace=\"http://docs.oasis-open.org/wsn/t-1\">"
			+ "<xsd:include schemaLocation=\"WS-Topics-1_3.xsd\" />"
			+ "</xsd:schema>";

	private static final String WSDL_TYPE_END = "</wsdl:types>\n";

	private static final String WSDL_RESOURCE_SCHEMA_START = "<xsd:schema"
			+ " elementFormDefault=\"qualified\"" + " targetNamespace=\"";

	private static final String WSDL_RESOURCE_PROPERTIES_START = "<xsd:element name=\"WsResourceProperties\">"
			+ "<xsd:complexType>" + "<xsd:sequence>";

	private static final String WSDL_MUWS1_PROPERTIES = "<xsd:element ref=\"muws1:ResourceId\" />"
			+ "<xsd:element ref=\"muws1:ManageabilityCapability\" minOccurs=\"0\" maxOccurs=\"unbounded\"/>";
	// "<xsd:element ref=\"muws2:Caption\" minOccurs=\"0\"
	// maxOccurs=\"unbounded\"/>"+
	// "<xsd:element ref=\"muws2:Description\" minOccurs=\"0\"
	// maxOccurs=\"unbounded\"/>"+
	// "<xsd:element ref=\"muws2:Version\" minOccurs=\"0\"/>"+
	// "<xsd:element ref=\"muws2:OperationalStatus\"/>";

	private static final String WSDL_WSRF_RESOURCE_LIFETIME_PROPERTIES = "<xsd:element ref=\"wsrf-rl:CurrentTime\" />"
			+ "<xsd:element ref=\"wsrf-rl:TerminationTime\" />";

	private static final String WSDL_WSRF_RESOURCE_QUERY_PROPERTIES = "<xsd:element ref=\"wsrf-rp:QueryExpressionDialect\" minOccurs=\"0\" maxOccurs=\"unbounded\"/>";

	private static final String WSDL_WSDM_RELATIONSHIPS_PROPERTIES = "<xsd:element ref=\"muws2:Relationship\" minOccurs=\"0\" maxOccurs=\"unbounded\"/>";

	private static final String WSDL_WSN_PROPERTIES = "<xsd:element ref=\"wsnt:FixedTopicSet\"/>"
			+ "<xsd:element ref=\"wst:TopicSet\" minOccurs=\"0\"/>"
			+ "<xsd:element ref=\"wsnt:TopicExpression\" minOccurs=\"0\" maxOccurs=\"unbounded\"/>"
			+ "<xsd:element ref=\"wsnt:TopicExpressionDialect\" minOccurs=\"0\" maxOccurs=\"unbounded\"/>";

	private static final String WSDL_METRIC_PROPERTIES = "<xsd:element ref=\"muws2:CurrentTime\"/>";

	private static final String WSDL_RESOURCE_PROPERTIES_END = "</xsd:sequence>"
			+ "</xsd:complexType>" + "</xsd:element>\n";

	private static final String WSDL_RESOURCE_SCHEMA_END = "</xsd:schema>\n";

	private static final String WSDL_END = "</wsdl:definitions>\n";

	private static final String WSDL_MEX_MESSAGES = "<wsdl:message name=\"GetMetadataMsg\">"
			+ "<wsdl:part name=\"GetMetadataMsg\" element=\"wsx:GetMetadata\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"GetMetadataResponseMsg\">"
			+ "<wsdl:part name=\"GetMetadataResponseMsg\" element=\"wsx:Metadata\" />"
			+ "</wsdl:message>";

	private static final String WSDL_WSRF_RESOURCE_FAULT_MESSAGES = "<wsdl:message name=\"ResourceUnknownFault\">"
			+ "<wsdl:part name=\"ResourceUnknownFault\" element=\"wsrf-r:ResourceUnknownFault\" />"
			+ "</wsdl:message> "
			+ "<wsdl:message name=\"ResourceUnavailableFault\">"
			+ "<wsdl:part name=\"ResourceUnavailableFault\" element=\"wsrf-r:ResourceUnavailableFault\" />"
			+ "</wsdl:message>";

	private static final String WSDL_WSRF_RESOURCE_LIFETIME_MESSAGES = "<wsdl:message name=\"DestroyRequest\">"
			+ "<wsdl:part name=\"DestroyRequest\" element=\"wsrf-rl:Destroy\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"DestroyResponse\">"
			+ "<wsdl:part name=\"DestroyResponse\" element=\"wsrf-rl:DestroyResponse\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"ResourceNotDestroyedFault\">"
			+ "<wsdl:part name=\"ResourceNotDestroyedFault\" element=\"wsrf-rl:ResourceNotDestroyedFault\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"SetTerminationTimeRequest\">"
			+ "<wsdl:part name=\"SetTerminationTimeRequest\" element=\"wsrf-rl:SetTerminationTime\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"SetTerminationTimeResponse\">"
			+ "<wsdl:part name=\"SetTerminationTimeResponse\" element=\"wsrf-rl:SetTerminationTimeResponse\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"UnableToSetTerminationTimeFault\">"
			+ "<wsdl:part name=\"UnableToSetTerminationTimeFault\" element=\"wsrf-rl:UnableToSetTerminationTimeFault\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"TerminationTimeChangeRejectedFault\">"
			+ "<wsdl:part name=\"TerminationTimeChangeRejectedFault\" element=\"wsrf-rl:TerminationTimeChangeRejectedFault\" />"
			+ "</wsdl:message>";

	private static final String WSDL_WSDM_RELATIONSHIP_MESSAGES = "<wsdl:message name=\"QueryRelationshipsByTypeRequest\">"
			+ "<wsdl:part name=\"QueryRelationshipsByTypeRequest\" element=\"muws2:QueryRelationshipsByType\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"QueryRelationshipsByTypeResponse\">"
			+ "<wsdl:part name=\"QueryRelationshipsByTypeResponse\" element=\"muws2:QueryRelationshipsByTypeResponse\" />"
			+ "</wsdl:message>";

	private static final String WSDL_WSRF_RESOURCE_PROPERTY_MESSAGES = "<wsdl:message name=\"GetResourcePropertyDocumentRequest\">"
			+ "<wsdl:part name=\"GetResourcePropertyDocumentRequest\" element=\"wsrf-rp:GetResourcePropertyDocument\"/>"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"GetResourcePropertyDocumentResponse\">"
			+ "<wsdl:part name=\"GetResourcePropertyDocumentResponse\" element=\"wsrf-rp:GetResourcePropertyDocumentResponse\"/>"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"GetResourcePropertyRequest\">"
			+ "<wsdl:part name=\"GetResourcePropertyRequest\" element=\"wsrf-rp:GetResourceProperty\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"GetResourcePropertyResponse\">"
			+ "<wsdl:part name=\"GetResourcePropertyResponse\" element=\"wsrf-rp:GetResourcePropertyResponse\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"InvalidResourcePropertyQNameFault\">"
			+ "<wsdl:part name=\"InvalidResourcePropertyQNameFault\" element=\"wsrf-rp:InvalidResourcePropertyQNameFault\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"GetMultipleResourcePropertiesRequest\">"
			+ "<wsdl:part name=\"GetMultipleResourcePropertiesRequest\" element=\"wsrf-rp:GetMultipleResourceProperties\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"GetMultipleResourcePropertiesResponse\">"
			+ "<wsdl:part name=\"GetMultipleResourcePropertiesResponse\" element=\"wsrf-rp:GetMultipleResourcePropertiesResponse\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"QueryResourcePropertiesRequest\">"
			+ "<wsdl:part name=\"QueryResourcePropertiesRequest\" element=\"wsrf-rp:QueryResourceProperties\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"QueryResourcePropertiesResponse\">"
			+ "<wsdl:part name=\"QueryResourcePropertiesResponse\" element=\"wsrf-rp:QueryResourcePropertiesResponse\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"UnknownQueryExpressionDialectFault\">"
			+ "<wsdl:part name=\"UnknownQueryExpressionDialectFault\" element=\"wsrf-rp:UnknownQueryExpressionDialectFault\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"InvalidQueryExpressionFault\">"
			+ "<wsdl:part name=\"InvalidQueryExpressionFault\" element=\"wsrf-rp:InvalidQueryExpressionFault\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"QueryEvaluationErrorFault\">"
			+ "<wsdl:part name=\"QueryEvaluationErrorFault\" element=\"wsrf-rp:QueryEvaluationErrorFault\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"SetResourcePropertiesRequest\">"
			+ "<wsdl:part name=\"SetResourcePropertiesRequest\" element=\"wsrf-rp:SetResourceProperties\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"SetResourcePropertiesResponse\">"
			+ "<wsdl:part name=\"SetResourcePropertiesResponse\" element=\"wsrf-rp:SetResourcePropertiesResponse\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"InvalidModificationFault\">"
			+ "<wsdl:part name=\"InvalidModificationFault\" element=\"wsrf-rp:InvalidModificationFault\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"UnableToModifyResourcePropertyFault\">"
			+ "<wsdl:part name=\"UnableToModifyResourcePropertyFault\" element=\"wsrf-rp:UnableToModifyResourcePropertyFault\" />"
			+ "</wsdl:message>"
			+ "<wsdl:message name=\"SetResourcePropertyRequestFailedFault\">"
			+ "<wsdl:part name=\"SetResourcePropertyRequestFailedFault\" element=\"wsrf-rp:SetResourcePropertyRequestFailedFault\" />"
			+ "</wsdl:message>";

	private static final String WSDL_PORT_TYPE_START = "<wsdl:portType ";
	// " wsrf-rp:ResourceProperties=\"dyn:WsResourceProperties\"";

	private static final String WSDL_MEX_OPERATIONS = "<wsdl:operation name=\"GetMetadata\">"
			+ "<wsdl:input wsa:Action=\"http://schemas.xmlsoap.org/ws/2004/09/mex/GetMetadata\""
			+ " name=\"GetMetadataMsg\" message=\"dyn:GetMetadataMsg\"/>"
			+ "<wsdl:output wsa:Action=\"http://schemas.xmlsoap.org/ws/2004/09/mex/GetMetadataResponse\""
			+ " name=\"GetMetadataResponseMsg\" message=\"dyn:GetMetadataResponseMsg\"/>"
			+ "</wsdl:operation>";

	private static final String WSDL_WSRF_RESOURCE_PROPERTY_OPERATIONS = "<wsdl:operation name=\"GetResourcePropertyDocument\">"
			+ "<wsdl:input wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/GetResourcePropertyDocument/GetResourcePropertyDocumentRequest\""
			+ " name=\"GetResourcePropertyDocumentRequest\" message=\"dyn:GetResourcePropertyDocumentRequest\"/>"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/GetResourcePropertyDocument/GetResourcePropertyDocumentResponse\""
			+ " name=\"GetResourcePropertyDocumentResponse\" message=\"dyn:GetResourcePropertyDocumentResponse\"/>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\" message=\"dyn:ResourceUnknownFault\"/>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\" message=\"dyn:ResourceUnavailableFault\"/>"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"GetResourceProperty\">"
			+ "<wsdl:input wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/GetResourceProperty/GetResourcePropertyRequest\""
			+ " name=\"GetResourcePropertyRequest\" message=\"dyn:GetResourcePropertyRequest\" />"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/GetResourceProperty/GetResourcePropertyResponse\""
			+ " name=\"GetResourcePropertyResponse\" message=\"dyn:GetResourcePropertyResponse\" />"
			+ "<wsdl:fault name=\"ResourceUnknownFault\" message=\"dyn:ResourceUnknownFault\"/>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\" message=\"dyn:ResourceUnavailableFault\"/>"
			+ "<wsdl:fault name=\"InvalidResourcePropertyQNameFault\" message=\"dyn:InvalidResourcePropertyQNameFault\" />"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"GetMultipleResourceProperties\">"
			+ "<wsdl:input wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/GetMultipleResourceProperties/GetMultipleResourcePropertiesRequest\""
			+ " name=\"GetMultipleResourcePropertiesRequest\" message=\"dyn:GetMultipleResourcePropertiesRequest\" />"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/GetMultipleResourceProperties/GetMultipleResourcePropertiesResponse\""
			+ " name=\"GetMultipleResourcePropertiesResponse\" message=\"dyn:GetMultipleResourcePropertiesResponse\" />"
			+ "<wsdl:fault name=\"ResourceUnknownFault\" message=\"dyn:ResourceUnknownFault\"/>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\" message=\"dyn:ResourceUnavailableFault\"/>"
			+ "<wsdl:fault name=\"InvalidResourcePropertyQNameFault\" message=\"dyn:InvalidResourcePropertyQNameFault\" />"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"QueryResourceProperties\">"
			+ "<wsdl:input wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/QueryResourceProperties/QueryResourcePropertiesRequest\""
			+ " name=\"QueryResourcePropertiesRequest\" message=\"dyn:QueryResourcePropertiesRequest\" />"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/QueryResourceProperties/QueryResourcePropertiesResponse\""
			+ " name=\"QueryResourcePropertiesResponse\" message=\"dyn:QueryResourcePropertiesResponse\" />"
			+ "<wsdl:fault name=\"ResourceUnknownFault\" message=\"dyn:ResourceUnknownFault\"/>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\" message=\"dyn:ResourceUnavailableFault\"/>"
			+ "<wsdl:fault name=\"UnknownQueryExpressionDialectFault\" message=\"dyn:UnknownQueryExpressionDialectFault\"/>"
			+ "<wsdl:fault name=\"InvalidQueryExpressionFault\" message=\"dyn:InvalidQueryExpressionFault\"/>"
			+ "<wsdl:fault name=\"QueryEvaluationErrorFault\" message=\"dyn:QueryEvaluationErrorFault\" />"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"SetResourceProperties\">"
			+ "<wsdl:input  wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/SetResourceProperties/SetResourcePropertiesRequest\""
			+ " name=\"SetResourcePropertiesRequest\" message=\"dyn:SetResourcePropertiesRequest\" />"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsrf/rpw-2/SetResourceProperties/SetResourcePropertiesResponse\""
			+ " name=\"SetResourcePropertiesResponse\" message=\"dyn:SetResourcePropertiesResponse\" />"
			+ "<wsdl:fault name=\"ResourceUnknownFault\" message=\"dyn:ResourceUnknownFault\"/>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\" message=\"dyn:ResourceUnavailableFault\"/>"
			+ "<wsdl:fault name=\"InvalidModificationFault\" message=\"dyn:InvalidModificationFault\" />"
			+ "<wsdl:fault name=\"UnableToModifyResourcePropertyFault\" message=\"dyn:UnableToModifyResourcePropertyFault\" />"
			+ "<wsdl:fault name=\"InvalidResourcePropertyQNameFault\" message=\"dyn:InvalidResourcePropertyQNameFault\" />"
			+ "<wsdl:fault name=\"SetResourcePropertyRequestFailedFault\" message=\"dyn:SetResourcePropertyRequestFailedFault\" />"
			+ "</wsdl:operation>";

	private static final String WSDL_WSRF_RESOURCE_LIFETIME_OPERATIONS = "<wsdl:operation name=\"Destroy\">"
			+ "<wsdl:input wsa:Action=\"http://docs.oasis-open.org/wsrf/rlw-2/ImmediateResourceTermination/DestroyRequest\""
			+ " name=\"DestroyRequest\" message=\"dyn:DestroyRequest\" />"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsrf/rlw-2/ImmediateResourceTermination/DestroyResponse\""
			+ " name=\"DestroyResponse\" message=\"dyn:DestroyResponse\" />"
			+ "<wsdl:fault name=\"ResourceNotDestroyedFault\" message=\"dyn:ResourceNotDestroyedFault\" />"
			+ "<wsdl:fault name=\"ResourceUnknownFault\" message=\"dyn:ResourceUnknownFault\" />"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\" message=\"dyn:ResourceUnavailableFault\"/>"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"SetTerminationTime\">"
			+ "<wsdl:input wsa:Action=\"http://docs.oasis-open.org/wsrf/rlw-2/ScheduledResourceTermination/SetTerminationTimeRequest\""
			+ " name=\"SetTerminationTimeRequest\" message=\"dyn:SetTerminationTimeRequest\" />"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsrf/rlw-2/ScheduledResourceTermination/SetTerminationTimeResponse\""
			+ " name=\"SetTerminationTimeResponse\" message=\"dyn:SetTerminationTimeResponse\" />"
			+ "<wsdl:fault name=\"UnableToSetTerminationTimeFault\" message=\"dyn:UnableToSetTerminationTimeFault\" />"
			+ "<wsdl:fault name=\"ResourceUnknownFault\" message=\"dyn:ResourceUnknownFault\" />"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\" message=\"dyn:ResourceUnavailableFault\"/>"
			+ "<wsdl:fault name=\"TerminationTimeChangeRejectedFault\" message=\"dyn:TerminationTimeChangeRejectedFault\" />"
			+ "</wsdl:operation>";

	private static final String WSDL_WSDM_RELATIONSHIP_OPERATIONS = "<wsdl:operation name=\"QueryRelationshipsByType\">"
			+ "<wsdl:input wsa:Action=\"http://docs.oasis-open.org/wsdm/muws/capabilities/Relationships/QueryRelationshipsByType\""
			+ " name=\"QueryRelationshipsByTypeRequest\" message=\"dyn:QueryRelationshipsByTypeRequest\"/>"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsdm/muws2-2.xsd/QueryRelationshipsByTypeResponse\""
			+ " name=\"QueryRelationshipsByTypeResponse\" message=\"dyn:QueryRelationshipsByTypeResponse\"/>"
			+ "</wsdl:operation>";

	private static final String WSDL_WSN_OPERATIONS = "<wsdl:operation name=\"Subscribe\">"
			+ "<wsdl:input  wsa:Action=\"http://docs.oasis-open.org/wsn/bw-2/NotificationProducer/SubscribeRequest\""
			+ " message=\"wsntw:SubscribeRequest\" />"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsn/bw-2/NotificationProducer/SubscribeResponse\""
			+ " message=\"wsntw:SubscribeResponse\" />"
			+ "<wsdl:fault  name=\"ResourceUnknownFault\" "
			+ " message=\"dyn:ResourceUnknownFault\" />"
			+ "<wsdl:fault  name=\"InvalidFilterFault\" "
			+ " message=\"wsntw:InvalidFilterFault\"/>"
			+ "<wsdl:fault  name=\"TopicExpressionDialectUnknownFault\" "
			+ " message=\"wsntw:TopicExpressionDialectUnknownFault\"/>"
			+ "<wsdl:fault  name=\"InvalidTopicExpressionFault\" "
			+ " message=\"wsntw:InvalidTopicExpressionFault\" />"
			+ "<wsdl:fault  name=\"TopicNotSupportedFault\" "
			+ " message=\"wsntw:TopicNotSupportedFault\" />"
			+ "<wsdl:fault  name=\"InvalidProducerPropertiesExpressionFault\" "
			+ " message=\"wsntw:InvalidProducerPropertiesExpressionFault\"/>"
			+ "<wsdl:fault  name=\"InvalidMessageContentExpressionFault\" "
			+ " message=\"wsntw:InvalidMessageContentExpressionFault\"/>"
			+ "<wsdl:fault  name=\"UnacceptableInitialTerminationTimeFault\" "
			+ " message=\"wsntw:UnacceptableInitialTerminationTimeFault\"/>"
			+ "<wsdl:fault  name=\"UnrecognizedPolicyRequestFault\" "
			+ " message=\"wsntw:UnrecognizedPolicyRequestFault\"/>"
			+ "<wsdl:fault  name=\"UnsupportedPolicyRequestFault\" "
			+ " message=\"wsntw:UnsupportedPolicyRequestFault\"/>"
			+ "<wsdl:fault  name=\"NotifyMessageNotSupportedFault\" "
			+ " message=\"wsntw:NotifyMessageNotSupportedFault\"/>"
			+ "<wsdl:fault  name=\"SubscribeCreationFailedFault\" "
			+ " message=\"wsntw:SubscribeCreationFailedFault\"/>"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"GetCurrentMessage\">"
			+ "<wsdl:input  wsa:Action=\"http://docs.oasis-open.org/wsn/bw-2/NotificationProducer/GetCurrentMessageRequest\""
			+ " message=\"wsntw:GetCurrentMessageRequest\"/>"
			+ "<wsdl:output wsa:Action=\"http://docs.oasis-open.org/wsn/bw-2/NotificationProducer/GetCurrentMessageResponse\""
			+ " message=\"wsntw:GetCurrentMessageResponse\"/>"
			+ "<wsdl:fault  name=\"ResourceUnknownFault\" "
			+ " message=\"dyn:ResourceUnknownFault\" />"
			+ "<wsdl:fault  name=\"TopicExpressionDialectUnknownFault\" "
			+ " message=\"wsntw:TopicExpressionDialectUnknownFault\"/>"
			+ "<wsdl:fault  name=\"InvalidTopicExpressionFault\" "
			+ " message=\"wsntw:InvalidTopicExpressionFault\" />"
			+ "<wsdl:fault  name=\"TopicNotSupportedFault\" "
			+ " message=\"wsntw:TopicNotSupportedFault\" />"
			+ "<wsdl:fault  name=\"NoCurrentMessageOnTopicFault\" "
			+ " message=\"wsntw:NoCurrentMessageOnTopicFault\" />"
			+ "<wsdl:fault  name=\"MultipleTopicsSpecifiedFault\" "
			+ " message=\"wsntw:MultipleTopicsSpecifiedFault\" />"
			+ "</wsdl:operation>";

	private static final String WSDL_WSN_CONSUMER_OPERATIONS = "<wsdl:operation name=\"Notify\">"
			+ "<wsdl:input wsa:Action=\"http://docs.oasis-open.org/wsn/bw-2/NotificationConsumer/NotifyRequest\" name=\"notify\" message=\"wsntw:Notify\" />"
			+ "</wsdl:operation>";

	private static final String WSDL_PORT_TYPE_STANDARD_CUSTOM_FAULTS = "<wsdl:fault name=\"ResourceUnknownFault\" message=\"dyn:ResourceUnknownFault\"/>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\" message=\"dyn:ResourceUnavailableFault\"/>";

	private static final String WSDL_PORT_TYPE_END = "</wsdl:portType>\n";
	private static final String WSDL_BINDING_START = "<wsdl:binding name=\"";

	// WsResourceBinding
	private static final String WSDL_BINDING_NAMESPACE = "Binding\" type=\"dyn:";
	// WsResource
	private static final String WSDL_BINDING_OPERATIONS_START = "PortType\">"
			+ " <wsdl-soap:binding style=\"document\" transport=\"http://schemas.xmlsoap.org/soap/http\" />";

	private static final String WSDL_BINDING_MEX_OPERATIONS = "<wsdl:operation name=\"GetMetadata\">"
			+ "<wsdl-soap:operation soapAction=\"GetMetadata\" />"
			+ "<wsdl:input>"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:input>"
			+ "<wsdl:output>"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:output>" + "</wsdl:operation>";
	private static final String WSDL_BINDING_WSRF_RESOURCE_LIFETIME_OPERATIONS = "<wsdl:operation name=\"Destroy\">"
			+ "<wsdl-soap:operation soapAction=\"Destroy\"/>"
			+ "<wsdl:input name=\"DestroyRequest\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:input>"
			+ "<wsdl:output name=\"DestroyResponse\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:output>"
			+ "<wsdl:fault name=\"ResourceNotDestroyedFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceNotDestroyedFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnknownFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnavailableFault\"/>"
			+ "</wsdl:fault>"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"SetTerminationTime\">"
			+ "<wsdl-soap:operation soapAction=\"SetTerminationTime\"/>"
			+ "<wsdl:input name=\"SetTerminationTimeRequest\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:input><wsdl:output name=\"SetTerminationTimeResponse\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:output>"
			+ "<wsdl:fault name=\"UnableToSetTerminationTimeFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"UnableToSetTerminationTimeFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnknownFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnavailableFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"TerminationTimeChangeRejectedFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"TerminationTimeChangeRejectedFault\"/>"
			+ "</wsdl:fault>" + "</wsdl:operation>";
	public static final String WSDL_BINDING_WSRF_RESOURCE_PROPERTY_OPERATIONS = "<wsdl:operation name=\"GetResourcePropertyDocument\">"
			+ "<wsdl-soap:operation soapAction=\"GetResourcePropertyDocument\"/>"
			+ "<wsdl:input name=\"GetResourcePropertyDocumentRequest\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:input><wsdl:output name=\"GetResourcePropertyDocumentResponse\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:output>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnknownFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnavailableFault\"/>"
			+ "</wsdl:fault>"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"GetResourceProperty\">"
			+ "<wsdl-soap:operation soapAction=\"GetResourceProperty\"/>"
			+ "<wsdl:input name=\"GetResourcePropertyRequest\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:input><wsdl:output name=\"GetResourcePropertyResponse\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:output>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnknownFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnavailableFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"InvalidResourcePropertyQNameFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"InvalidResourcePropertyQNameFault\"/>"
			+ "</wsdl:fault>"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"GetMultipleResourceProperties\">"
			+ "<wsdl-soap:operation soapAction=\"GetMultipleResourceProperties\"/>"
			+ "<wsdl:input name=\"GetMultipleResourcePropertiesRequest\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:input><wsdl:output name=\"GetMultipleResourcePropertiesResponse\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:output>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnknownFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnavailableFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"InvalidResourcePropertyQNameFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"InvalidResourcePropertyQNameFault\"/>"
			+ "</wsdl:fault>"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"QueryResourceProperties\">"
			+ "<wsdl-soap:operation soapAction=\"QueryResourceProperties\"/>"
			+ "<wsdl:input name=\"QueryResourcePropertiesRequest\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:input><wsdl:output name=\"QueryResourcePropertiesResponse\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:output>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnknownFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnavailableFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"UnknownQueryExpressionDialectFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"UnknownQueryExpressionDialectFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"InvalidQueryExpressionFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"InvalidQueryExpressionFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"QueryEvaluationErrorFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"QueryEvaluationErrorFault\"/>"
			+ "</wsdl:fault>"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"SetResourceProperties\">"
			+ "<wsdl-soap:operation soapAction=\"http://oasis.org/SetResourceProperties\"/>"
			+ "<wsdl:input name=\"SetResourcePropertiesRequest\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:input><wsdl:output name=\"SetResourcePropertiesResponse\">"
			+ "<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />"
			+ "</wsdl:output>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnknownFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnavailableFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"InvalidModificationFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"InvalidModificationFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"UnableToModifyResourcePropertyFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"UnableToModifyResourcePropertyFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"InvalidResourcePropertyQNameFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"InvalidResourcePropertyQNameFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"SetResourcePropertyRequestFailedFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"SetResourcePropertyRequestFailedFault\"/>"
			+ "</wsdl:fault>" + "</wsdl:operation>";

	private static final String WSDL_BINDING_WSDM_RELATIONSHIP_OPERATIONS = "<wsdl:operation name=\"QueryRelationshipsByType\">"
			+ "<wsdl-soap:operation style=\"document\"/>"
			+ "<wsdl:input><wsdl-soap:body use=\"literal\"/></wsdl:input>"
			+ "<wsdl:output><wsdl-soap:body use=\"literal\"/></wsdl:output>"
			+ "</wsdl:operation>";

	private static final String WSDL_BINDING_WSN_OPERATIONS = "<wsdl:operation name=\"Subscribe\">"
			+ "<wsdl-soap:operation"
			+ " soapAction=\"%TARGET_NAMESPACE%/Subscribe\" />"
			+ "<wsdl:input><wsdl-soap:body use=\"encoded\" /></wsdl:input>"
			+ "<wsdl:output><wsdl-soap:body use=\"encoded\" /></wsdl:output>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"ResourceUnknownFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"InvalidFilterFault\">"
			+ "<wsdl-soap:fault namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"InvalidFilterFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"TopicExpressionDialectUnknownFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"TopicExpressionDialectUnknownFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"InvalidTopicExpressionFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"InvalidTopicExpressionFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"TopicNotSupportedFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"TopicNotSupportedFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault"
			+ " name=\"InvalidProducerPropertiesExpressionFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"InvalidProducerPropertiesExpressionFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"InvalidMessageContentExpressionFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"InvalidMessageContentExpressionFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault"
			+ " name=\"UnacceptableInitialTerminationTimeFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"UnacceptableInitialTerminationTimeFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"UnrecognizedPolicyRequestFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"UnrecognizedPolicyRequestFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"UnsupportedPolicyRequestFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"UnsupportedPolicyRequestFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"NotifyMessageNotSupportedFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"NotifyMessageNotSupportedFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"SubscribeCreationFailedFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"SubscribeCreationFailedFault\" />"
			+ "</wsdl:fault>"
			+ "</wsdl:operation>"
			+ "<wsdl:operation name=\"GetCurrentMessage\">"
			+ "<wsdl-soap:operation"
			+ " soapAction=\"%TARGET_NAMESPACE%/GetCurrentMessage\"/>"
			+ "<wsdl:input><wsdl-soap:body use=\"encoded\" /></wsdl:input>"
			+ "<wsdl:output><wsdl-soap:body use=\"encoded\" /></wsdl:output>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"ResourceUnknownFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"TopicExpressionDialectUnknownFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"TopicExpressionDialectUnknownFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"InvalidTopicExpressionFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"InvalidTopicExpressionFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"TopicNotSupportedFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"TopicNotSupportedFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"NoCurrentMessageOnTopicFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"NoCurrentMessageOnTopicFault\" />"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"MultipleTopicsSpecifiedFault\">"
			+ "<wsdl-soap:fault"
			+ " namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"MultipleTopicsSpecifiedFault\" />"
			+ "</wsdl:fault>" + "</wsdl:operation>";

	private static final String WSDL_BINDING_WSN_CONSUMER_OPERATIONS = "<wsdl:operation name=\"Notify\">"
			+ "<wsdl-soap:operation"
			+ " soapAction=\"%TARGET_NAMESPACE%/NotifyRequest\" />"
			+ "<wsdl:input><wsdl-soap:body use=\"encoded\" /></wsdl:input>"
			+ "<wsdl:output><wsdl-soap:body use=\"encoded\" /></wsdl:output>"
			+ "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault namespace=\"%TARGET_NAMESPACE%\""
			+ " use=\"encoded\" name=\"ResourceUnknownFault\" />"
			+ "</wsdl:fault>" + "</wsdl:operation>";

	private static final String WSDL_BINDING_STANDARD_CUSTOM_FAULTS = "<wsdl:fault name=\"ResourceUnknownFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnknownFault\"/>"
			+ "</wsdl:fault>"
			+ "<wsdl:fault name=\"ResourceUnavailableFault\">"
			+ "<wsdl-soap:fault use=\"encoded\" name=\"ResourceUnavailableFault\"/>"
			+ "</wsdl:fault>";

	private static final String WSDL_BINDING_END = "</wsdl:binding>\n";

	/*
	 * Boiler-plate namespaces
	 * 
	 */

	/*
	 * Namespace registry
	 * 
	 * Schema includes
	 * 
	 * Inline schema
	 * 
	 * Messages in,out,fault
	 * 
	 * port types operations
	 * 
	 * binding??? service???
	 * 
	 */

	private ManagedResourceInfo info;
	private WSDMInfo wsdmInfo;
	private Object contribution; 
	private HashMap<String, Method> dialectMap = new HashMap<String, Method>();

	public WSDLSource(WSDMInfo wsdmInfo, Object contribution) {
		this.wsdmInfo = wsdmInfo;
		this.info = wsdmInfo.getInfo();
		this.contribution = contribution;
		// TODO populate the dialectMap
		ManagedOperationInfo[] operations = (ManagedOperationInfo[]) info
				.getOperations();
		for (ManagedOperationInfo operation : operations) {
			ManagedOperation managedOperation = (ManagedOperation) operation
					.getOperationMethod().getAnnotation(ManagedOperation.class);
			if (managedOperation != null && managedOperation.isMetaDataMethod()) {
				String key = managedOperation.MetaDataType();
				// make sure it's not WSDL!!!
				if (WsdlUtils.NAMESPACE_URI.equals(key)) {
					// TODO should throw exception - or at least skip this
					// method!!!
				} else {
					dialectMap.put(key, operation.getOperationMethod());
				}
			}
		}

	}

	void addDialectHandler(String dialect, Method method) {
		this.dialectMap.put(dialect, method);
	}

	public Element[] getMetadata(String dialect) {
		//
		// the resource factory has all its metadata/deployment docs
		//
		Element metadata = null;

		//
		// missing dialect has no defined fault in ws-mex
		//
		if (dialect == null)
			return super.getMetadata(dialect);

		//
		// get WSDL - this is fully resolved (very large!)
		//
		else if (dialect.equals(WsxConstants.WSDL_DIALECT))
			metadata = getWSDL();
		else if (dialectMap.get(dialect) != null) {
			Method metaMethod = dialectMap.get(dialect);
			// TODO - no provision for exceptions here...
			try {
				metadata = (Element) metaMethod.invoke(((WSDMBinding) this
						.getResource()).getResource(), null);
			} catch (IllegalArgumentException e) {
			} catch (IllegalAccessException e) {
			} catch (InvocationTargetException e) {
			}
		}

		//
		// dialect provided, but not type matches it
		//
		if (metadata == null)
			return super.getMetadata(dialect);

		return new Element[] { metadata };
	}

	protected Element getWSDL() {
		Document doc;
		try {
			doc = getWSDLDocument();
			if (doc != null)
				return doc.getDocumentElement();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;
	}

	public Document getWSDLDocument() {

		StringBuffer wsdl = new StringBuffer(WSDL_START);
		wsdl.append(info.getTargetNamespace());
		wsdl.append("\" ");
		for (String namespace : wsdmInfo.getNamespaces()) {
			wsdl.append(" xmlns:" + wsdmInfo.getPrefixForNamespace(namespace)
					+ "=\"");
			wsdl.append(namespace);
			wsdl.append("\" ");
		}
		wsdl.append(WSDL_DEFAULT_NAMESPACES);

		wsdl.append(" xmlns:wsrf-rp=\"http://docs.oasis-open.org/wsrf/rp-2\"");
		
		if (info.getNotifications().length > 0 || info.isAdvertised()) {
			wsdl.append(WSDL_NOTIFICATION_NAMESPACES);
		}
		// if(info.hasMetrics()){
		// wsdl.append(WSDL_RMD_NAMESPACE);
		// }
		wsdl.append(" name=\"");
		wsdl.append(info.getResourceName());
		wsdl.append("\">");

		if (info.getNotifications().length > 0 || info.isAdvertised()) {
			wsdl.append(WSDL_NOTIFICATION_WSDL_IMPORT);
		}
		
		//Policy work
		if(contribution instanceof PolicyAssertionProvider){
			Collection<PolicyAssertion> assertions = ((PolicyAssertionProvider)contribution).getPolicyAssertions();
			processAssertions(wsdl, assertions);
		}
		if(contribution instanceof ManagementProxy){
			String uris[] = wsdmInfo.getCapabilityURIs();
			for(String uri : uris){
				Object cap = ((ManagementProxy)contribution).getCapability(uri);
				if(cap != null){
					System.out.println("Object for " + uri);
				}
				if(cap != null && cap instanceof PolicyAssertionProvider){
					Collection<PolicyAssertion> assertions = ((PolicyAssertionProvider)cap).getPolicyAssertions();
					processAssertions(wsdl, assertions);
				}
			}
		}
		
		wsdl.append(WSDL_TYPE_START);
		wsdl.append(WSDL_WSRF_TYPES);
		wsdl.append(WSDL_MUWS1_TYPES);

		if (info.getNotifications().length > 0 || info.isAdvertised()
				|| info.hasEventConsumers()) {
			wsdl.append(WSDL_WSN_TYPES);
		}
		// Specified includes
		String includedSchema[] = wsdmInfo.getSchemaImports();
		for (String include : includedSchema) {
			wsdl.append("<xsd:schema" + " elementFormDefault=\"qualified\""
					+ " targetNamespace=\"" + include + "\">"
					+ "<xsd:include schemaLocation=\""
					+ wsdmInfo.getSchemaLocation(include) + "\" />"
					+ "</xsd:schema>");
		}

		// Custom types...
		for (String namespace : info.getNamespaces()) {
			MBeanAttributeInfo[] attributes = info.getAttributes();
			wsdl.append(WSDL_RESOURCE_SCHEMA_START);
			wsdl.append(namespace);
			wsdl.append("\">");

			HashMap<String, SchemaHelper> customTypes = new HashMap<String, SchemaHelper>();

			for (MBeanAttributeInfo attribute : attributes) {
				ManagedAttributeInfo attr_info = (ManagedAttributeInfo) attribute;
				// Resource ID already mapped
				if (attr_info.isResourceID())
					continue;
				String ns = attr_info.getNamespace();
				if ("".equals(ns))
					ns = attr_info.getOwnerNamespace();
				if ("".equals(ns))
					ns = info.getTargetNamespace();
				if (!namespace.equals(ns))
					continue;
				wsdl.append(" <xsd:element name=\"");
				wsdl.append(attr_info.getName());
				wsdl.append("\" type=\"");
				String xsdType = java2xsdType.get(attr_info.getType());
				if (xsdType == null) {
					SchemaHelper helper = WSDMBindingUtil
							.getSchemaHelper(attr_info.getType());
					if (helper != null) {
						xsdType = helper.getXsdName(attr_info.getType());
						customTypes.put(attr_info.getType(), helper);
					}
				}
				wsdl.append(xsdType);
				if (attr_info.getType().startsWith("[L")) {
					wsdl.append("\" minoccurs=\"0\" maxoccurs=\"unbounded");
				}
				wsdl.append("\"/>");
			}

			ManagedOperationInfo[] operations = (ManagedOperationInfo[]) info
					.getOperations();
			for (ManagedOperationInfo operInfo : operations) {
				String ns = operInfo.getNamespace();
				if ("".equals(ns))
					ns = info.getTargetNamespace();
				if (!namespace.equals(ns))
					continue;
				
				WSDMOperation wsdmOper = operInfo.getAnnotation(WSDMOperation.class);
				String requestName = operInfo.getName() + "RequestType";
				String responseName = operInfo.getName() + "ResponseType";
				if (wsdmOper != null) {
					if (!"".equals(wsdmOper.inputName()))
						requestName = wsdmOper.inputName()+"Type";
					if (!"".equals(wsdmOper.outputName()))
						responseName = wsdmOper.outputName()+"Type";
				}
				
				MBeanParameterInfo[] parms = operInfo.getSignature();
				if (parms.length > 0) {
					
					wsdl.append("<xsd:element name=\"");
					wsdl.append(requestName);
					wsdl.append("\"><xsd:complexType><xsd:sequence>");
					int i = 1;
					for (MBeanParameterInfo parm : parms) {
						Annotation[] annotations = operInfo
								.getAnnotationsForParameter(parm);
						String xsdType = java2xsdType.get(parm.getType());
						if (xsdType == null) {
							SchemaHelper helper = WSDMBindingUtil
									.getSchemaHelper(parm.getType());
							if (helper != null) {
								xsdType = helper.getXsdName(operInfo
										.getReturnType());
								customTypes.put(parm.getType(), helper);
							}
							if (xsdType == null) {
								for (Annotation annotation : annotations) {
									if (annotation instanceof WSDMParameterType) {
										WSDMParameterType parmInfo = (WSDMParameterType) annotation;
										wsdmInfo
												.addNamespace(parmInfo.schema());
										parmInfo.schema();
										parmInfo.type();
										xsdType = wsdmInfo
												.getPrefixForNamespace(parmInfo
														.schema())
												+ ":" + parmInfo.type();
										break;
									}
								}
							}
						}
						String displayName = "parm" + i++;
						for (Annotation annotation : annotations) {
							if (annotation instanceof ManagedParameter) {
								ManagedParameter parmInfo = (ManagedParameter) annotation;
								if (!"".equals(parmInfo.displayName())) {
									displayName = parmInfo.displayName();
								}
							}
						}
						wsdl.append("<xsd:element name=\"");
						wsdl.append(displayName);
						wsdl.append("\" type=\"");
						wsdl.append(xsdType);
						if (parm.getType().startsWith("[L")) {
							wsdl
									.append("\" minoccurs=\"0\" maxoccurs=\"unbounded");
						}
						wsdl.append("\"/>");
					}
					wsdl
							.append("</xsd:sequence></xsd:complexType></xsd:element>");
				} else {
					wsdl.append("<xsd:element name=\"");
					wsdl.append(requestName);
					wsdl.append("\"><xsd:complexType/></xsd:element>");
				}
				if (!"void".equals(operInfo.getReturnType())) {
					wsdl.append("<xsd:element name=\"");
					wsdl.append(responseName);
					// wsdl.append("\"><xsd:complexType><xsd:simpleContent>");
					wsdl.append("\" type=\"");
					String xsdType = java2xsdType.get(operInfo.getReturnType());
					if (xsdType == null) {
						SchemaHelper helper = WSDMBindingUtil
								.getSchemaHelper(operInfo.getReturnType());
						if (helper != null) {
							xsdType = helper.getXsdName(operInfo
									.getReturnType());
							customTypes.put(operInfo.getReturnType(), helper);
						}
					}
					wsdl.append(xsdType);
					if (operInfo.getReturnType().startsWith("[L")) {
						wsdl.append("\" minoccurs=\"0\" maxoccurs=\"unbounded");
					}
					wsdl.append("\"/>");
					// wsdl.append("</xsd:simpleContent></xsd:complexType></xsd:element>");
				} else {
					wsdl.append("<xsd:element name=\"");
					wsdl.append(responseName);
					// wsdl.append("\"><xsd:complexType><xsd:simpleContent>");
					wsdl.append("\"/>");
				}
			}

			if (namespace.equals(info.getTargetNamespace())) {
				wsdl.append(WSDL_RESOURCE_PROPERTIES_START);
				wsdl.append(WSDL_MUWS1_PROPERTIES);
				wsdl.append(WSDL_WSRF_RESOURCE_QUERY_PROPERTIES);
				// if(info.resourceLifetimeSupport()){
				// wsdl.append(WSDL_WSRF_RESOURCE_LIFETIME_PROPERTIES);
				// }
				if (info.getRelations().length > 0) {
					wsdl.append(WSDL_WSDM_RELATIONSHIPS_PROPERTIES);
				}
				if (info.getNotifications().length > 0 || info.isAdvertised()) {
					wsdl.append(WSDL_WSN_PROPERTIES);
				}
				 if(info.hasMetrics()){
					 wsdl.append(WSDL_METRIC_PROPERTIES);
				 }

				for (MBeanAttributeInfo attribute : info.getAttributes()) {
					ManagedAttributeInfo attr_info = (ManagedAttributeInfo) attribute;
					// Skip resourceID - already mapped
					String ns = attr_info.getNamespace();
					if ("".equals(ns))
						ns = attr_info.getOwnerNamespace();
					if ("".equals(ns))
						ns = info.getTargetNamespace();
					if (attr_info.isResourceID())
						continue;
					wsdl.append(" <xsd:element ref=\""
							+ info.getPrefixForNamespace(ns) + ":");
					wsdl.append(attr_info.getName());
					wsdl.append("\"/>");
				}

				wsdl.append(WSDL_RESOURCE_PROPERTIES_END);
			}

			if (!customTypes.isEmpty()) {
				for (String type : customTypes.keySet()) {
					SchemaHelper helper = customTypes.get(type);
					if (helper != null)
						wsdl.append(helper.getXsdType(type));
				}
			}

			wsdl.append(WSDL_RESOURCE_SCHEMA_END);
		}

		wsdl.append(WSDL_TYPE_END);

		 wsdl.append(WSDL_MEX_MESSAGES);
		 wsdl.append(WSDL_WSRF_RESOURCE_PROPERTY_MESSAGES);
		 wsdl.append(WSDL_WSRF_RESOURCE_FAULT_MESSAGES);

		// if(info.resourceLifetimeSupport()){
		// wsdl.append(WSDL_WSRF_RESOURCE_LIFETIME_MESSAGES);
		// }
		//		
		if (info.getRelations().length > 0) {
			wsdl.append(this.WSDL_WSDM_RELATIONSHIP_MESSAGES);
		}

		// Do custom messages...
		ManagedOperationInfo[] operations = (ManagedOperationInfo[]) info
				.getOperations();
		for (ManagedOperationInfo operInfo : operations) {
			// JOEL no need to make these part of the WSDL
			if (operInfo.isEventConsumer())
				continue;
			MBeanParameterInfo[] parms = operInfo.getSignature();
			String requestName = operInfo.getName() + "Request";
			String responseName = operInfo.getName() + "Response";
			String inputPart = requestName;
			String outputPart = responseName;
			WSDMOperation wsdmOper = operInfo
					.getAnnotation(WSDMOperation.class);
			if (wsdmOper != null) {
				if (!"".equals(wsdmOper.inputMessage()))
					requestName = wsdmOper.inputName();
				if (!"".equals(wsdmOper.outputMessage()))
					responseName = wsdmOper.outputMessage();
				if (!"".equals(wsdmOper.inputPart()))
					inputPart = wsdmOper.inputMessage();
				if (!"".equals(wsdmOper.outputPart()))
					outputPart = wsdmOper.inputMessage();
				if (!"".equals(wsdmOper.inputName()))
					requestName = wsdmOper.inputName();
				if (!"".equals(wsdmOper.outputName()))
					responseName = wsdmOper.outputName();
			}
			
			
			// if(parms.length > 0){
			wsdl.append("<wsdl:message name=\"");
			wsdl.append(requestName);
			wsdl.append("\"><wsdl:part name=\"");
			wsdl.append(inputPart);
			wsdl
					.append("\" element=\""
							+ info.getPrefixForNamespace(operInfo
									.getNamespace()) + ":");
			wsdl.append(requestName);
			wsdl.append("\"/></wsdl:message>");
			// }
			// if(!"void".equals(operInfo.getReturnType())){
			wsdl.append("<wsdl:message name=\"");
			wsdl.append(responseName);
			wsdl.append("\"><wsdl:part name=\"");
			wsdl.append(outputPart);
			wsdl
					.append("\" element=\""
							+ info.getPrefixForNamespace(operInfo
									.getNamespace()) + ":");
			wsdl.append(responseName);
			wsdl.append("\"/></wsdl:message>");
			// }
		}

		wsdl.append(WSDL_PORT_TYPE_START);
		// if(info.hasMetrics()){
		// wsdl.append(" wsrmd:Descriptor=\"");
		// wsdl.append(info.getResourceName());
		// wsdl.append("MetaData\" wsrmd:DescriptorLocation=\"");
		// wsdl.append(info.getResourceName());
		// wsdl.append(".rmd\"");
		// }
		// We'll always have resourceid...

		wsdl.append(" wsrf-rp:ResourceProperties=\"dyn:WsResourceProperties\"");

		wsdl.append(" name=\"");
		wsdl.append(info.getResourceName());
		wsdl.append("PortType\"");
		wsdl.append(">");
		
		wsdl.append(this.WSDL_MEX_OPERATIONS);
		
		 wsdl.append(WSDL_WSRF_RESOURCE_PROPERTY_OPERATIONS);
		
		// if(info.resourceLifetimeSupport()){
		// wsdl.append(WSDL_WSRF_RESOURCE_LIFETIME_OPERATIONS);
		// }
		if (info.getRelations().length > 0) {
			wsdl.append(WSDL_WSDM_RELATIONSHIP_OPERATIONS);
		}
		if (info.getNotifications().length > 0 || info.isAdvertised()) {
			wsdl.append(WSDL_WSN_OPERATIONS);
		}
		if (info.hasEventConsumers()) {
			wsdl.append(WSDL_WSN_CONSUMER_OPERATIONS);
		}
		// take care of custom operations
		operations = (ManagedOperationInfo[]) info.getOperations();
		for (ManagedOperationInfo operInfo : operations) {
			// JOEL skip event consumer methods
			if (operInfo.isEventConsumer())
				continue;

			String operName = operInfo.getName();
			String inputAction = operInfo.getNamespace() + "/"
					+ operInfo.getName();
			String inputName = operInfo.getName() + "Request";
			String inputMessage = operInfo.getName() + "Request";
			String outputAction = operInfo.getNamespace() + "/"
					+ operInfo.getName();
			String outputName = operInfo.getName() + "Request";
			String outputMessage = operInfo.getName() + "Request";

			WSDMOperation wsdmOper = operInfo
					.getAnnotation(WSDMOperation.class);
			if (wsdmOper != null) {
				if (!"".equals(wsdmOper.inputAction()))
					inputAction = wsdmOper.inputAction();
				if (!"".equals(wsdmOper.inputName()))
					inputName = wsdmOper.inputName();
				if (!"".equals(wsdmOper.inputMessage()))
					inputMessage = wsdmOper.inputMessage();
				if (!"".equals(wsdmOper.outputAction()))
					outputAction = wsdmOper.outputAction();
				if (!"".equals(wsdmOper.outputName()))
					outputName = wsdmOper.outputName();
				if (!"".equals(wsdmOper.outputMessage()))
					outputMessage = wsdmOper.outputMessage();
			}

			wsdl.append("<wsdl:operation name=\"");
			wsdl.append(operName);
			wsdl.append("\">");

			wsdl.append("<wsdl:input name=\"");
			wsdl.append(inputName);
			wsdl.append("\" wsa:Action=\"");
			wsdl.append(inputAction);
			wsdl.append("\" message=\"dyn:");
			wsdl.append(inputMessage);
			wsdl.append("\"/>");

			wsdl.append("<wsdl:output name=\"");
			wsdl.append(outputName);
			wsdl.append("\" wsa:Action=\"");
			wsdl.append(outputAction);
			wsdl.append("\" message=\"dyn:");
			wsdl.append(outputMessage);
			wsdl.append("\"/>");

			wsdl.append(WSDL_PORT_TYPE_STANDARD_CUSTOM_FAULTS);
			wsdl.append("</wsdl:operation>");
		}

		wsdl.append(WSDL_PORT_TYPE_END);

		// take care of binding
		wsdl.append(WSDL_BINDING_START);
		wsdl.append(info.getResourceName());
		wsdl.append(WSDL_BINDING_NAMESPACE);
		wsdl.append(info.getResourceName());
		wsdl.append(WSDL_BINDING_OPERATIONS_START);
		wsdl.append(WSDL_BINDING_MEX_OPERATIONS);
		
		 wsdl.append(WSDL_BINDING_WSRF_RESOURCE_PROPERTY_OPERATIONS);
		// if(info.resourceLifetimeSupport()){
		// wsdl.append(WSDL_BINDING_WSRF_RESOURCE_LIFETIME_OPERATIONS);
		// }
		if (info.getRelations().length > 0) {
			wsdl.append(WSDL_BINDING_WSDM_RELATIONSHIP_OPERATIONS);
		}
		if (info.getNotifications().length > 0 || info.isAdvertised()) {
			wsdl.append(WSDL_BINDING_WSN_OPERATIONS.replaceAll(
					"%TARGET_NAMESPACE%", info.getTargetNamespace()));
		}
		if (info.hasEventConsumers()) {
			wsdl.append(WSDL_BINDING_WSN_CONSUMER_OPERATIONS.replaceAll(
					"%TARGET_NAMESPACE%", info.getTargetNamespace()));
		}
		// Do custom operations
		operations = (ManagedOperationInfo[]) info.getOperations();
		for (ManagedOperationInfo operInfo : operations) {
			// JOEL skip event consumer methods
			if (operInfo.isEventConsumer())
				continue;

			String operName = operInfo.getName();
			String soapAction = operInfo.getName();
			String inputName = operInfo.getName() + "Request";
			String outputName = operInfo.getName() + "Request";

			WSDMOperation wsdmOper = operInfo
					.getAnnotation(WSDMOperation.class);
			if (wsdmOper != null) {
				if (!"".equals(wsdmOper.soapAction()))
					soapAction = wsdmOper.soapAction();
				if (!"".equals(wsdmOper.inputName()))
					inputName = wsdmOper.inputName();
				if (!"".equals(wsdmOper.outputName()))
					outputName = wsdmOper.outputName();
			}

			wsdl.append("<wsdl:operation name=\"");
			wsdl.append(operName);
			wsdl.append("\">");
			wsdl.append("<wsdl-soap:operation soapAction=\"");
			wsdl.append(soapAction);
			wsdl.append("\"/>");

			wsdl.append("<wsdl:input name=\"");
			wsdl.append(inputName);
			wsdl.append("\">");
			wsdl
					.append("<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />");
			wsdl.append("</wsdl:input>");

			wsdl.append("<wsdl:output name=\"");
			wsdl.append(outputName);
			wsdl.append("\">");
			wsdl
					.append("<wsdl-soap:body use=\"encoded\" encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\" />");
			wsdl.append("</wsdl:output>");

			wsdl.append(WSDL_BINDING_STANDARD_CUSTOM_FAULTS);
			wsdl.append("</wsdl:operation>");
		}

		wsdl.append(WSDL_BINDING_END);

		wsdl.append(" <wsdl:service name=\"");
		wsdl.append(info.getResourceName());
		wsdl.append("Service\">");
		wsdl.append(" <wsdl:port name=\"");
		wsdl.append(info.getResourceName());
		wsdl.append("Port\" binding=\"dyn:");
		wsdl.append(info.getResourceName());
		wsdl.append("Binding\">");
		wsdl.append(" <wsdl-soap:address location=\"");
		try {
			wsdl.append(WSDMBinding.getLocalAddress(WSDMBindingFactory.getDefault().getContextPath(),
					WSDMBindingFactory.getDefault().getPort()));
		} catch (Throwable t) {

		}
		wsdl.append("/" + info.getResourceName());
		wsdl.append("\"/></wsdl:port></wsdl:service>");

		wsdl.append(WSDL_END);

		// Stop filling up the console!
		//System.out.println(wsdl.toString());

		Document document = null;
		try {
			document = XmlUtils.createDocument(new InputSource(
					new StringReader(wsdl.toString())));

			//
			//
			// otherwise we must process all imports/includes, then add in
			// any WSDL imports and merge them together
			//

			Set namespaces = WsdlUtils.getSchemaNamespaces(document);

			String wsdlPath = "";

			//
			// take care of our imports first...
			//
			WsdlUtils.addSchemaIncludes(document, wsdlPath, getEnvironment(),
					namespaces);
			WsdlUtils.addSchemaImports(document, wsdlPath, getEnvironment(),
					namespaces);

			//
			// ...then create all WSDLs that were imported...
			//
			Collection otherWSDLs = WsdlUtils.getWsdlImports(document,
					wsdlPath, getEnvironment());
			Iterator i = otherWSDLs.iterator();

			//
			// ...and merge them together
			//
			while (i.hasNext()) {
				Document next = (Document) i.next();
				WsdlUtils.mergeWSDL(next, document, namespaces);
			}
			//
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (SAXException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		if (document != null){
			return document;
		}
		return null;
	}

	private void processAssertions(StringBuffer wsdl,
			Collection<PolicyAssertion> assertions) {
		if(assertions != null){
			for(PolicyAssertion assertion: assertions){
				String id = assertion.getPolicyId();
				String xmlString = assertion.toXmlString();
				if(id != null && xmlString != null){
					wsdl.append("<wsp:Policy xml:Id=\"" + id + "\">");
					wsdl.append(xmlString);
					wsdl.append("</wsp:Policy>");
				}
			}
		}
	}

}
