/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.me.dmtoolkit.j2ee.internal.delegates;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.cosmos.me.dmtoolkit.common.Utilities;
import org.eclipse.cosmos.me.dmtoolkit.common.artifacts.IDMToolkitConstants;
import org.eclipse.cosmos.me.dmtoolkit.common.artifacts.IProjectConfig;
import org.eclipse.cosmos.me.dmtoolkit.j2ee.internal.DMJ2EEPlugin;
import org.eclipse.cosmos.me.dmtoolkit.j2ee.internal.util.J2EEUtilities;
import org.eclipse.osgi.util.NLS;
import org.eclipse.wst.common.project.facet.core.IDelegate;
import org.eclipse.wst.common.project.facet.core.IProjectFacetVersion;
import org.osgi.framework.Bundle;

/**
 * Delegate to setup a data manager server project.
 * 
 * @author David Whiteman
 */
public class DataManagerFacetInstallDelegate implements IDelegate {
    private static final String RESOURCES_FOLDER = "resources/";
	private IProjectConfig config;

	/* (non-Javadoc)
	 * @see org.eclipse.wst.common.project.facet.core.IDelegate#execute(org.eclipse.core.resources.IProject, org.eclipse.wst.common.project.facet.core.IProjectFacetVersion, java.lang.Object, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void execute(IProject project,
            IProjectFacetVersion fv,
            Object config,
            IProgressMonitor monitor) throws CoreException {
        setConfig((IProjectConfig) config);
        monitor.beginTask("Generating data manager project", taskCount());
        try
        {
        	copyBundles(project, monitor);
        	copyBundleContents(project, monitor);
        	copyResources(project, monitor);
        	
        	createServiceGroupClass(project, monitor);
        	
        	populateServicesDir(project, monitor);
        }
        finally
        {
            monitor.done();
        }
    	
    }

	// Ensure the services directory exists, creating services.xml and copying required WSDLs
	protected void populateServicesDir(IProject project, IProgressMonitor monitor) throws CoreException {
		String dataManagerName = getConfig().getName();
		IFolder servicesDir = J2EEUtilities.getWebInfServicesDir(project);;
		IFolder dataManagerDir = servicesDir.getFolder(dataManagerName);
		if (!dataManagerDir.exists()) {
			dataManagerDir.create(true, true, monitor);
		}
		IFolder metaInfDir = dataManagerDir.getFolder("META-INF");
		if (!metaInfDir.exists()) {
			metaInfDir.create(true, true, monitor);
		}
		copyWsdls(metaInfDir, monitor);
		copyDatamodel(metaInfDir, monitor);
		createServicesXml(metaInfDir, monitor);
	}

	// Copy the data model schema
	protected void copyDatamodel(IFolder metaInfDir, IProgressMonitor monitor) throws CoreException {
		final Bundle bundle = DMJ2EEPlugin.getDefault().getBundle();
		J2EEUtilities.copyFromPlugin(bundle, new Path(RESOURCES_FOLDER+IDMToolkitConstants.CMDBF_DATAMODEL_XML_FILE_NAME), metaInfDir);
		monitor.worked(1);
	}

	// Create the services.xml file
	protected void createServicesXml(IFolder metaInfDir, IProgressMonitor monitor) throws CoreException {
		File propsFile = metaInfDir.getLocation().append(IDMToolkitConstants.SERVICES_XML_FILE_NAME).toFile();
		String servicesXmlContents = null;
		PrintWriter pw = null;
		try {
			if (propsFile.exists()) {
				// Get the contents of the existing file
				servicesXmlContents = getFileContentsAsString(propsFile);
			} else {
				// Build a new services.xml, including the data manager service group
				String[] substitutionMap = {
					"%DATA_MANAGER_NAME%", getConfig().getName(),
				};
				servicesXmlContents = J2EEUtilities.populateResourceTemplate(substitutionMap, RESOURCES_FOLDER+"servicesXmlTemplate.txt");
			}
			// Insert the query or registration service group at the end of the file
			servicesXmlContents = servicesXmlContents.replace("</serviceGroup>", getOtherServicesString()+"</serviceGroup>");
			pw = new PrintWriter(new FileWriter(propsFile));
			pw.write(servicesXmlContents);
		} catch (FileNotFoundException e) {
			throw createCoreException(e);
		} catch (IOException e) {
			throw createCoreException(e);
		} finally {
			pw.close();
		}
		IFile file = metaInfDir.getFile(IDMToolkitConstants.SERVICES_XML_FILE_NAME);
		file.refreshLocal(1, monitor);
		monitor.worked(1);
	}

	// Answer the contents of a file as a string
	protected String getFileContentsAsString(File propsFile)
			throws FileNotFoundException, IOException {
		String servicesXmlContents;
		BufferedReader reader = new BufferedReader(new FileReader(propsFile));
		String line = null;
		StringWriter writer = new StringWriter();
		while ((line = reader.readLine()) != null) {
			writer.write(line+"\n");
		}
		writer.close();
		reader.close();
		servicesXmlContents = writer.toString();
		return servicesXmlContents;
	}

	protected String getOtherServicesString() throws CoreException {
		StringWriter writer = new StringWriter();
		String[] substitutionMap = {
			"%DATA_MANAGER_NAME%", getConfig().getName(),
			"%FULLY_QUALIFIED_SERVICE_CLASS%", getFullyQualifiedClassName()
		};
		String fileContents = J2EEUtilities.populateResourceTemplate(substitutionMap, RESOURCES_FOLDER+getServiceTemplateFileName());
		writer.append(fileContents+"\n");
		return writer.toString();
	}

	protected String getFullyQualifiedClassName() {
		return getConfig().getPackageName()+"."+getConfig().getGeneratedClassName();
	}

	protected String getServiceTemplateFileName() {
		if (getConfig().isQueryServiceConfig()) {
			return "servicesXmlQueryTemplate.txt";
		} else {
			return "servicesXmlRegistrationTemplate.txt";
		}
	}

	/**
	 * Copy the query and registration WSDLs to the MDR project.
	 * 
	 * @param metaInfDir
	 * @param monitor
	 * @throws CoreException
	 */
	protected void copyWsdls(IFolder metaInfDir, IProgressMonitor monitor) throws CoreException {
		List<String> bundleList = getConfig().getWsdlFilesToCopy();
		Iterator<String> iterator = bundleList.iterator();
		Bundle bundle = DMJ2EEPlugin.getDefault().getBundle();
		while (iterator.hasNext()) {
			String fileName = (String) iterator.next();
			IPath sourceFile = new Path(RESOURCES_FOLDER+fileName);
			try {
				IFile destFile = metaInfDir.getFile(sourceFile.lastSegment());
				if (!destFile.exists()) {
					final InputStream in = FileLocator.openStream(bundle, sourceFile, false);
					BufferedReader reader = new BufferedReader(new InputStreamReader(in));
					String line = null;
					StringWriter writer = new StringWriter();
					while ((line = reader.readLine()) != null) {
						String newLine = line.replace("%SERVICE_NAME%", getConfig().getGeneratedClassName());
						writer.write(newLine+"\n");
					}
					try {
						writer.close();
						reader.close();
					} catch (IOException e) {
						e.printStackTrace();
					}
					FileWriter fileForRewrite = new FileWriter(destFile.getLocation().toOSString());
					fileForRewrite.write(writer.toString());
					fileForRewrite.close();
					destFile.refreshLocal(1, monitor);
				}
			} catch (IOException e) {
				throw DMJ2EEPlugin.createCoreException(e);
			}
			monitor.worked(1);
		}
	}

	protected static String readLine(BufferedReader reader) throws CoreException {
		String line = null;
		try {
			line = reader.readLine();
		} catch (IOException e) {
			throw createCoreException(e);
		}
		return line;
	}

	protected static CoreException createCoreException(Exception e) {
		return new CoreException(DMJ2EEPlugin.createErrorStatus(e.getMessage(), e));
	}

	/**
	 * Create the class representing the service group
	 * 
	 * @param project
	 * @param monitor
	 * @throws CoreException
	 */
	protected void createServiceGroupClass(IProject project,
			IProgressMonitor monitor) throws CoreException {
		Utilities.createServiceGroupClass(project, getConfig(), monitor);
	}

	/**
	 * Answer the count of tasks for the progress monitor.  This is the count of files to
	 * be copied, plus other tasks, such as generating classes.
	 * @return
	 */
	protected int taskCount() {
		return fileCount() + 3; // files + create data manager + create services xml + copy data model schema
	}

	/**
	 * Copy bundle jars from the Eclipse plugins directory to the new project's WEB-INF/lib
	 * directory.
	 * 
	 * @param project
	 * @param monitor
	 * @throws CoreException
	 */
	protected void copyBundles(IProject project, IProgressMonitor monitor)
			throws CoreException {
		Iterator<String> iterator = getBundlesToCopy().iterator();
		while (iterator.hasNext()) {
			String bundleName = (String) iterator.next();
			final IFolder webInfLib = J2EEUtilities.getWebInfLibDir(project);
			J2EEUtilities.copyBundle(bundleName, webInfLib);
			monitor.worked(1);
		}
	}

	/**
	 * Copy resources relative to this plug-in to the new project's WEB-INF/lib directory.
	 *  
	 * @param project
	 * @param monitor
	 * @throws CoreException
	 */
	protected void copyResources(IProject project, IProgressMonitor monitor)
			throws CoreException {
		final Bundle bundle = DMJ2EEPlugin.getDefault().getBundle();
		final IFolder webInfLib = J2EEUtilities.getWebInfLibDir(project);
		Iterator<String> iterator = getResourcesToCopy().iterator();
		while (iterator.hasNext()) {
			String fileName = (String) iterator.next();
			J2EEUtilities.copyFromPlugin(bundle, new Path(fileName), webInfLib);
			monitor.worked(1);
		}
	}

	/**
	 * Copy files from a bundle directory to the generated project's WEB-INF/lib directory
	 * 
	 * @param project
	 * @param monitor
	 * @throws CoreException
	 */
	protected void copyBundleContents(IProject project, IProgressMonitor monitor)
			throws CoreException {
		final IFolder webInfLib = J2EEUtilities.getWebInfLibDir(project);
		Map<String, String> bundleMap = getBundleContentsToCopy();
		Iterator<String> iterator = bundleMap.keySet().iterator();
		while (iterator.hasNext()) {
			String jarFileName = (String) iterator.next();
			String symbolicName = bundleMap.get(jarFileName);
			Bundle bundle = Platform.getBundle(symbolicName);
			if (bundle == null) {
				throw new CoreException(DMJ2EEPlugin.createErrorStatus(NLS.bind("Required bundle {0} is missing.", symbolicName)));
			}
			J2EEUtilities.copyFromPlugin(bundle, new Path(jarFileName),
					webInfLib);
			monitor.worked(1);
		}
	}
	
	
    /**
     * Answer the count of all files copied to the WEB-INF/lib directory of the generated project.
     * This is used to help accurately set the progress monitor.
     * 
     * @return
     */
    protected int fileCount() {
    	return getBundlesToCopy().size() + getResourcesToCopy().size() + getBundleContentsToCopy().size() + getConfig().getWsdlFilesToCopy().size();
    }

	/**
	 * Answer the list of bundle jars to copy from the eclipse plugins to the generated
	 * project's WEB-INF/lib directory.
	 * 
	 * @return
	 */
	protected ArrayList<String> getBundlesToCopy() {
		String[] bundleList = {
			"org.eclipse.cosmos.common"
		};
		return new ArrayList<String>(Arrays.asList(bundleList));
	}


	/**
	 * Answer a list of resources to be copied from the current bundle to the new project
	 * 
	 * @return
	 */
	protected ArrayList<String> getResourcesToCopy() {
		String[] resourceList = {
			//"resources/muse-platform-mini-2.2.0.jar"
		};
		return new ArrayList(Arrays.asList(resourceList));
	}

	/**
	 * Answer the set of files being copied from other bundles in the environment
	 * to the new project.  The returned map should have file names as keys, and
	 * bundle names as values.
	 * 
	 * @return
	 */
	protected HashMap<String, String> getBundleContentsToCopy() {
		HashMap<String, String> dmList = new HashMap<String, String>();
		String[] bundleList = {
			"CMDBfCommon.jar", "org.eclipse.cosmos.dc.cmdbf.services", 
			"CMDBfQueryTransformation.jar", "org.eclipse.cosmos.dc.cmdbf.services", 
			"CMDBfQueryService.jar", "org.eclipse.cosmos.dc.cmdbf.services",
			"cmdbf-ws.jar", "org.eclipse.cosmos.dc.cmdbf.ws",
			"datamanager-ws.jar", "org.eclipse.cosmos.dc.datamanager.ws"/*,
			"CMDBfservices.jar", "org.eclipse.cosmos.samples.cmdbf.services"*/
		};
		int i = 0;
		while (i < bundleList.length) {
			dmList.put(bundleList[i], bundleList[i+1]);
			i=i+2;
		}
		if (getConfig().isRegistrationServiceConfig()) {
			dmList.put("CMDBfRegistrationService.jar", "org.eclipse.cosmos.dc.cmdbf.services");
			dmList.put("CMDBfRegistrationTransformation.jar", "org.eclipse.cosmos.dc.cmdbf.services");
			dmList.put("CMDBfMetadata.jar", "org.eclipse.cosmos.dc.cmdbf.services");
		}
		return dmList;
	}

	public IProjectConfig getConfig() {
		return config;
	}

	public void setConfig(IProjectConfig config) {
		this.config = config;
	}
}
