/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.me.dmtoolkit.j2ee.internal.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.cosmos.me.dmtoolkit.common.Utilities;
import org.eclipse.cosmos.me.dmtoolkit.common.artifacts.IProjectConfig;
import org.eclipse.cosmos.me.dmtoolkit.j2ee.internal.DMJ2EEPlugin;
import org.eclipse.osgi.util.NLS;
import org.eclipse.wst.common.componentcore.ComponentCore;
import org.eclipse.wst.common.componentcore.resources.IVirtualComponent;
import org.eclipse.wst.common.componentcore.resources.IVirtualFolder;
import org.osgi.framework.Bundle;

/**
 * Common utilities used by the data manager toolkit J2EE tooling
 * 
 * @author David Whiteman
 */
public final class J2EEUtilities {
	/**
	 * Returns the location of the web project's "lib" directory.
	 * 
	 * @param project
	 *            the web project
	 * @return the location of the WEB-INF/lib directory
	 */
	public static IFolder getWebInfLibDir(final IProject project) {
		IVirtualFolder vf = getWebInfSubdir(project, "lib");
		return (IFolder) vf.getUnderlyingFolder();
	}
	
	public static IFolder getWebInfServicesDir(final IProject project) {
		IVirtualFolder vf = getWebInfSubdir(project, "services");
		return (IFolder) vf.getUnderlyingFolder();
	}

	/**
	 * Returns the location of the web project's "classes" directory.
	 * 
	 * @param project
	 * @return
	 * @throws CoreException
	 */
	public static IFolder getWebInfClassesDir(final IProject project) throws CoreException {
		IFolder lib = getWebInfLibDir(project);
		// Kind of a lame hack to get WEB-INF/classes. Using getFolder()
		// directly to get it causes it to resolve to the project's "src" folder, 
		// which we don't want. Once I have the "lib" handle, I can go to the 
		// parent (WEB-INF) and then its child "classes".
		IFolder classesFolder = ((IFolder) lib.getParent()).getFolder("classes");
		if (!classesFolder.exists()) {
			classesFolder.create(true, true, null);
		}
		return classesFolder;
	}

	protected static IVirtualFolder getWebInfSubdir(final IProject pj,
			String subdir) {
		final IVirtualComponent vc = ComponentCore.createComponent(pj);
		final IVirtualFolder vf = vc.getRootFolder().getFolder(
				"WEB-INF/" + subdir);
		return vf;
	}

	/**
	 * Copy a bundle jar to the indicated directory.  An exception is thrown
	 * if the bundle cannot be located, or if it is packaged as a directory.
	 * 
	 * @param bundleName
	 * @param dest
	 * @throws CoreException
	 */
	public static void copyBundle(final String bundleName, final IFolder dest)
			throws CoreException {
		try {
			final Bundle bundle = Platform.getBundle(bundleName);
			if (bundle == null) {
				throw new CoreException(DMJ2EEPlugin.createErrorStatus(NLS.bind("Required bundle {0} is missing.", bundleName)));
			}
			final File file = FileLocator.getBundleFile(bundle);
			if (file.isDirectory()) {
				throw new CoreException(DMJ2EEPlugin.createErrorStatus(NLS.bind("Bundle {0} is a directory.  This bundle must not exist in the workspace, but instead should be a jar file in the plugins directory.", bundleName)));
			}
			IFile destFile = dest.getFile(file.getName());
			if (!destFile.exists()) {
				InputStream in = new FileInputStream(file);
				destFile.create(in, true, null);
			}
		} catch (IOException e) {
			throw DMJ2EEPlugin.createCoreException(e);
		}
	}

	/**
	 * Copy the file named <code>sourceFile</code> from the indicated bundle
	 * to the destination folder.
	 * 
	 * @param bundle
	 * @param sourceFile
	 * @param dest
	 * @throws CoreException
	 */
	public static void copyFromPlugin(Bundle bundle, final IPath sourceFile,
			final IFolder dest) throws CoreException {
		try {
			IFile destFile = dest.getFile(sourceFile.lastSegment());
			if (!destFile.exists()) {
				final InputStream in = FileLocator.openStream(bundle, sourceFile, false);
				destFile.create(in, true, null);
			}
		} catch (IOException e) {
			throw DMJ2EEPlugin.createCoreException(e);
		}
	}

	/**
	 * Answer the source for the data manager initializer class.
	 * 
	 * @param config
	 * @return
	 * @throws CoreException
	 */
	public static String getDataManagerInitializerContents(IProjectConfig config) throws CoreException {
		String[] substitutionMap = {
				"%PACKAGE_DECLARATION%", getPackageDeclaration(config),
				"%INITIALIZER_CLASS%", getInitializerClassName(config),
				"%DATA_MANAGER_CLASS%", config.getName()
		};
		String resourceLocation = "resources/initializerSourceTemplate.txt";
		return populateResourceTemplate(substitutionMap, resourceLocation);
	}
	
	public static String populateResourceTemplate(String[] substitutionMap,
			String resourceLocation) throws CoreException {
		return Utilities.populateResourceTemplate(substitutionMap, resourceLocation, DMJ2EEPlugin.getDefault().getBundle());
	}

	protected static String getPackageDeclaration(IProjectConfig config) {
		return "".equals(config.getPackageName()) ? "" : "package "+config.getPackageName()+";";
	}

	public static String getInitializerClassName(IProjectConfig projectConfig) {
		return projectConfig.getName()+"Initializer";
	}

}
