/*
* Copyright (c) 2005-2007 Compuware Corporation and others.
* All rights reserved. This program and the accompanying
* materials are made available under the terms of the
* Eclipse Public License v1.0 which accompanies this
* distribution, and is available at:
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*     Compuware Corporation - initial API and implementation
*
*/
package org.eclipse.cosmos.me.management.provisional.wsdm.impl.info;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.management.MBeanParameterInfo;
import javax.xml.namespace.QName;

import org.apache.muse.core.Capability;
import org.apache.muse.core.Environment;
import org.apache.muse.core.routing.MessageHandler;
import org.apache.muse.core.routing.ReflectionMessageHandler;
import org.apache.muse.core.serializer.Serializer;
import org.apache.muse.core.serializer.SerializerRegistry;
import org.apache.muse.util.xml.XmlUtils;
import org.apache.muse.ws.dm.muws.MuwsConstants;
import org.apache.muse.ws.dm.muws.impl.SimpleManageabilityCharacteristics;
import org.apache.muse.ws.dm.muws.impl.SimpleMetrics;
import org.apache.muse.ws.dm.muws.impl.SimpleRelationships;
import org.apache.muse.ws.metadata.WsxConstants;
import org.apache.muse.ws.notification.WsnConstants;
import org.apache.muse.ws.notification.impl.SimpleNotificationProducer;
import org.apache.muse.ws.notification.impl.SimpleSubscriptionManager;
import org.apache.muse.ws.resource.WsrfConstants;
import org.apache.muse.ws.resource.properties.WsrpConstants;
import org.apache.muse.ws.resource.properties.get.impl.SimpleGetCapability;
import org.apache.muse.ws.resource.properties.query.impl.SimpleQueryCapability;
import org.apache.muse.ws.resource.properties.set.impl.SimpleSetCapability;
import org.eclipse.cosmos.me.management.common.info.ComposedCapabilityInfo;
import org.eclipse.cosmos.me.management.common.info.ManagedAttributeInfo;
import org.eclipse.cosmos.me.management.common.info.ManagedOperationInfo;
import org.eclipse.cosmos.me.management.common.info.ManagedRelationInfo;
import org.eclipse.cosmos.me.management.common.info.ManagedResourceInfo;
import org.eclipse.cosmos.me.management.common.util.BindingUtil;
import org.eclipse.cosmos.me.management.provisional.wsdm.annotations.WSDMOperation;
import org.eclipse.cosmos.me.management.provisional.wsdm.annotations.WSDMParameterType;
import org.eclipse.cosmos.me.management.provisional.wsdm.annotations.WSDMSchemaImport;
import org.eclipse.cosmos.me.management.provisional.wsdm.impl.DynamicAdvertisementCapability;
import org.eclipse.cosmos.me.management.provisional.wsdm.impl.DynamicNotificationConsumer;
import org.eclipse.cosmos.me.management.provisional.wsdm.impl.WSDLSource;
import org.eclipse.cosmos.me.management.provisional.wsdm.impl.info.WSDMOperationInfo;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.NodeList;

public class WSDMInfo {

	private String targetNamespace = "";
	private ManagedResourceInfo info;
	private int namespaceCount = 0;
	private Object contribution;

	private HashMap<String, WSDMOperationInfo> metaMethodMap = new HashMap<String, WSDMOperationInfo>();
	
	private HashMap<String, String> namespaceList = new HashMap<String, String>();
	
	//namespace to import
	private HashMap<String, String> importList = new HashMap<String, String>();
	
	private HashMap<String, Class> capabilityMap = new HashMap<String, Class>();

	private class HandlerDescriptor {
		public HandlerDescriptor(Method method, QName inputName, QName outputName){
			this.method = method;
			this.inputName = inputName;
			this.outputName = outputName;
		}

		Method method;
		QName inputName;
		QName outputName;
	};

	private HashMap<Class, HashMap<String,HandlerDescriptor>> capabilityMethodMap = new HashMap<Class, HashMap<String,HandlerDescriptor>>();

	private <T extends Annotation> T getAnnotationFromList(Class <T> target, Annotation[] annotations){
		for(Annotation annotation : annotations){
			if(annotation.annotationType().equals(target))return (T)annotation;
		}
		return null;
	}

	public WSDMInfo(ManagedResourceInfo info, Object contribution) throws IllegalArgumentException {
		this.info = info;
		this.contribution = contribution;
		this.targetNamespace = info.getTargetNamespace();
		namespaceList.put(targetNamespace, "dyn");
		//populate the namespace list:
		String[] nsSet = info.getNamespaces();
		for(String ns : nsSet){
			String pfx = info.getPrefixForNamespace(ns);
			namespaceList.put(ns, pfx);
		}
		
		//sweep through the rest of info to add additional namespaces
		ManagedAttributeInfo[] atts = (ManagedAttributeInfo[])info.getAttributes();
		//Currently no field-level annotations
		ManagedOperationInfo[] ops = (ManagedOperationInfo[])info.getOperations();
		for(ManagedOperationInfo op : ops){
			WSDMOperation wsdmOp = op.getAnnotation(WSDMOperation.class);
			if(wsdmOp != null){
				//FIXME honor overrides!
			}
			MBeanParameterInfo[] parms = op.getSignature();
			for(MBeanParameterInfo parm : parms){
				WSDMParameterType parmType = this.getAnnotationFromList(WSDMParameterType.class, op.getAnnotationsForParameter(parm));
				if(parmType != null){
					addNamespace(parmType.schema());
					if(parmType.serializer() != null && !"".equals(parmType.serializer())){
						try{
							Class parmClass = op.getOperationMethod().getDeclaringClass().getClassLoader().loadClass(parm.getType());
							Serializer ser = null;
							try{
								ser = SerializerRegistry.getInstance().getSerializer(parmClass);
							}catch(RuntimeException rte){
								if(ser == null){
									Class serializer = op.getOperationMethod().getDeclaringClass().getClassLoader().loadClass(parmType.serializer());
									Object serObj = serializer.newInstance();
									if(serObj instanceof Serializer){
										SerializerRegistry.getInstance().registerSerializer(parmClass, (Serializer)serObj);
									}
								}
							}
								
						}catch(Throwable t){
							t.printStackTrace();
						}
					}
				}
			}
		}
		
		extractAdditionalNamespaces(info.getResourceClass());
		ComposedCapabilityInfo caps[] = info.getComposedCapabilities();
		for(ComposedCapabilityInfo cap : caps){
			extractAdditionalNamespaces(cap.getImplClass());
			addCapabilityClass(cap.getNamespace(),cap.getImplClass());
		}
		//Add in capabilities derived from resource class
		for(String cap : info.getCapabilities()){
			if(this.getCapabilityClass(cap) == null)
				addCapabilityClass(cap, info.getResourceClass());
		}
		
	}
	
	private void extractAdditionalNamespaces(Class clzz){
		HashMap<String, Field> fieldMap = new HashMap<String,Field>();
		BindingUtil.extractFields(clzz, fieldMap);
		
		for(Field field : fieldMap.values()){
			WSDMSchemaImport schemaImport = field.getAnnotation(WSDMSchemaImport.class);
			if(schemaImport != null){
				Class decl = field.getDeclaringClass();
				try {
					String value = (String)field.get(null);
					this.addNamespace(value);
					importList.put(value,schemaImport.location());
				} catch (IllegalArgumentException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				} catch (IllegalAccessException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
		
		Class superClzz = clzz.getSuperclass();
		if(superClzz != null && !superClzz.getPackage().getName().startsWith("java")) extractAdditionalNamespaces(superClzz);
		Class ifaces[] = clzz.getInterfaces();
		for(Class iface : ifaces){
			if(!iface.getPackage().getName().startsWith("java")) extractAdditionalNamespaces(iface);
		}
	}
	
	public ManagedResourceInfo getInfo(){
		return info;
	}
	
	public boolean hasNamespace(String namespace){
		return namespaceList.containsKey(namespace);
	}
	
	public String[] getNamespaces(){
		return namespaceList.keySet().toArray(new String[]{});
	}
	
	public String[] getSchemaImports(){
		return importList.keySet().toArray(new String[]{});
	}
	
	public String getSchemaLocation(String namespace){
		return importList.get(namespace);
	}
	
	public String getPrefixForNamespace(String namespace){
		if("".equals(namespace)) return "dyn";
		return namespaceList.get(namespace);
	}

	public ManagedAttributeInfo getResourceIdField(){
	    return info.getResourceIdField();
	}

	public boolean resourceLifetimeSupport(){
		return false;
	}
	
	public boolean hasMetrics(){
		return info.hasMetrics();
	}

	public boolean isAdvertised(){
		return info.isAdvertised();
	}
	
	public boolean hasEventConsumers(){
		return info.hasEventConsumers();
	}

	public String getAdvertisementConsumer(){
		return info.getAdvertisementConsumer();
	}

	public String getTargetNamespace(){
		return info.getTargetNamespace();
	}

	public String getResourceName(){
		return info.getClassName();
	}

	public ManagedRelationInfo[] getRelations(){
		return info.getRelations();
	}
	
	public Class getCapabilityClass(String capability){
		return capabilityMap.get(capability);
	}
	
	public void addCapabilityClass(String capability, Class implClass){
		capabilityMap.put(capability, implClass);
	}
	
	public String[] getCapabilityURIs(){
		return capabilityMap.keySet().toArray(new String[]{});
	}
	

	//Compute handlers for capabilities
	public void compileCapabilties(Environment environment, ManagedResourceInfo info) throws Exception {
		WSDLSource metaData = new WSDLSource(this,contribution);
		metaData.setEnvironment(environment);
		Document wsdl = metaData.getWSDLDocument();
//		Definition wsdlDef = WsdlUtils.createDefinition(wsdl,new File("."));
		Element binding = (Element)wsdl.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", "binding").item(0);
		Element portType = (Element)wsdl.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", "portType").item(0);

		HashMap actionToElementMap = new HashMap();

		NodeList children = binding.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/","operation");
		for(int i=0;i<children.getLength();i++){
			Element child = (Element)children.item(i);
			String operName = child.getAttribute("name");
			NodeList inputChild = child.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/soap/", "operation");
			String action = ((Element)inputChild.item(0)).getAttribute("soapAction");
			actionToElementMap.put(action, child);
		}
		
		//Populate capability map as required.
		if(capabilityMap.get(WsrpConstants.GET_CAPABILITY) == null){
			capabilityMap.put(WsrpConstants.GET_CAPABILITY, SimpleGetCapability.class);
		}
		if(capabilityMap.get(WsrpConstants.QUERY_CAPABILITY) == null){
			capabilityMap.put(WsrpConstants.QUERY_CAPABILITY, SimpleQueryCapability.class);
		}
		if(capabilityMap.get(MuwsConstants.CHARACTERISTICS_URI) == null){
			capabilityMap.put(MuwsConstants.CHARACTERISTICS_URI, SimpleManageabilityCharacteristics.class);
		}
		if(capabilityMap.get(WsrpConstants.SET_CAPABILITY) == null && hasSettableAttributes()){
			capabilityMap.put(WsrpConstants.SET_CAPABILITY, SimpleSetCapability.class);
		}
		if((info.getNotifications().length > 0 || isAdvertised()) && capabilityMap.get(WsnConstants.PRODUCER_URI) == null){
			capabilityMap.put(WsnConstants.PRODUCER_URI, SimpleNotificationProducer.class);
		}
		if(getRelations().length > 0 && capabilityMap.get(MuwsConstants.RELATIONSHIPS_URI) == null){
			capabilityMap.put(MuwsConstants.RELATIONSHIPS_URI, SimpleRelationships.class);
		}		
		if(hasMetrics() && capabilityMap.get(MuwsConstants.METRICS_URI) == null){
			capabilityMap.put(MuwsConstants.METRICS_URI, SimpleMetrics.class);
		}
		if(isAdvertised() && capabilityMap.get(MuwsConstants.ADVERTISEMENT_URI) == null){
			capabilityMap.put(MuwsConstants.ADVERTISEMENT_URI, DynamicAdvertisementCapability.class);
		}
		if(this.hasEventConsumers() && capabilityMap.get(WsnConstants.CONSUMER_URI) == null){
			capabilityMap.put(WsnConstants.CONSUMER_URI, DynamicNotificationConsumer.class);
		}
		
		//JOEL comment out as test for MDR
//		for(String namespace : getNamespaces()){
//			if(capabilityMap.get(namespace) == null /* && namespace.equals(info.getTargetNamespace())*/)
//				capabilityMap.put(namespace, info.getResourceClass());
//		}
		
		capabilityMap.put(getTargetNamespace(), info.getResourceClass());

		for(Class capClass : capabilityMap.values()){
			capabilityMethodMap.put(capClass, compileMessageHandlers(wsdl,portType,actionToElementMap, capClass));
		}

		if(capabilityMap.get(WsxConstants.GET_METADATA_URI) == null){
			capabilityMethodMap.put(WSDLSource.class, compileMessageHandlers(wsdl,portType,actionToElementMap, WSDLSource.class));
		}
		


//		if(resourceLifetimeSupport()){
//			capabilityMethodMap.put(SimpleImmediateTermination.class,compileMessageHandlers(wsdl, portType, actionToElementMap, SimpleImmediateTermination.class));
//			capabilityMethodMap.put(SimpleScheduledTermination.class,compileMessageHandlers(wsdl, portType, actionToElementMap, SimpleScheduledTermination.class));
//		}

//		if((info.getNotifications().length > 0 || isAdvertised()) && capabilityMap.get(WsnConstants.PRODUCER_URI) != null){
//			capabilityMethodMap.put(SimpleNotificationProducer.class,compileMessageHandlers(wsdl, portType, actionToElementMap, SimpleNotificationProducer.class));
//		}
		
//		if(getRelations().length > 0){
//			capabilityMethodMap.put(SimpleRelationships.class,compileMessageHandlers(wsdl, portType, actionToElementMap, SimpleRelationships.class));
//		}
		
//		if(hasMetrics()){
//			capabilityMethodMap.put(SimpleMetrics.class,compileMessageHandlers(wsdl, portType, actionToElementMap, SimpleMetrics.class));
//		}
		
//		if(isAdvertised()){
//			capabilityMethodMap.put(DynamicAdvertisementCapability.class,compileMessageHandlers(wsdl, portType, actionToElementMap, DynamicAdvertisementCapability.class));
//		}
//		if(this.hasEventConsumers()){
//			capabilityMethodMap.put(DynamicNotificationConsumer.class,compileMessageHandlers(wsdl, portType, actionToElementMap, DynamicNotificationConsumer.class));
//		}

	}

    private boolean hasSettableAttributes() {
    	ManagedAttributeInfo[] attrs = (ManagedAttributeInfo[])info.getAttributes();
    	for(ManagedAttributeInfo attr : attrs){
    		if(attr.getSetter() != null) return true;
    	}
		return false;
	}

	protected MessageHandler createMessageHandler(String actionURI,
			QName operationName, QName returnValueName) {
		return new ReflectionMessageHandler(actionURI, operationName,
				returnValueName);
	}

    protected HashMap compileMessageHandlers(Document wsdl, Element portType, Map opsByAction, Class theClass){

    	HashMap<String, HandlerDescriptor> descriptorMap = new HashMap<String, HandlerDescriptor>();
        Map methodsByName = new HashMap();

        Method[] methods = theClass.getMethods();

    	NamedNodeMap docAttrs = wsdl.getDocumentElement().getAttributes();
    	HashMap<String, String> prefixMap = new HashMap<String, String>();
    	for(int i=0;i<docAttrs.getLength();i++){
    		Attr attr = (Attr)docAttrs.item(i);
    		if("http://www.w3.org/2000/xmlns/".equals(attr.getNamespaceURI())){
    			if("xmlns".equals(attr.getName())){
    				prefixMap.put("", attr.getValue());
    			}else{
    				prefixMap.put(attr.getName().substring("xmlns:".length(),attr.getName().length()), attr.getValue());
    			}
    		} else if("targetNamespace".equals(attr.getName())){
    			prefixMap.put("dyn",attr.getValue());
    		}
    	}


        for (int n = 0; n < methods.length; ++n)
        {
            Class declaringClass = methods[n].getDeclaringClass();

            if (declaringClass != Object.class &&
                declaringClass != Capability.class)
                methodsByName.put(methods[n].getName(), methods[n]);
        }

        Collection handlers = new ArrayList(opsByAction.size());
        Iterator i = opsByAction.keySet().iterator();

        while (i.hasNext())
        {
            String action = (String)i.next();
            Element next = (Element)opsByAction.get(action);

            QName inputName = getInputPartName(prefixMap, wsdl, portType, next);
            QName outputName = getOutputPartName(prefixMap, wsdl, portType, next);

            String localName = next.getAttribute("name");
            String methodName = getMethodName(localName);
            Method method = (Method)methodsByName.get(methodName);
            String qualifiedAction = getQualifiedAction(portType,next);
            if(qualifiedAction == null) qualifiedAction = action;
            if (method != null)
            {
            	descriptorMap.put(qualifiedAction, new HandlerDescriptor(method,inputName,outputName));
                i.remove();
            }
        }

    	return descriptorMap;
    }

    public Collection createMessageHandlers(Class theClass)
    {
        Map<String, HandlerDescriptor> actionMap = this.capabilityMethodMap.get(theClass);
        ArrayList<MessageHandler> handlers = new ArrayList<MessageHandler>();

        if(actionMap == null){
        	System.out.println("Oops");
        }
        for(String action: actionMap.keySet()){
        	HandlerDescriptor desc = actionMap.get(action);
            MessageHandler handler = createMessageHandler(action, desc.inputName, desc.outputName);
            handler.setMethod(desc.method);
            handlers.add(handler);
        }

        return handlers;
    }

    private QName getInputPartName(HashMap<String, String> prefixMap, Document doc, Element portType, Element oper){
    	return this.getPartName("input", doc, prefixMap, portType, oper);
    }

    private QName getOutputPartName(HashMap<String, String> prefixMap, Document doc, Element portType, Element oper){
    	return this.getPartName("output", doc, prefixMap, portType, oper);
    }

    private QName getPartName(String part, Document doc, HashMap<String,String> prefixMap, Element portType, Element oper){
    	NodeList inputChild = oper.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", "input");
    	String inputName = ((Element)inputChild.item(0)).getAttribute("name");
    	String message = null;
    	NodeList  operTypeList = portType.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", "operation");
    	for(int i=0;i<operTypeList.getLength();i++){
    		Element type = (Element)operTypeList.item(i);
    		if(oper.getAttribute("name").equals(type.getAttribute("name"))){
    			Element operType = type;
    			NodeList messages = type.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", part);
    			if(messages == null || messages.getLength() == 0) return null;
    			message = ((Element)type.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", part).item(0)).getAttribute("message");
    			break;
    		}
    	}



    	String namespace = prefixMap.get("dyn");
    	String name = message;
    	if(message == null){
    		int x = 1;
    	}
    	if(message.indexOf(":") > 0){
    		String prefix = message.substring(0,message.indexOf(":"));
    		name = message.substring(message.indexOf(":")+1,message.length());
    		namespace = prefixMap.get(prefix);
    	}

    	//now get the message declaration
    	NodeList messageList = doc.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", "message");
    	for(int i=0;i<messageList.getLength();i++){
    		Element msg = (Element)messageList.item(i);
    		if(name.equals(msg.getAttribute("name"))){
    			String uri = msg.getNamespaceURI();
    			String partName = ((Element)msg.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", "part").item(0)).getAttribute("element");
    			message = partName;
    	    	name = partName;
    	    	if(message.indexOf(":") > 0){
    	    		String prefix = message.substring(0,message.indexOf(":"));
    	    		name = message.substring(message.indexOf(":")+1,message.length());
    	    		namespace = prefixMap.get(prefix);
    	    		return new QName(namespace, name, prefix);
    	    	}
    		}
    	}

    	return null;

    }

    private String getQualifiedAction(Element portType, Element oper){
    	NodeList  operTypeList = portType.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", "operation");
    	for(int i=0;i<operTypeList.getLength();i++){
    		Element type = (Element)operTypeList.item(i);
    		if(oper.getAttribute("name").equals(type.getAttribute("name"))){
    			return ((Element)type.getElementsByTagNameNS("http://schemas.xmlsoap.org/wsdl/", "input").item(0)).getAttributeNS("http://www.w3.org/2005/08/addressing","Action");
    		}
    	}
    	return null;
    }

    private String getMethodName(String actionName)
    {
        return Character.toLowerCase(actionName.charAt(0)) + actionName.substring(1);
    }

	public void addNamespace(String schema) {
		String ns = namespaceList.get(schema);
		if(ns == null){
			namespaceList.put(schema, "wsdm"+namespaceCount++);
		}
	}

}
