/*******************************************************************************
 * Copyright (c) 2005, 2007 Compuware Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Compuware Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.me.management.provisional.wsdm.persistence;

import org.apache.muse.core.Resource;
import org.apache.muse.core.ResourceManager;
import org.apache.muse.core.routing.RouterFilePersistence;
import org.apache.muse.util.xml.XmlUtils;
import org.apache.muse.ws.addressing.EndpointReference;
import org.apache.muse.ws.addressing.WsaConstants;
import org.apache.muse.ws.addressing.soap.SoapFault;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class FilePersistor extends RouterFilePersistence {

    /**
     * 
     * {@inheritDoc}
     * <br><br>
     * This implementation treats the XML fragment as a wsa:ReferenceParameters 
     * element. It creates an EPR for the given resource type (context path) 
     * and then adds the reference parameters to it. Finally, it creates an 
     * instance of the resource type and (re-)sets the EPR that it has constructed.
     * 
     */
    protected Resource reloadResource(String contextPath, Element resourceXML)
        throws SoapFault
    {
        ResourceManager manager = getResourceManager();
        
        //
        // the XML from the file is the reference parameters of an EPR, so 
        // we're going to construct the rest of the EPR XML around it and 
        // then turn it into an EPR object
        //
        
        //
        // wrap parameter XML in a wsa:EndpointReference element
        //        
        Document doc = resourceXML.getOwnerDocument();
        Element eprXML = XmlUtils.createElement(doc, WsaConstants.EPR_QNAME);
        eprXML.appendChild(resourceXML);
        
        //
        // get the right address URI for the wsa:Address element - this is 
        // the default URI (has the proper host/port/app) with the context 
        // path for the resource type at the end
        //
        String address = manager.getEnvironment().getDefaultURI();
        int lastSlash = address.lastIndexOf('/');
        address = address.substring(0, lastSlash + 1) + contextPath;
        
        XmlUtils.setElement(eprXML, WsaConstants.ADDRESS_QNAME, address);
        
        //
        // create EPR object from XML and set it on the newly-created resource
        //
        EndpointReference epr = new EndpointReference(eprXML);
        
        Resource resource = manager.createResource(contextPath);
        resource.setEndpointReference(epr);
        
        //
        // continue initialization/registration
        //
        resource.initialize();
        manager.addResource(epr, resource);
        
        return resource;
    }
    
    /**
     * 
     * This implementation checks to see if the resource type is one that 
     * is being persisted, and if so, creates a file for the instance.
     * 
     */
    public void resourceAdded(EndpointReference epr, Resource resource) 
        throws SoapFault
    {
        String contextPath = resource.getContextPath();
        
        if (getResourceManager().isUsingPersistence(contextPath))
            createResourceFile(epr, resource);
    }
    
    /**
     * 
     * This implementation checks to see if the resource type is one that 
     * is being persisted, and if so, tries to delete the resource's file.
     * 
     */
    public void resourceRemoved(EndpointReference epr)
        throws SoapFault
    {
        String contextPath = getContextPath(epr);
        
        if (getResourceManager().isUsingPersistence(contextPath))
            destroyResourceFile(epr);
    }

}
