/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.samples.cmdbf.services.common;

import java.io.StringWriter;

import org.eclipse.cosmos.dc.cmdbf.services.common.INotificationHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IInstanceId;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRecord;

/**
 * A common utility class used by the item and relationship
 * notification handler
 */
public class SampleNotificationHandlerUtil
{
	/**
	 * Handle the registration or deregistration of an item or relationship.
	 * 
	 * If this is a registration request, then the element is accepted iff
	 * for record[j] of element, the following will hold true:
	 * 
	 * There is an i-th entry under supportedTypes where 
	 * supportedTypes[i][0] = record[j].getId() and 
	 * supportedTypes[i][1] = root element name of record[j].getValue()
	 * 
	 * If this is a deregistration request, then the element is accepted iff
	 * for instanceId[j] of element, the following will hold true:
	 * 
	 * There is an i-th entry under supportedTypes where
	 * supportedTypes[i][0] = instanceId[j].getMdrId()
	 * supportedTypes[i][1] = instanceId[j].getLocalId()
	 * 
	 * @param type The type of request.  See {@link INotificationHandler#REGISTRATION}
	 * and {@link INotificationHandler#DEREGISTRATION}
	 * @param element The element to be handled
	 * @param supportedTypes The supported types
	 * @return An error if the request is declined or null if the request
	 * is accepted
	 */
	public static String handle(int type, IGraphElement element, String[][] supportedTypes)
	{				
		if (type == INotificationHandler.REGISTRATION)
		{
			IRecord[] records = element.getRecords();
			for (int i = 0; i < records.length; i++)
			{
				String namespace = records[i].getRecordMetadata().getRecordId().toString();
				StringWriter sw = new StringWriter();
				records[i].getValue().toXML(sw, 0);
				int openAngle = sw.getBuffer().indexOf("<");
				int colon = sw.getBuffer().indexOf(":");
								
				int spaceInx = sw.getBuffer().indexOf(" ", openAngle);
				int closeAngle = sw.getBuffer().indexOf(">");
				int close = spaceInx < 0 ? closeAngle : (closeAngle < 0 ? spaceInx : Math.min(spaceInx, closeAngle));
				
				if (openAngle > 0 && close > 0)
				{
					String localName = sw.getBuffer().substring((colon > 0 && (colon > openAngle && colon < close)) ? colon + 1 : openAngle + 1, close);
					if (isTypeSupported (supportedTypes, namespace, localName))
					{
						return null;
					}
				}
			}
		}
		else if (type == INotificationHandler.DEREGISTRATION)
		{
			IInstanceId[] instanceIds = element.getInstanceIds();
			for (int i = 0; i < instanceIds.length; i++)
			{
				if (isTypeSupported(supportedTypes, instanceIds[i].getMdrId().toString(), instanceIds[i].getLocalId().toString()))
				{
					return null;
				}
			}
		}
				
		return "Type is not supported";
	}

	
	/**
	 * A helper method used to determined if there is an i-th entry
	 * in supportedTypes where supportedTypes[i][0] = namespace and
	 * supportedTypes[i][1] = localName
	 * 
	 * @param supportedTypes The supported types
	 * @param namespace The namespace
	 * @param localName The local name
	 * @return true iff the condition mentioned above is satisfied; false
	 * otherwise
	 */
	private static boolean isTypeSupported(String[][] supportedTypes, String namespace, String localName)
	{
		for (int i = 0; i < supportedTypes.length; i++)
		{
			if (!supportedTypes[i][0].equals(namespace) || !supportedTypes[i][1].equals(localName))
				continue;
			
			return true;				
		}
		
		return false;
	}
}
