/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.samples.cmdbf.services.query;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.IRelationshipTemplateHandler;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.impl.AbstractQueryHandler;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IRelationshipTemplate;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IEdges;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IQueryResult;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.QueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRelationship;
import org.eclipse.cosmos.samples.cmdbf.services.query.XMLRepository.ClassSession;
import org.eclipse.cosmos.samples.cmdbf.services.query.XMLRepository.Student;
import org.eclipse.cosmos.samples.cmdbf.services.query.XMLRepository.Teacher;

/**
 * Represents the relationship record type handler.  There is only one type
 * of relationship that this handler accepts: "teaches".  The "teaches"
 * relationship can only exist between a teacher A and a student B (i.e.
 * A teaches B).  
 * Adopters can either extend AbstractRelationshipConstraintHandler or provide
 * a direct implementation of IRelationshipConstraintHandler
 */
public class RelationshipTemplateHandler extends AbstractQueryHandler implements IRelationshipTemplateHandler
{		
	public IEdges execute(IQueryResult context, IRelationshipTemplate relationshipTemplate, IItem source, IItem target) throws CMDBfServiceException
	{
		IEdges results = QueryOutputArtifactFactory.getInstance().createEdges(relationshipTemplate);
		IRecord[] sourceRecords = source.getRecords();
		IRecord[] targetRecords = target.getRecords();
				
		// Only one record is expected
		if (sourceRecords.length != 1 || targetRecords.length != 1)
		{
			return results;
		}
		
		if (sourceRecords[0].getValue() instanceof Teacher && targetRecords[0].getValue() instanceof Student)
		{
			XMLRepository repo = (XMLRepository)getValue(ICMDBfSampleConstants.DATA_PROVIDER);		
			String teacherId = ((Teacher)sourceRecords[0].getValue()).identity.id;
			String studentId = ((Student)targetRecords[0].getValue()).identity.id;
			
			ClassSession[] classSessions = findClass (repo, teacherId, studentId);
			for (int i = 0; i < classSessions.length; i++)
			{
				IRelationship relationship = classSessions[i].toRelationship(results);
				relationship.setSourceId(QueryOutputArtifactFactory.getInstance().createInstanceId(XMLRepository.MDR_ID, teacherId));
				relationship.setTargetId(QueryOutputArtifactFactory.getInstance().createInstanceId(XMLRepository.MDR_ID, studentId));
				results.addRelationship(relationship);
			}
		}
				
		return results;
	}
	
	private ClassSession[] findClass(XMLRepository repo, String teacherId, String studentId)
	{
		List<ClassSession> discoveredClasses = new ArrayList<ClassSession>();		
		ClassSession[] classes = repo.classes;
		for (int i = 0; i < classes.length; i++)
		{
			if (teacherId.equals(classes[i].teacher.identity.id))
			{
				Student[] students = classes[i].students;
				for (int j = 0; j < students.length; j++)
				{
					if (studentId.equals(students[j].identity.id))
					{
						discoveredClasses.add(classes[i]);
					}
				}
			}
		}
		
		return discoveredClasses.toArray(new ClassSession[discoveredClasses.size()]);
	}	
}
