/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.samples.cmdbf.services.query;

import java.io.IOException;
import java.io.StringWriter;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IItemConvertible;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.INodes;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IRelationshipConvertible;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.QueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElementCollection;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRecord;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRelationship;
import org.xml.sax.SAXException;

/**
 * This class represents a simple repository that is populated
 * by the data stored in a static XML file.  An implementation of
 * IDataProvider is used to provide a binding to an underlying
 * repository.
 */
public class XMLRepository
{
	/**
	 * The MDR ID
	 */
	public static final String MDR_ID = "org.eclipse.cosmos.samples.cmdbf.XMLRepository";
	
	/**
	 * The XML file that stores the data
	 */
	private static final String XML_DATA = "data.xml";
	
	/**
	 * The namespace of the data
	 */
	private static final String SCHOOL_NAMESPACE = "http://school";
	
	/**
	 * The students of the school
	 */
	public Student[] students;
	
	/**
	 * The teachers of the school
	 */
	public Teacher[] teachers;
	
	/**
	 * The classes of the school
	 */
	public ClassSession[] classes;
	
	
	/**
	 * Constructor
	 * 
	 * @throws ParserConfigurationException 
	 * @throws IOException 
	 * @throws SAXException 
	 */
	public XMLRepository() throws ParserConfigurationException, SAXException, IOException
	{			
		// Read in the XML file and populate fields accordingly
		SchoolXMLHandler schoolXMLHandler = new SchoolXMLHandler();
		SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();		
		saxParserFactory.newSAXParser().parse(XMLRepository.class.getClassLoader().getResourceAsStream(XML_DATA), schoolXMLHandler);		
		
		students = schoolXMLHandler.getStudents();
		teachers = schoolXMLHandler.getTeachers();
		classes = schoolXMLHandler.getSchoolClasses();
	}
	
	
	/**
	 * Represents an identity, which is consisted
	 * of a first name, last name, and an id.
	 */
	public static class Identity
	{
		public String firstName;
		public String lastName;
		public String id;
	}
	
	/**
	 * An abstract class representing a school 
	 * member
	 */
	public abstract static class SchoolMember implements IItemConvertible, IXMLWritable
	{
		public Identity identity = new Identity();		
		
		public IItem toItem(INodes parent)
		{
			IItem item = QueryOutputArtifactFactory.getInstance().createItem();			
			item.addInstanceId(QueryOutputArtifactFactory.getInstance().createInstanceId(MDR_ID, identity.id));
			IRecord record = QueryOutputArtifactFactory.getInstance().createRecord(item, identity.id);
			record.addNamespace("", SCHOOL_NAMESPACE);
			record.setValue(this);
			item.addRecord(record);
			return item;
		}
		
		private static String valueWithIndent(int indent, String value) 
		{
			String tempValue = value;
			StringWriter tabsWriter = new StringWriter();		
			tabsWriter.append('\n');
			CMDBfServicesUtil.addIndent(tabsWriter, indent);				
			tempValue = tempValue.replace("\n", tabsWriter.toString());
			tempValue = tabsWriter.toString() + tempValue;
			return tempValue;
		}
		
		public void toXML(StringWriter writer, int indentLevel)
		{
			StringBuffer buffer = new StringBuffer();
			buffer.append ("<" + getElementName() + ">\n");
			buffer.append (" <identity firstName=\"" + identity.firstName + "\" lastName=\"" + identity.lastName + "\" id=\"" + identity.id + "\"/>\n");
			buffer.append ("</" + getElementName() + ">");
			writer.write(SchoolMember.valueWithIndent(indentLevel, buffer.toString()) + "\n\n");
		}

		protected abstract String getElementName();
	}
	
	/**
	 * Represents a student
	 */
	public static class Student extends SchoolMember
	{
		@Override
		protected String getElementName()
		{
			return "student";
		}

	}
	
	/**
	 * Represents a teacher of a school
	 */
	public static class Teacher extends SchoolMember
	{
		@Override
		protected String getElementName()
		{
			return "teacher";
		}
	}
	
	/**
	 * Represents a class with enrolled students
	 * and a teacher
	 */
	public static class ClassSession implements IItemConvertible, IRelationshipConvertible, IXMLWritable
	{
		public Student[] students;
		public Teacher teacher;
		public String name;
		public String courseCode;
		public IItem toItem(INodes parent)
		{
			IItem item = QueryOutputArtifactFactory.getInstance().createItem();
			item.addRecord(createRecord(item));
			return item;
		}
		
		
		public IRelationship toRelationship(IGraphElementCollection parent)
		{
			IRelationship relationship = QueryOutputArtifactFactory.getInstance().createRelationship();
			relationship.addInstanceId(QueryOutputArtifactFactory.getInstance().createInstanceId(MDR_ID, courseCode));
			relationship.addRecord(createRecord(relationship));			
			return relationship;
		}
		
		private IRecord createRecord(IGraphElement parent)
		{
			IRecord record = QueryOutputArtifactFactory.getInstance().createRecord(parent, courseCode);
			record.addNamespace("", SCHOOL_NAMESPACE);
			record.setValue(this);
			return record;
		}


		public void toXML(StringWriter writer, int indentLevel)
		{
			StringBuffer buffer = new StringBuffer();
			buffer.append("<class name=\"" + name + "\" courseCode=\"" + courseCode + "\">\n");
			buffer.append(" <students>\n");
			for (int i = 0; i < students.length; i++)
			{
				buffer.append("  <enrolledStudent idRef=\"" + students[i].identity.id + "\"/>\n");
			}
			buffer.append(" </students>\n");
			buffer.append(" <teacher idRef=\"" + teacher.identity.id + "\"/>\n");
			buffer.append("</class>");
			
			writer.write(SchoolMember.valueWithIndent(indentLevel, buffer.toString()) + "\n\n");
		}
	}
}
