/**********************************************************************
 * Copyright (c) 2008 CA Inc. and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * CA - Initial API and implementation
 *
 * $Id: GraphResponseOutputter.java,v 1.12 2008/04/17 19:06:21 jtodd Exp $
 **********************************************************************/

package org.eclipse.cosmos.internal.dr.drs.service.outputter;
	
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringReader;
import java.util.Vector;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.muse.util.xml.XmlUtils;
import org.eclipse.cosmos.provisional.dr.drs.service.handler.common.AbstractOutputter;
import org.eclipse.cosmos.provisional.dr.drs.service.handler.common.IParameters;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class GraphResponseOutputter extends AbstractOutputter {

	public static final String XMLXTREAM="org.eclipse.cosmos.dr.drs.service.handler.xml.XMLStream";;
	
	private StringBuffer FilterBuffer = null;
	private StringBuffer DetailBuffer = null;
	private StringBuffer ARTBuffer = null;
	
	private InternalUtility ut = null;
	protected String rootId;

	public GraphResponseOutputter() {
		ut = new InternalUtility();
		FilterBuffer = new StringBuffer();
		DetailBuffer = new StringBuffer();
		ARTBuffer = new StringBuffer();
	}
	/* (non-Javadoc)
	 * @see org.eclipse.cosmos.dr.drs.service.handler.sml.IOutputter#render(java.io.PrintWriter, java.util.Map)
	 * Entry point from the UI
	 *
	 */
	public void render(PrintWriter output, IParameters input) throws Exception {
		String queryResponse = input.getParameter("queryResponse");
		Element response = deserialize(queryResponse);
			
		GraphResponseOutputter local = new GraphResponseOutputter();
		local.render(output, response);
	}

	protected Object getId(Object value){
		return idResolver.getId(value);
	}

	public void render(PrintWriter output, Element response) {
		StringBuffer buffer = new StringBuffer("{ items : [");
		buffer.append(renderElements(response));
		buffer.append("]}");
		output.println(buffer.toString());
	}
	
	public Element deserialize(String xmlString) throws Exception{
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		// Read the entire document into memory
		InputSource s = new InputSource(new StringReader(xmlString));  
		Document d = builder.parse(s);
		
		return d.getDocumentElement();
	}
	/* 
	 * Used by the entry point function called from the UI
	 * 
	 * Does not return anything, simply writes to the provided output stream
	 */
	public void render(PrintWriter pw, InputSource input) {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder builder = factory.newDocumentBuilder();
			// Read the entire document into memory
			Document d = builder.parse(input);
			
			Element el = d.getDocumentElement();
						
			StringBuffer buffer = new StringBuffer("{ items : [");
			buffer.append(renderElements(el));
			buffer.append("]}");
			pw.println(buffer.toString());
		} catch (ParserConfigurationException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (SAXException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	/* method to actually start doing the work, 
	 * takes an XML element 
	 * Adds bits bits for comboBox (Unique templateIds)
	 * top level table (Nodes and Edges)
	 * bottom level table (records)
	 * 
	 */
	
	protected StringBuffer renderElements(Element elem) {
		StringBuffer buffer = new StringBuffer();
		NodeList nodes = elem.getElementsByTagName("cmdbf:nodes");
		NodeList edges = elem.getElementsByTagName("cmdbf:edges");
				
		FilterBuffer.append("{ Filter: 'All' }");
			
		boolean b = ProcessNodes(buffer,nodes);
		if (b)
			buffer.append(",");
		b = ProcessEdges(buffer,edges);
		if (b) 
			buffer.append(",");
		buffer.append(FilterBuffer);
		buffer.append(DetailBuffer);
		buffer.append(ARTBuffer);
		return buffer;
	}
	
	/* 
	 * top level table - nodes from the XML query response
	 * walk through the nodelist, processing each node individually.
	 */
	
	protected boolean ProcessNodes(StringBuffer buffer,NodeList n) {
		int i = 0;
		Element elem = null;
		
		for (i=0;i<n.getLength();i++) {
		  Node node = n.item(i);
		  if (node.getNodeType() == Element.ELEMENT_NODE) {
			  elem = (Element) node;
			  String t = elem.getAttribute("templateId");
			  if (i>0) 
				  buffer.append(",");
			  else
				  FilterBuffer.append(",");
			  buffer.append(ProcessNode(elem,t));
		  }
		}
	
		return (n.getLength() > 0);
	}
	
	/* 
	 *  process an individual node, which all it does is walk through the item
	 *  list and proces each item individually.
	 */
	
	protected StringBuffer ProcessNode(Element n,String templateId)
	{
		StringBuffer buffer = new StringBuffer();
		NodeList items = n.getElementsByTagName("cmdbf:item");
		Element item = null;
		int i = 0;
		
		FilterBuffer.append("{ Filter: 'Node: " + templateId + "' },");
		for (i=0;i<items.getLength();i++) {
			Node node = items.item(i);
			if (node.getNodeType() == Element.ELEMENT_NODE) {
				item = (Element) node;
				if (i>0) buffer.append(",");
				buffer.append(ProcessItem(item,templateId));
			}
		}
		
		return buffer;
	}
	
	/* 
	 * process an item, because this is a node, there is no Edge or Target
	 * so there's just Source (processed by addNodeInstanceId()) for the top level table
	 * and the record (processed by addRecord()) for the bottom level table
	 */
	
	protected StringBuffer ProcessItem(Element item,String templateId) 
	{
		StringBuffer buffer = new StringBuffer();
		NodeList Idl = null;
		Node Id = null;
		NodeList records = null;
		Element record = null;
		NodeList arts = null;
		Element art = null;
		String tlidstr = null;
		String localId = null;
		int i = 0;
		boolean artdata = false;
		
		buffer.append(" { ");
		
		Idl = item.getElementsByTagName("cmdbf:instanceId");
		Id = Idl.item(0);
		if (Id.getNodeType() == Element.ELEMENT_NODE)
			localId = addNodeInstanceId(buffer,(Element) Id,templateId);
		buffer.append("Edge:'',Target:'',TLSID:'" + ut.getTLID() + "', TLTID:'',CosmosId:''");
		
		tlidstr = new String(templateId + ":" + localId);
		ut.addTLIDMap(tlidstr);
						
		addMdrId(buffer,(Element) Id);
		records = item.getElementsByTagName("cmdbf:record");
		if (records != null) {
			for (i=0;i<records.getLength();i++) {
				Node node = records.item(i);
				if (node.getNodeType() == Element.ELEMENT_NODE) {
					record = (Element) node;
					addRecord(record,false);
				}
			}
		}
		arts = item.getElementsByTagName("cmdbf:additionalRecordType");
		if (arts != null) {
			if (arts.getLength() == 0) 
				addArt(null,false);
			else {
				for (i=0;i<arts.getLength();i++) {
					Node node = arts.item(i);
					if (node.getNodeType() == Element.ELEMENT_NODE) {
						art = (Element) node;
						artdata = true;
						addArt(art,false);
					}
				}
			}
		}
		else
			addArt(null,false);
		
		ut.newTLID();
		if (artdata)
			buffer.append(", ARTData:'Y'}");
		else
			buffer.append(", ARTData:'N'}");
		return buffer;
	}
	
	/* 
	 *  process an art bit
	 */
	
	public void addArt(Element art,boolean edge)
	{
		NodeList Id = null;
		Node n = null;
		String tmp = null;
		
		ARTBuffer.append(",");
	
		ARTBuffer.append("{ NameSpace:'");
		if (art != null) {
			Id = art.getElementsByTagName("cmdbf:namespace");
			n = Id.item(0);
			tmp = XmlUtils.extractText((Element) n);
			ARTBuffer.append(tmp);
		}
		
		ARTBuffer.append("', LocalName:'");
		if (art != null) {
			Id = art.getElementsByTagName("cmdbf:localName");
			n = Id.item(0);
			tmp = XmlUtils.extractText((Element) n);
			ARTBuffer.append(tmp);
		}
		
		if (edge) 
			ARTBuffer.append("', TLSID:'', TLTID:'', CosmosId: '" + ut.getCosmosId() + "' ");
		else 
			ARTBuffer.append("', TLSID: '" + ut.getTLID() + "', TLTID:'', CosmosId:'' ");
		ARTBuffer.append("}");
	}
	
	/* 
	 * process a record, which includes a recordId, and some fields
	 * which are currently hard coded to be null - future enhancement to change this
	 * a cosmosId to link it with the top level table and the actual content
	 * which is processed by addContent()
	 */
	
	protected void addRecord(Element record,boolean edge)
	{
		NodeList Id = null;
		Node n = null;
		String tmp = null;
		
		if (DetailBuffer.length() > 0)
			DetailBuffer.append(",");
		DetailBuffer.append("{ recordId:");
		
		Id = record.getElementsByTagName("cmdbf:recordId");
		n = Id.item(0);
		tmp = XmlUtils.extractText((Element) n);
		DetailBuffer.append("'" + tmp + "', LastModified:'',BaseLineId:'', SnapShortId:'',Detail:'");
		
		addContent(record);
		if (edge) 
			DetailBuffer.append("', TLSID:'', TLTID:'', CosmosId: '" + ut.getCosmosId() + "' ");
		else 
			DetailBuffer.append("', TLSID: '" + ut.getTLID() + "', TLTID:'" + ut.getTLID() + "', CosmosId:'' ");
		DetailBuffer.append("}");
	}
	
	/* 
	 * process the node content which is raw XML
	 * does 2 things
	 * strips out the new lines from the string buffer that holds the generated XML text
	 * i.e. instead of 
	 * <student> 
	 *   Mike Lee
	 * </student>
	 * 
	 * its <student>Mike Lee</Student>
	 * 
	 * and translates < to &lt and > to &gr
	 */
	
	protected void addContent(Element record)
	{
		NodeList l = record.getChildNodes();
		StringBuffer nb = new StringBuffer();
		String tmp = null;
		int idx = 0;
		
		tmp = XmlUtils.toString((Element) l.item(1));
		idx = tmp.indexOf('>');
		idx += 2;
		nb.append(tmp.substring(idx));
		
		TransformXML(nb);
		
		DetailBuffer.append(nb);
	}
	private void TransformXML(StringBuffer nb)
	{
		int i = 0;
			
		char ltb[];
		char gtb[];
		char hash[];
		
		ltb = new char[4];
		gtb = new char[4];
		hash = new char[3];
		
		ltb[0] = '&';
		ltb[1] = 'l';
		ltb[2] = 't';
		ltb[3] = ';';
		
		gtb[0] = '&';
		gtb[1] = 'g';
		gtb[2] = 't';
		gtb[3] = ';';
		
		hash[0] = '%';
		hash[1] = '2';
		hash[2] = '3';
		
		for (i=0;i<nb.length();i++) {
			if (nb.charAt(i) == '<') {
				nb.deleteCharAt(i);
				nb.insert(i, ltb);
			}
			if (nb.charAt(i) == '#') {
				nb.deleteCharAt(i);
				nb.insert(i, hash);
			}
			if (nb.charAt(i) == '>') {
				if (i != nb.length()-1) {
  				  if (nb.charAt(i+1) == '\n') 
					nb.deleteCharAt(i+1);
				}
				nb.deleteCharAt(i);
				nb.insert(i, gtb);
			}
		}
		for (i=0;i<nb.length();i++) {
			if (nb.charAt(i) == '\n')
				nb.replace(i,i+1,"\\n");
            if (nb.charAt(i) == '\'') {
            	nb.insert(i,"\\");
            	i++;
            }
        }
		
	}
	
	/*
	 * adds localId (Source)
	 * also associates whatever is in templateId with the localId
	 * so there's a Map that says for example
	 * 'staff01' = 'Teachers'
	 * 
	 */
	protected String addNodeInstanceId(StringBuffer b,Element Id,String templateId) 
	{
		NodeList localId = null;
		Node l = null;
		String tmp = null;
						
		localId = Id.getElementsByTagName("cmdbf:localId");
		l = localId.item(0);
		tmp = XmlUtils.extractText((Element) l);
		b.append("Source:'" + templateId + ":" + tmp + "'");
		b.append(",");

		ut.addTemplateIdMap(templateId,tmp);
				
		return tmp;
	}
	
	/* 
	 * processes the edges, which walks the list of edges and processes each of those individually.
	 */
	
	protected StringBuffer ProcessEdge(Element e,String templateId) {
		NodeList rs = e.getElementsByTagName("cmdbf:relationship");
		Element r = null;
		int i = 0;
		StringBuffer buffer = new StringBuffer();
		
		FilterBuffer.append("{ Filter: 'Edge: " + templateId + "' },");
		for (i=0;i<rs.getLength();i++) {
			Node node = rs.item(i);
			if (node.getNodeType() == Element.ELEMENT_NODE) {
				r = (Element) node;
				if (i>0) buffer.append(",");
				buffer.append(ProcessRelationship(r,templateId));
			}
		}
		
		return buffer;
	}
	
	/*
	 * process an edge, which walks the list of relationships and processes each of thoes individually
	 */
	
	protected boolean ProcessEdges(StringBuffer b,NodeList e) {
		int i = 0;
		Element elem = null;
		
		
		for (i=0;i<e.getLength();i++) {
			Node node = e.item(i);
			if (node.getNodeType() == Element.ELEMENT_NODE) {
				elem = (Element) node;
			    String t = elem.getAttribute("templateId");
		   	    b.append(ProcessEdge(elem,t));
			}
		}
		return (e.getLength() > 0);
	}
	
	/* processes a relationship
	 * 
	 * a relationship has a Source, Target and a Record
	 * So each of these are processed in turn, the Source and Target are part of the top level table
	 * and the record is part of the bottom level table.
	 * 
	 * the addRecord() is the same method used by Node processing
	 * but the InstanceId is a little different as it has to retrieve info from the attrs map as oppose
	 * to put it in.
	 */
	
	protected StringBuffer ProcessRelationship(Element r,String templateId) {
		StringBuffer buffer = new StringBuffer();
		NodeList records = null;
		Element record = null;
		NodeList arts = null;
		Element art = null;
		NodeList sl = null;
		Node s = null;
		NodeList tl = null;
		Node t = null;
		int i = 0;
		String tlsid = null;
		String tltid = null;
		boolean artdata = false;
		
		buffer.append("{ ");
		
		sl = r.getElementsByTagName("cmdbf:source");
		s = sl.item(0);
		if (s.getNodeType() == Element.ELEMENT_NODE)
			tlsid = addEdgeInstanceId(buffer,(Element) s,"Source");
		
		buffer.append("Edge:'" + templateId + "',");
		
		tl = r.getElementsByTagName("cmdbf:target");
		t = tl.item(0);
		if (t.getNodeType() == Element.ELEMENT_NODE)
			tltid = addEdgeInstanceId(buffer,(Element) t,"Target");
				
		buffer.append("TLSID:'" + tlsid + "', TLTID:'" + tltid + "', CosmosId:'" + ut.getCosmosId() + "'");
		addMdrId(buffer,(Element) t);
		
		records = r.getElementsByTagName("cmdbf:record");

		if (records != null) {
			for (i=0;i<records.getLength();i++) {
				Node node = records.item(i);
				if (node.getNodeType() == Element.ELEMENT_NODE) {
					record = (Element) node;
					addRecord(record,true);
				}
			}
		}
		arts = r.getElementsByTagName("cmdbf:additionalRecordType");
		if (arts != null) {
			if (arts.getLength() == 0) 
				addArt(null,true);
			else {
				for (i=0;i<arts.getLength();i++) {
					Node node = arts.item(i);
					if (node.getNodeType() == Element.ELEMENT_NODE) {
						art = (Element) node;
						artdata = true;
						addArt(art,true);
					}
				}
			}
		}
		else
			addArt(null,true);
		
		ut.newCosmosId();			
		if (artdata)
			buffer.append(", ARTData:'Y'}");
		else
			buffer.append(", ARTData:'N'}");
		return buffer;
	}

	/* 
	 * This is a multi purpose method used by the edge processing methods
	 * it adds the localId 
	 * 
	 * prefix can be 'Source' or 'Target'
	 * c is a boolean indicated whether or not a comma is needed after the entry
	 *
	 * also retrieved the mapping so if it gets 'staff01' it can then say for example
	 * 'Source: teachers:staff01'
	 */
	
	protected String addEdgeInstanceId(StringBuffer b,Element Id,String prefix) 
	{
		NodeList localId = null;
		Node l = null;
		String tmp = null;
		String templateId = null;
		String tlidstr = null;

		localId = Id.getElementsByTagName("cmdbf:localId");
		l = localId.item(0);
		tmp = XmlUtils.extractText((Element) l);
		templateId = ut.FindByValue(tmp);
		
		tlidstr = new String(templateId + ":" + tmp);
		
		b.append(prefix + ":'" + tlidstr + "'");
		b.append(",");

		return ut.MapTLID(tlidstr);
	}
	
	/*
	 * adds MdrId found in the instanceId
	 * 
	 */
	
	protected void addMdrId(StringBuffer b,Element Id) 
	{
		NodeList mdrId = null;
		Node l = null;
		String tmp = null;
		
		mdrId = Id.getElementsByTagName("cmdbf:mdrId");
		l = mdrId.item(0);
		tmp = XmlUtils.extractText((Element) l);
		b.append(", MdrId:'" + tmp + "'");
	}

	
}

/*
 * This class facilitates the mapping between templateId/localId and their associated records
 * which are displayed separated in the graph response viewer widget
 * in brief, nodes key on the TLID and edges key on cosmosId
 */

class InternalUtility {
	private int _cosmosId = 1;
	private int _TLID = 1;
	
	private Vector _templateIds = null;
	private Vector _mapValues = null;
	private Vector _tlidm = null;
	private Vector _tlidv = null;

	public InternalUtility()
	{
		_cosmosId = 1;
		_TLID = 1;
		
		_templateIds = new Vector();
		_mapValues = new Vector();

		_tlidm = new Vector();
		_tlidv = new Vector();
	}
	
	int getCosmosId() { return _cosmosId; }
	void newCosmosId() { _cosmosId++; }
	
	int getTLID() { return _TLID; }
	void newTLID() { _TLID++; }
	
	/*
	 * lookup key by value
	 */
	
	public String FindByValue(String v) {
		return FindMap(v,_templateIds,_mapValues);
	}
	public String MapTLID(String v) {
		return FindMap(v,_tlidv,_tlidm);
	}
	
	private String FindMap(String v,Vector tv,Vector mv) {
		String t = null;
		String m = null;
		int i = 0;
		
		for (i=0;i<tv.size();i++) {
			t = (String) tv.get(i);
			m = (String) mv.get(i);
			
			if (m.equals(v)) 
				return t;
		}
		return null;
	}
	
	public void addTemplateIdMap(String templateId,String localId) 
	{
		_templateIds.add(templateId);
		_mapValues.add(localId);
	}
	
	public void addTLIDMap(String tlid)
	{
		_tlidm.add(tlid);
		_tlidv.add(String.valueOf(_TLID));
	}
	
	
}
