/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform;

import java.io.InputStream;
import java.io.Reader;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.input.artifacts.IDeregisterInputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.input.artifacts.IDeregisterRequest;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.TransformerException;
import org.xml.sax.InputSource;

/**
 * Convert input stream containing an XML-based deregistration request to and from a data structure
 * 
 * @provisional
 * @author David Whiteman
 */
public class DeregistrationInputTransformer {

	/**
	 * Transform an {@link IDeregisterRequest} into an InputStream containing the XML
	 * output for that object
	 * 
	 * @param request
	 * @return
	 */
	public static InputStream transform(IDeregisterRequest request) {
		return CMDBfServicesUtil.transformImpl((IXMLWritable) request);
	}

	/**
	 * Transform a stream of XML input containing a <deregisterRequest> structure to an IDeregisterRequest.
	 * 
	 * @param xmlInputStream
	 * @return {@link IDeregisterRequest}
	 * @throws TransformerException
	 */
	public static IDeregisterRequest transform(InputStream xmlInputStream) throws TransformerException 
	{
		return transform(xmlInputStream, null, null);
	}
	
	/**
	 * Transform a stream of XML input containing a <deregisterRequest> structure to an IDeregisterRequest.
	 * 
	 * @param xmlInputStream
	 * @return {@link IDeregisterRequest}
	 * @throws TransformerException
	 */
	public static IDeregisterRequest transform(InputStream xmlInputStream, IDeregisterInputArtifactFactory artifactFactory) throws TransformerException 
	{
		return transform(xmlInputStream, artifactFactory, null);
	}

	/**
	 * Transform a Reader containing XML beginning with a <deregisterRequest> structure to an IDeregisterRequest.
	 * 
	 * @param reader
	 * @return {@link IDeregisterRequest}
	 * @throws TransformerException
	 */
	public static IDeregisterRequest transform(Reader reader) throws TransformerException 
	{
		return transform(reader, null, null);
	}

	public static IDeregisterRequest transform(Reader reader, IDeregisterInputArtifactFactory artifactFactory) throws TransformerException 
	{
		return transform(reader, artifactFactory, null);
	}

	/**
	 * Transform a stream of XML input containing a <deregisterRequest> structure to an IDeregisterRequest.
	 * Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param xmlInputStream
	 * @param transformerOptions
	 * @return {@link IDeregisterRequest}
	 * @throws TransformerException
	 */
	public static IDeregisterRequest transform(InputStream xmlInputStream, IDeregisterInputArtifactFactory artifactFactory, Map<String, Object> transformerOptions) throws TransformerException 
	{
		return transformerOptions == null ? 
				(IDeregisterRequest) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new DeregistrationInputStreamHandler(artifactFactory)) :
				(IDeregisterRequest) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new DeregistrationInputStreamHandler(artifactFactory), transformerOptions);	
	}
	
	/**
	 * Transform a stream of XML input containing a <deregisterRequest> structure to an IDeregisterRequest.
	 * Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param reader
	 * @param transformerOptions
	 * @return {@link IDeregisterRequest}
	 * @throws TransformerException
	 */
	public static IDeregisterRequest transform(Reader reader, IDeregisterInputArtifactFactory artifactFactory, Map<String, Object> transformerOptions) throws TransformerException 
	{
		return transformerOptions == null ? 
				(IDeregisterRequest) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new DeregistrationInputStreamHandler(artifactFactory)) :
				(IDeregisterRequest) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new DeregistrationInputStreamHandler(artifactFactory), transformerOptions);	
	}

}
