/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.registration.client;

import java.io.StringReader;
import java.io.StringWriter;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axis2.Constants;
import org.apache.axis2.addressing.EndpointReference;
import org.apache.axis2.client.Options;
import org.apache.axis2.client.ServiceClient;
import org.apache.axis2.util.XMLUtils;
import org.eclipse.cosmos.dc.cmdbf.exception.CMDBfException;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.DeregistrationOutputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.input.artifacts.IDeregisterRequest;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.output.artifacts.IDeregisterResponse;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.RegistrationOutputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.IRegisterRequest;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IRegisterResponse;
import org.eclipse.cosmos.dc.cmdbf.services.transform.TransformerException;
import org.w3c.dom.Element;

public class RegistrationServiceClient {

	private EndpointReference serviceEPR = null;
	
	public RegistrationServiceClient(String epr) {
		serviceEPR = new EndpointReference(epr);
	}
	
	public RegistrationServiceClient(EndpointReference epr) {
		serviceEPR = epr;
	}

	/**
	 * Provides the ability to send a registration request to a 
	 * federating CMDB.
	 * 
	 * @param registrationRequest The registration request as it conforms
	 * to the CMDBf specification
	 * 
	 * @return An element representing the CMDBf registration response, conforming
	 * to the CMDBf specification
	 * 
	 * @throws DataManagerException In case of any unexpected error
	 */
	public Element register(Element registrationRequest) throws CMDBfException 
	{
		try {
			OMElement request = XMLUtils.toOM(registrationRequest);
			OMElement result = register(request);
			return XMLUtils.toDOM(result);
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}
	
	public IRegisterResponse register(IRegisterRequest registrationRequest) throws CMDBfException {
		try {
			StringWriter writer = new StringWriter();
			registrationRequest.toXML(writer, 0);
			StringReader regReader = new StringReader(writer.toString());
			
			//create the parser
			XMLStreamReader parser = XMLInputFactory.newInstance().createXMLStreamReader(regReader);
			//create the builder
			StAXOMBuilder builder = new StAXOMBuilder(parser);
	
			//get the root element
			OMElement requestPayload = builder.getDocumentElement();
			
			OMElement result = register(requestPayload);

	        IRegisterResponse resultObj = RegistrationOutputTransformer.transform(new StringReader(result.toString()));
	        
	        return resultObj;
		} catch (XMLStreamException e) {
			throw new CMDBfException(e);
		} catch (TransformerException te) {
			throw new CMDBfException(te);
		}
	}
	
	public OMElement register(OMElement registrationRequest) throws CMDBfException {
		try {
	        Options options = new Options();
	        options.setTo(serviceEPR);
	        options.setTransportInProtocol(Constants.TRANSPORT_HTTP);
	
	        ServiceClient sender = new ServiceClient();
	        sender.setOptions(options);
	        OMElement result = sender.sendReceive(registrationRequest);
	        return result;
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}
	
	/**
	 * Provides the ability to send a deregistration request to a 
	 * federating CMDB.
	 * 
	 * @param deregistrationRequest The deregistration request as it conforms
	 * to the CMDBf specification
	 * 
	 * @return An element representing the CMDBf deregistration response, conforming
	 * to the CMDBf specification
	 * 
	 * @throws DataManagerException In case of any unexpected error
	 */
	public Element deregister(Element deregistrationRequest) throws CMDBfException 
	{
		try {
			OMElement request = XMLUtils.toOM(deregistrationRequest);
			OMElement result = deregister(request);
			return XMLUtils.toDOM(result);
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}
	
	public IDeregisterResponse deregister(IDeregisterRequest deregistrationRequest) throws CMDBfException {
		try {
			StringWriter writer = new StringWriter();
			deregistrationRequest.toXML(writer, 0);
			StringReader reader = new StringReader(writer.toString());
			
			//create the parser
			XMLStreamReader parser = XMLInputFactory.newInstance().createXMLStreamReader(reader);
			//create the builder
			StAXOMBuilder builder = new StAXOMBuilder(parser);
	
			//get the root element
			OMElement requestPayload = builder.getDocumentElement();
			
			OMElement result = deregister(requestPayload);

	        IDeregisterResponse resultObj = DeregistrationOutputTransformer.transform(new StringReader(result.toString()));
	        
	        return resultObj;
		} catch (XMLStreamException e) {
			throw new CMDBfException(e);
		} catch (TransformerException te) {
			throw new CMDBfException(te);
		}

	}
	
	public OMElement deregister(OMElement deregistrationRequest) throws CMDBfException {
		try {
	        Options options = new Options();
	        options.setTo(serviceEPR);
	        options.setTransportInProtocol(Constants.TRANSPORT_HTTP);
	
	        ServiceClient sender = new ServiceClient();
	        sender.setOptions(options);
	        OMElement result = sender.sendReceive(deregistrationRequest);
	        return result;
		} catch (Exception e) {
			throw new CMDBfException(e);
		}
	}
	
	
}
