/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.registration.ws;

import java.io.InputStream;
import java.io.StringReader;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.cmdbf.services.common.INotificationHandlerFactory;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.service.ICMDBfDeregistrationOperation;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.service.impl.CMDBfDeregistrationOperation;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.DeregistrationOutputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.output.artifacts.IDeregisterResponse;
import org.eclipse.cosmos.dc.cmdbf.services.registration.service.ICMDBfRegistrationOperation;
import org.eclipse.cosmos.dc.cmdbf.services.registration.service.impl.CMDBfRegistrationOperation;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.RegistrationOutputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IRegisterResponse;

public abstract class AbstractRegistrationService {

	/**
	 * The registration operation
	 */
	private ICMDBfRegistrationOperation registrationOperation;
	
	/**
	 * The deregistration operation
	 */
	private ICMDBfDeregistrationOperation deregistrationOperation;

	/**
	 * Default constructor
	 */
	public AbstractRegistrationService()
	{
		INotificationHandlerFactory factory = getNotificationHandlerFactory();
		registrationOperation = new CMDBfRegistrationOperation(factory, null);
		deregistrationOperation = new CMDBfDeregistrationOperation(factory, null);
	}

	public OMElement Register(OMElement registerRequest) throws CMDBfServiceException {
		
		registerRequest.build();
		registerRequest.detach();
		
		OMElement documentElement = null;
		IRegisterResponse result = null;
		
		try {							
			result = registrationOperation.execute(new StringReader(registerRequest.toString())); 
		} catch (CMDBfServiceException e) {
			// TODO: log
			throw e;
		}

		InputStream output = RegistrationOutputTransformer.transform(result);
		XMLStreamReader parser = null;
		try {
			parser = XMLInputFactory.newInstance().createXMLStreamReader(output);
		} catch (XMLStreamException e) {
			// TODO: handle exception
		}
		//create the builder
		StAXOMBuilder builder = new StAXOMBuilder(parser);
		//get the root element
		documentElement =  builder.getDocumentElement();
		return documentElement;
	}

	public OMElement Deregister(OMElement deregisterRequest) throws CMDBfServiceException {
		
		deregisterRequest.build();
		deregisterRequest.detach();
		
		OMElement documentElement = null;
		IDeregisterResponse result = null;
		
		try {							
			result = deregistrationOperation.execute(new StringReader(deregisterRequest.toString())); 
		} catch (CMDBfServiceException e) {
			// TODO: log
			throw e;
		}

		InputStream output = DeregistrationOutputTransformer.transform(result);
		XMLStreamReader parser = null;
		try {
			parser = XMLInputFactory.newInstance().createXMLStreamReader(output);
		} catch (XMLStreamException e) {
			// TODO: handle exception
		}
		//create the builder
		StAXOMBuilder builder = new StAXOMBuilder(parser);
		//get the root element
		documentElement =  builder.getDocumentElement();
		return documentElement;
	}

	/**
	 * This method is expected to be implemented by subclasses.
	 * The method returns a notification handler factory that is needed
	 * for the registration/deregistration operations
	 * 
	 * @return The notification handler factory
	 */
	public abstract INotificationHandlerFactory getNotificationHandlerFactory();

	/**
	 * @return the registrationOperation
	 */
	public ICMDBfRegistrationOperation getRegistrationOperation() {
		return registrationOperation;
	}

	/**
	 * @return the deregistrationOperation
	 */
	public ICMDBfDeregistrationOperation getDeregistrationOperation() {
		return deregistrationOperation;
	}

}
