/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.broker.ws.persistence.internal;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.eclipse.cosmos.dc.broker.ws.datamodel.Group;
import org.eclipse.cosmos.dc.broker.ws.datamodel.Groups;
import org.eclipse.cosmos.dc.broker.ws.datamodel.Service;
import org.eclipse.cosmos.dc.broker.ws.datamodel.Services;
import org.eclipse.cosmos.dc.broker.ws.persistence.BrokerDatabase;

public class BrokerDatabaseImpl implements BrokerDatabase {
	
//	private final String databaseLocation = "database/brokerdb.xml";
//	private ClassLoader loader;
//	private OMElement docRoot = null;

	private static File databaseDir = null;
	private static File databaseFile = null;
	private static Groups groups = null;
	private static Object writeLock = new Object();
	
	private static BrokerDatabaseImpl instance = new BrokerDatabaseImpl(); 
	
	public static BrokerDatabaseImpl getInstance() {
		return instance;
	}

	private BrokerDatabaseImpl() {
	}
	
	public void init(Map<String, Object> initMap) {
		String dbdir = initMap.get("dbdir").toString();
		setDatabaseDir(new File(dbdir));
	}
	
	private void setDatabaseDir(File dbdir) {
		if (dbdir.exists() == false) {
			// TODO: log.
			dbdir.mkdir();
		}
		databaseDir = dbdir;
	}
	
	public Groups getAllGroups() {
		if (groups == null) {
			loadxml();
		}
		return groups;
	}
	
	/**
	 * Get all groups that include a service with the given EPR.
	 */
	public Groups getGroupsByNamespace(String ns) {
		if (groups == null) {
			loadxml();
		}
		
		Groups result = new Groups();
		List<Group> groupList = new ArrayList<Group>();
		result.setGroups(groupList);
		
		for (Group group : groups.getGroups()) {
			Services services = group.getServices();
			if (services != null) {
				List<Service> serviceList = services.getServices();
				if (serviceList != null) {
					for (Service service : serviceList) {
						if (service.getNamespace().equals(ns)){
							groupList.add(group);
							break;
						}
					}
				}
			}
		}
		return result;
	}
	
	public Group getGroup(String hostname, String groupName) {
		if (groups == null) {
			loadxml();
		}
		for (Group group : groups.getGroups()) {
			if (group.getHost().equals(hostname) && 
					group.getName().equals(groupName)) {
				return group;
			}
		}
		return null;
	}
	
	public void addGroup(Group newgroup) {
		if (groups == null) {
			loadxml();
		}
		synchronized(writeLock) {
			// Check if a group with the same ID already exist.
			// if so, throw Duplicated GroupIDException
			String host = newgroup.getHost();
			String name = newgroup.getName();
			List<Group> groupList = groups.getGroups();
			for (Group group : groupList) {
				if (group.getHost().equals(host) && 
						group.getName().equals(name)) {
					// TODO: throw exception
					System.out.println("Service has already been registered at this broker.  No new brokers are registered. ");
					return;
				}
			}
			groupList.add(newgroup);
			save();
		}
	}
	
	public void deleteGroup(String hostname, String groupName) {
		if (groups == null) {
			loadxml();
		}
		// find group with the specified ID and delete the group
		boolean exist = false;
		synchronized (writeLock) {
			List<Group> groupList = groups.getGroups();
			for (Group group : groupList) {
				if (group.getHost().equals(hostname) && 
						group.getName().equals(groupName)) {
					groupList.remove(group);
					exist = true;
					break;
				}
			}
			save();
		}
		if (exist == false) {
			// TODO: throw exception: Group with name %s at %s is not registered at this broker. No groups are deleted.
		}
	}
	
	private void loadxml() {
		if (databaseDir == null) {
			// TODO: throw exception: Database directory not set.
		}
		
		if (databaseFile == null) {
			databaseFile = new File(databaseDir, "brokerdb.xml");
			if (databaseFile.exists() == false) {
				// Create database XML if it does not exist in database directory
				try {
					databaseFile.createNewFile();
				} catch (IOException e) {
					// TODO: handle exception
				}
				groups = new Groups();
				save();
				return;
			}
		}

		try {
	    	InputStream stream = new FileInputStream(databaseFile);
	    	//create the parser
	    	XMLStreamReader parser = XMLInputFactory.newInstance().createXMLStreamReader(stream);
	    	//create the builder
	    	StAXOMBuilder builder = new StAXOMBuilder(parser);
	
	    	//get the root element (in this case the envelope)
	    	OMElement docRoot = builder.getDocumentElement();
	    	groups = Groups.getGroups(docRoot);
		} catch (Exception e) {
			// TODO: log here
		}
	}
	
	private synchronized void save() {
		try {
			FileWriter fw = new FileWriter(databaseFile);
			
			OMElement om = groups.toOM();
			XMLStreamWriter writer = XMLOutputFactory.newInstance().createXMLStreamWriter(fw);
			om.serialize(writer);
			writer.flush();
			writer.close();
		} catch (Exception e) {
			// TODO: throw exception
			e.printStackTrace();
		}
	}
	
}
