/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.mdr.registration;

import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.muse.util.xml.XmlUtils;
import org.apache.muse.ws.addressing.EndpointReference;
import org.apache.muse.ws.addressing.WsaConstants;
import org.apache.muse.ws.dm.muws.MuwsConstants;
import org.eclipse.cosmos.dc.broker.client.BrokerClient;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.INotificationHandlerFactory;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.service.ICMDBfDeregistrationOperation;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.service.impl.CMDBfDeregistrationOperation;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.DeregistrationInputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.DeregistrationOutputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.input.artifacts.DeregisterInputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.input.artifacts.IDeregisterInputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.input.artifacts.IDeregisterRequest;
import org.eclipse.cosmos.dc.cmdbf.services.deregistration.transform.output.artifacts.IDeregisterResponse;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.IQueryTransformerConstants;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.QueryOutputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IQueryResult;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IQueryServiceElementCollection;
import org.eclipse.cosmos.dc.cmdbf.services.registration.service.ICMDBfRegistrationOperation;
import org.eclipse.cosmos.dc.cmdbf.services.registration.service.impl.CMDBfRegistrationOperation;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.IRegistrationTransformerConstants;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.RegistrationInputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.RegistrationOutputTransformer;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.IRegisterRequest;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.IRequest;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.input.artifacts.RegisterInputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.registration.transform.output.artifacts.IRegisterResponse;
import org.eclipse.cosmos.dc.cmdbf.services.transform.TransformerException;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IInstanceId;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IItem;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IRelationship;
import org.eclipse.cosmos.dc.management.domain.client.ManagementDomainClient;
import org.eclipse.cosmos.dc.mdr.api.IMdrQuery;
import org.eclipse.cosmos.dc.mdr.client.MdrClient;
import org.eclipse.cosmos.dc.mdr.exception.CMDBfException;
import org.eclipse.cosmos.dc.mdr.impl.AbstractMdr;
import org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.EndPointReferenceSerializer;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.MDRIdentifierSerializer;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.SelectiveRegistrationInputSerializer;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.SerializerUtil;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.EndPointReferenceSerializer.MDRReference;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.SelectiveRegistrationInputSerializer.SelectiveRegistrationRequest;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;


/**
 * An abstract implementation of {@link IFederatingCMDB} that adopters
 * can use when implementing a federating CMDB.  Optionally, adopters
 * can provide a direct implementation of IFederatingCMDB
 * 
 * @provisional
 * @author Ali Mehregani
 */
public abstract class AbstractFederatingCMDB extends AbstractMdr implements IFederatingCMDB
{	
	/**
	 * The registration operation
	 */
	private ICMDBfRegistrationOperation registrationOperation;
	
	/**
	 * The deregistration operation
	 */
	private ICMDBfDeregistrationOperation deregistrationOperation;
	
	
	/**
	 * Default constructor
	 */
	public AbstractFederatingCMDB()
	{
		INotificationHandlerFactory factory = getNotificationHandlerFactory();
		registrationOperation = new CMDBfRegistrationOperation(factory, null);
		deregistrationOperation = new CMDBfDeregistrationOperation(factory, null);
	}
	
	
	/**
	 * This method is expected to be implemented by subclasses.
	 * The method returns a notification handler factory that is needed
	 * for the registration/deregistration operations
	 * 
	 * @return The notification handler factory
	 */
	public abstract INotificationHandlerFactory getNotificationHandlerFactory();
	
	
	/**
	 * @see org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB#register(org.w3c.dom.Element)
	 */
	public Element register(Element registrationRequest) throws CMDBfException 
	{		
		return register(toInputStream(registrationRequest, IRegistrationTransformerConstants.REGISTER_REQUEST_ELEMENT));
	}

	
	private Element register (InputStream registrationRequest) throws CMDBfException
	{
		try 
		{							
			IRegisterResponse result = getRegistrationOperation().execute(registrationRequest); 
			InputStream output = RegistrationOutputTransformer.transform(result);
			
			return DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(output).getDocumentElement();
		} 
		catch (Exception e) 
		{
			handleException(e);
		}
		return null;
		
	}
	
	/**
	 * @see org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB#register(java.lang.String[])
	 */
	public Element registerMDREntries(Element mdrReferences) throws CMDBfException
	{
		return registerMDREntries(EndPointReferenceSerializer.getInstance().fromXML(mdrReferences));
	}

	
	private Element registerMDREntries(MDRReference[] mdrIdentifiers) throws CMDBfException
	{
		try
		{
			InputStream is = AbstractFederatingCMDB.class.getClassLoader().getResourceAsStream("query-all.txt");
			Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(is);
			Element queryRequest = XmlUtils.getDocumentRoot(document);
			return registerSelectiveMDREntries (mdrIdentifiers, queryRequest);
		}
		catch (Exception e)
		{
			handleException(e);
		}
		
		return null;
	}
	
	/**
	 * @see org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB#registerSelectiveMDREntries(org.w3c.dom.Element)
	 */
	public Element registerSelectiveMDREntries(Element selectiveRegistration) throws CMDBfException
	{		
		SelectiveRegistrationRequest request = SelectiveRegistrationInputSerializer.getInstance().fromXML(selectiveRegistration);
		return registerSelectiveMDREntries(request.getMDRReferences(), request.getQuery());
	}
	

	private Element registerSelectiveMDREntries(MDRReference[] mdrIdentifiers, Element queryRequest) throws CMDBfException
	{
		String currentMDRId = null;
		
		try
		{
			IRegisterRequest registerRequest = RegisterInputArtifactFactory.getInstance().createRegisterRequest();			
			
			// For each MDR
			for (int i = 0; i < mdrIdentifiers.length; i++)
			{				
				// Contact the MDR				
				EndpointReference epr = mdrIdentifiers[i].getEpr();
				if (epr == null)
				{
					epr = new EndpointReference(new URI(mdrIdentifiers[i].getEndPointReferenceURI()));
					epr.addParameter(WsaConstants.DEFAULT_RESOURCE_ID_QNAME, mdrIdentifiers[i].getDefaultResourceId());
				}
				MdrClient mdrClient = new MdrClient(epr);
				
				// Submit query to MDR				
	            Element queryResponse = mdrClient.query(queryRequest);
	            if (queryResponse == null)
	            {
	            	continue;
	            }
	            
	            // Construct registration request from query response
	            createRegisterRequest(registerRequest, queryResponse);       	           	       	           	        
			}
			
			// Set a dummy mdr id
			registerRequest.setMdrId(new URI(""));
			
			// Register items/relationships
			return register(RegistrationInputTransformer.transform(registerRequest));         		
		}
		catch (Exception e)
		{
			// TODO: Log
			System.err.println("There was an error handling the registration for MDR: " + currentMDRId);
			e.printStackTrace();
			throw new CMDBfException("There was an error handling the registration request", e);
		} 		
	}
	
	
	/**
	 * @see org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB#superSetRegistration()
	 */
	public Element registerAll(Element managementDomainEPR) throws CMDBfException
	{
		try
		{
			Element rootNode = SerializerUtil.findNode(managementDomainEPR, IFederatingCMDB.MANAGEMENT_DOMAIN_ELEMENT);
			String mgtDomainEPR = rootNode == null ? null : rootNode.getTextContent();
			if (mgtDomainEPR == null)
			{
				throw new CMDBfException("The argument representing a management domain EPR is invalid");
			}
			
			// Contact the management domain
			ManagementDomainClient managementDomainClient= new ManagementDomainClient(new EndpointReference(new URI(mgtDomainEPR)));
			String[] brokers = managementDomainClient.getBrokers();
			
			// For every broker
			List<MDRReference> mdrEndPoints = new ArrayList<MDRReference>();
			for (int i = 0; i < brokers.length; i++)
			{								
				// Retrieve the MDR end points
				BrokerClient brokerClient = new BrokerClient(new EndpointReference(XmlUtils.createDocument(brokers[i]).getDocumentElement()));
				String[] endPoints = brokerClient.getServices(MuwsConstants.MANAGEABILITY_CAPABILITY_QNAME, IMdrQuery.NAMESPACE_URI);				
				for (int j = 0; j < endPoints.length; j++)
				{
					MDRReference entry = new MDRReference(new EndpointReference(XmlUtils.createDocument(endPoints[j]).getDocumentElement()));
					mdrEndPoints.add(entry);
				}
			}
			
			// Perform the registration
			return registerMDREntries(mdrEndPoints.toArray(new MDRReference[mdrEndPoints.size()]));
		}
		catch (Exception e)
		{
			handleException(e);
		}
		
		return null;
	}
	
	
	private void createRegisterRequest(IRegisterRequest registerRequest, Element queryResponse) throws CMDBfException, TransformerException
	{        
		InputStream inputStream = toInputStream(cleanseResponse(queryResponse));
		IQueryResult queryResult = QueryOutputTransformer.transform(inputStream);
		
		addCollection (queryResult.getAllNodes(), registerRequest, true);
		addCollection (queryResult.getAllEdges(), registerRequest, false);		
	}
	
	
	/**
	 * This method should not be needed.  It is currently here as
	 * a workaround to the following defect: https://bugs.eclipse.org/bugs/show_bug.cgi?id=217141
	 * 
	 * @param queryResponse The query response
	 * @return Cleansed version of the response
	 */
	private Element cleanseResponse(Element queryResponse)
	{
		Element cleansedRoot = XmlUtils.createElement(new QName(ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE, IQueryTransformerConstants.QUERY_RESULT_ELEMENT, ICMDBfServicesConstants.CMDBF_PREFIX_RAW));
		Document document = cleansedRoot.getOwnerDocument();

		NodeList children = queryResponse.getChildNodes();
		int counter = 0;
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node child = children.item(counter);			
			Node adoptedNode = document.adoptNode(child);
			if (adoptedNode == null)
			{
				adoptedNode = document.importNode(child, true);
				counter++;
			}
			cleansedRoot.appendChild(adoptedNode);
		}		
				
		return cleansedRoot;
	}


	private void addCollection(IQueryServiceElementCollection[] collection, IRequest registerRequest, boolean isNode)
	{
		for (int i = 0; i < collection.length; i++)
		{
			IGraphElement[] graphElements = collection[i].getElements();
			for (int j = 0; j < graphElements.length; j++)
			{
				if (isNode)
				{				
					registerRequest.addItem((IItem)graphElements[j]);
				}
				else
				{
					registerRequest.addRelationship((IRelationship)graphElements[j]);
				}		
			}				
		}
	}
	
	
	/**
	 * @see org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB#deregister(org.w3c.dom.Element)
	 */
	public Element deregister(Element deregistrationRequest) throws CMDBfException
	{
		return processDeregisterRequest(deregistrationRequest);
	}
	
	
	/**
	 * A CMDBf query can't be constructed to only return items/relationships belonging
	 * to an MDR.  It's also expensive to query for all items/relationships registered
	 * with the federating CMDB and filtering out the ones matching an MDR ID.  This implementation
	 * submits a deregistration request with items/relationships that have an
	 * instanceId, where mdrId = mdrIdentifier[i] and localId = '*'
	 * 
	 * @see org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB#deregister(java.lang.String[])
	 */
	public Element deregisterMDREntries(Element mdrReferences) throws CMDBfException
	{		
		String[] mdrIds = MDRIdentifierSerializer.getInstance().fromXML(mdrReferences);
		IDeregisterInputArtifactFactory artifactFactory = DeregisterInputArtifactFactory.getInstance();
		IDeregisterRequest deregisterRequest = artifactFactory.createDeregisterRequest();
		try
		{
			deregisterRequest.setMdrId(new URI(""));
		} 
		catch (URISyntaxException e)
		{
			// Unlikely
			e.printStackTrace();
		}
		
		for (int i = 0; i < mdrIds.length; i++)
		{
			IInstanceId instanceId = artifactFactory.createInstanceId();
			instanceId.setMdrId(mdrIds[i]);
			instanceId.setLocalId(IFederatingCMDBServerConstants.ALL_ENTITIES);
			
			IItem item = artifactFactory.createItem();
			item.addInstanceId(instanceId);
			
			IRelationship relationship = artifactFactory.createRelationship();
			relationship.addInstanceId(instanceId);
			
			deregisterRequest.addItem(item);
			deregisterRequest.addRelationship(relationship);			
		}		
		
		return processDeregisterRequest(DeregistrationInputTransformer.transform(deregisterRequest));
	}


	/**
	 * @see org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB#deregister(java.io.InputStream)
	 */
	public Element deregisterSelectiveEntries(Element queryRequest) throws CMDBfException
	{
		try
		{
			IQueryResult queryResult = getQueryOperation().execute(toInputStream((SerializerUtil.findNode(queryRequest, IQueryTransformerConstants.QUERY_ELEMENT))));			
			IDeregisterRequest request = DeregisterInputArtifactFactory.getInstance().createDeregisterRequest();
			addCollection(queryResult.getAllNodes(), request, true);
			addCollection(queryResult.getAllEdges(), request, true);
			
			request.setMdrId(new URI(""));
			return processDeregisterRequest(DeregistrationInputTransformer.transform(request));			
		} 
		catch (Exception e)
		{
			handleException(e);
		} 
		
		return null;
	}
	
	
	/**
	 * Similar to {@link this#deregister(String[])}, the following method
	 * submits a deregistration request with mdrId and localId equaling to '*'
	 * 
	 * @see org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB#superSetDeregistration()
	 */
	public Element deregisterAll() throws CMDBfException
	{
		IDeregisterInputArtifactFactory artifactFactory = DeregisterInputArtifactFactory.getInstance();
		IDeregisterRequest request = artifactFactory.createDeregisterRequest();
		
		try
		{
			request.setMdrId(new URI(""));
		}
		catch (URISyntaxException e)
		{
			e.printStackTrace();
			return null;
		}
		
		IInstanceId instanceId = artifactFactory.createInstanceId();
		instanceId.setMdrId(IFederatingCMDBServerConstants.ALL_ENTITIES);
		instanceId.setLocalId(IFederatingCMDBServerConstants.ALL_ENTITIES);
		
		IItem item = artifactFactory.createItem();
		item.addInstanceId(instanceId);
		request.addItem(item);
		
		IRelationship relationship = artifactFactory.createRelationship();
		relationship.addInstanceId(instanceId);
		request.addRelationship(relationship);
		
		return processDeregisterRequest(DeregistrationInputTransformer.transform(request));
	}

	/**
	 * @return the registrationOperation
	 */
	protected ICMDBfRegistrationOperation getRegistrationOperation()
	{
		return registrationOperation;
	}


	/**
	 * @param registrationOperation the registrationOperation to set
	 */
	protected void setRegistrationOperation(ICMDBfRegistrationOperation registrationOperation)
	{
		this.registrationOperation = registrationOperation;
	}


	/**
	 * @return the deregistrationOperation
	 */
	protected ICMDBfDeregistrationOperation getDeregistrationOperation()
	{
		return deregistrationOperation;
	}


	/**
	 * @param deregistrationOperation the deregistrationOperation to set
	 */
	protected void setDeregistrationOperation(ICMDBfDeregistrationOperation deregistrationOperation)
	{
		this.deregistrationOperation = deregistrationOperation;
	}

	
	private void handleException(Exception e) throws CMDBfException
	{
		if (e instanceof CMDBfException)
		{
			throw (CMDBfException)e;
		}
		
		// TODO: Log
		e.printStackTrace();
		throw new CMDBfException(e);
	}
	
	
	private Element processDeregisterRequest(Object deregistrationRequest) throws CMDBfException
	{
		try
		{						
			InputStream request = deregistrationRequest instanceof Element ? toInputStream((Element)deregistrationRequest, IRegistrationTransformerConstants.DEREGISTER_REQUEST_ELEMENT) : (InputStream)deregistrationRequest;
			IDeregisterResponse result = getDeregistrationOperation().execute(request);
			InputStream output = DeregistrationOutputTransformer.transform(result);
			
			return DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(output).getDocumentElement();
		} 
		catch (Exception e) 
		{
			handleException(e);
		}
		return null;
	}
}
