/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.internal.query.transform.input.artifacts;

import java.io.StringWriter;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.IQueryTransformerConstants;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IConstraint;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IPrefixMapping;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IXPathExpression;

/**
 * Constraint that allows queries using an XPath expression
 * 
 * @author David Whiteman
 */
public class XPathExpression implements IXPathExpression {
	private String expression;
	private List<IPrefixMapping> prefixMappings = new ArrayList<IPrefixMapping>();
	private URI dialect;

	public XPathExpression(URI dialect) {
		super();
		this.dialect = dialect;
	}

	public String getExpression() {
		return expression;
	}

	public void setExpression(String xpathExpression) {
		this.expression = xpathExpression;
	}

	public IPrefixMapping[] getPrefixMappings() {
		return prefixMappings.toArray(new IPrefixMapping[prefixMappings.size()]);
	}

	public boolean addPrefixMapping(IPrefixMapping mapping) {
		return prefixMappings.add(mapping);
	}

	/**
	 * @see org.eclipse.cosmos.dc.cmdbf.artifacts.input.IConstraint#getType()
	 */
	public int getType()
	{
		return IConstraint.XPATH_EXPRESSION;
	}

	public void toXML(StringWriter writer, int indent) {
		CMDBfServicesUtil.addIndent(writer, indent);
		writer.write(ICMDBfServicesConstants.OPEN_ANGLE_BRACKET+ICMDBfServicesConstants.CMDBF_PREFIX + IQueryTransformerConstants.XPATH_EXPRESSION_ELEMENT);
		writer.write(ICMDBfServicesConstants.SINGLE_SPACE);
		CMDBfServicesUtil.writeAttribute(writer, IQueryTransformerConstants.DIALECT_ATTRIBUTE, dialect.toString());
		writer.write(ICMDBfServicesConstants.CLOSE_ANGLE_BRACKET + ICMDBfServicesConstants.nl);

		CMDBfServicesUtil.outputListToXML(writer, indent+1, prefixMappings);

		CMDBfServicesUtil.addIndent(writer, indent+1);
		writer.write(CMDBfServicesUtil.beginTagFor(ICMDBfServicesConstants.CMDBF_PREFIX + IQueryTransformerConstants.EXPRESSION_ELEMENT));
		writer.write(getExpression());
		writer.write(CMDBfServicesUtil.endTagFor(ICMDBfServicesConstants.CMDBF_PREFIX + IQueryTransformerConstants.EXPRESSION_ELEMENT));
		writer.write(ICMDBfServicesConstants.nl);

		CMDBfServicesUtil.addIndent(writer, indent);
		writer.write(CMDBfServicesUtil.endTagFor(ICMDBfServicesConstants.CMDBF_PREFIX + IQueryTransformerConstants.XPATH_EXPRESSION_ELEMENT)+ICMDBfServicesConstants.nl);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((dialect == null) ? 0 : dialect.hashCode());
		result = prime * result
				+ ((prefixMappings == null) ? 0 : prefixMappings.hashCode());
		result = prime * result
				+ ((expression == null) ? 0 : expression.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final XPathExpression other = (XPathExpression) obj;
		if (dialect == null) {
			if (other.dialect != null)
				return false;
		} else if (!dialect.equals(other.dialect))
			return false;
		if (prefixMappings == null) {
			if (other.prefixMappings != null)
				return false;
		} else if (!prefixMappings.equals(other.prefixMappings))
			return false;
		if (expression == null) {
			if (other.expression != null)
				return false;
		} else if (!expression.equals(other.expression))
			return false;
		return true;
	}

	public URI getDialect() {
		return dialect;
	}

	public void setDialect(URI dialectUri) {
		this.dialect = dialectUri;
	}
}
