/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.query.transform;

import java.net.URI;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.IRootElement;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.artifacts.IContentSelector;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.artifacts.ITemplate;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IConstraint;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IDepthLimit;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IInstanceIdConstraint;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IItemReference;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IItemTemplate;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IOperator;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IPrefixMapping;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IPropertyId;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IPropertyValue;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IQuery;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IQueryInputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IRecordConstraint;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IRecordType;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IRelationshipEndpoint;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IRelationshipTemplate;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.ISelectedRecordType;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IXPathExpression;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.QueryInputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.artifacts.IInstanceId;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.ext.Attributes2;
import org.xml.sax.helpers.DefaultHandler;

/**
 * SAX parser that builds Query object from XML input
 * 
 * @provisional
 * @author David Whiteman
 * @author Ali Mehregani
 */
public class QueryInputStreamHandler extends DefaultHandler implements ITransformerHandler
{
	private IQuery query;
	private ITemplate currentTemplate;
	private String currentElementData;
	private IConstraint currentConstraint;
	private IContentSelector currentContentSelector;
	private IOperator currentOperator;
	private ISelectedRecordType currentSelectedRecordType;
	private IInstanceId currentInstanceId;
	private IPropertyValue currentPropertyValue;
	private IQueryInputArtifactFactory artifactFactory;

	/**
	 * Flags used to indicate the start/close of certain elements
	 */
	private boolean relationshipTemplate,		// <relationshipTemplate>
					instanceIdConstraint,		// <instanceIdConstraint>
					propertyValue,				// <propertyValue>
					xpathExpression,			// <xpathExpression>
					contentSelector;			// <contentSelector>	
	
	public QueryInputStreamHandler(IQueryInputArtifactFactory artifactFactory) {
		this.artifactFactory = artifactFactory;
	}


	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException 
	{
		if (!ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE.equals(uri))
		{
			return;
		}
				
		boolean temporaryFlag = false;
		
		if (propertyValue)
		{
			handlePropertyElements(uri, localName, name, attributes, true);
		}

		else if (xpathExpression)
		{
			handleXPathElements (uri, localName, name, attributes, true);
		}
		
		else if (contentSelector)
		{
			handleContentSelector (uri, localName, name, attributes, true);
		}
		
		// <query> element
		else if (IQueryTransformerConstants.QUERY_ELEMENT.equals(localName)) 
		{
			query = getArtifactFactory().createQuery();
		}
		
		// <itemTemplate> or <relationshipTemplate> elements
		else if (IQueryTransformerConstants.ITEM_TEMPLATE_ELEMENT.equals(localName) || (temporaryFlag = IQueryTransformerConstants.RELATIONSHIP_TEMPLATE_ELEMENT.equals(localName))) 
		{
			if (temporaryFlag) 
			{
				relationshipTemplate = true;
				currentTemplate = getArtifactFactory().createRelationshipTemplate(null);
			} 
			else 
			{
				currentTemplate = getArtifactFactory().createItemTemplate(null);
			}
			
			
			String id = attributes.getValue(IQueryTransformerConstants.ID_ATTRIBUTE);
			currentTemplate.setId(id);
			String suppressFromResultValue = attributes.getValue(IQueryTransformerConstants.SUPPRESS_FROM_RESULT_ATTRIBUTE);
			// Avoid setting suppressFromResult for default values, since want to recreate the XML
			// the same way we received it.
			if ((suppressFromResultValue != null) && ((Attributes2) attributes).isSpecified(IQueryTransformerConstants.SUPPRESS_FROM_RESULT_ATTRIBUTE)) {
				currentTemplate.setSuppressFromResult(Boolean.parseBoolean(suppressFromResultValue));
			}
		} 
		
		// <source> element 
		else if (relationshipTemplate && IQueryTransformerConstants.SOURCE_TEMPLATE_ELEMENT.equals(localName)) 
		{			
			addEndpointItems(attributes, IRelationshipEndpoint.SOURCE, localName);
		}
		
		// <target> elements
		else if (relationshipTemplate &&  IQueryTransformerConstants.TARGET_TEMPLATE_ELEMENT.equals(localName))
		{
			addEndpointItems(attributes, IRelationshipEndpoint.TARGET, localName);
		}
		
		// <depthLimit> element
		else if (relationshipTemplate && IQueryTransformerConstants.DEPTH_LIMIT_ELEMENT.equals(localName))
		{
			IItemReference element = getArtifactFactory().createItemReference();
			String referenceId = attributes.getValue(IQueryTransformerConstants.INTERMEDIATE_ITEM_TEMPLATE_ATTRIBUTE);
			element.setId(referenceId);
			element.setRef(query.getItemTemplate(referenceId));
			String maxItems = attributes.getValue(IQueryTransformerConstants.MAX_INTERMEDIATE_ITEMS_ATTRIBUTE);
			int value = isNumber(maxItems) ? Integer.parseInt(maxItems) : -1;
			if (value >= 0)
			{
				IDepthLimit depthLimit = getArtifactFactory().createDepthLimit();
				depthLimit.setIntermediateItemTemplate(element);
				depthLimit.setMaxIntermediateItems(value);
				((IRelationshipTemplate)currentTemplate).setDepthLimit(depthLimit);
			}
		}
		
		// <instanceIdConstraint> element
		else if (IQueryTransformerConstants.INSTANCE_ID_CONSTRAINT_ELEMENT.equals(localName)) 
		{
			instanceIdConstraint = true;
			currentConstraint = getArtifactFactory().createInstanceIdConstraint();
		}
		
		// <instanceId> element
		else if (IQueryTransformerConstants.INSTANCE_ID_ELEMENT.equals(localName)) 
		{
			currentInstanceId = getArtifactFactory().createInstanceId();
			((IInstanceIdConstraint) currentConstraint).addInstanceId(currentInstanceId);
		}
		
		// <recordConstraint> element
		else if (IQueryTransformerConstants.RECORD_CONSTRAINT_ELEMENT.equals(localName)) 
		{
			currentConstraint = getArtifactFactory().createRecordConstraint();
		}
		
		// <propertyValue> element
		else if (IQueryTransformerConstants.PROPERTY_VALUE_ELEMENT.equals(localName)) 
		{
			propertyValue = true;
			IPropertyValue propertyValue = getArtifactFactory().createPropertyValue();
			String namespace = attributes.getValue(ICMDBfServicesConstants.NAMESPACE_ATTRIBUTE);
			String locName = attributes.getValue(ICMDBfServicesConstants.LOCAL_NAME_ATTRIBUTE);
			propertyValue.setNamespace(CMDBfServicesUtil.createURI(namespace));
			propertyValue.setLocalName(locName);
			if (shouldRetrieveAttribute(attributes, IQueryTransformerConstants.RECORD_METADATA_ATTRIBUTE)) {
				String recordMetadata = attributes.getValue(IQueryTransformerConstants.RECORD_METADATA_ATTRIBUTE);
				propertyValue.setMatchAny(Boolean.parseBoolean(recordMetadata));
			}
			if (shouldRetrieveAttribute(attributes, IQueryTransformerConstants.MATCH_ANY_ATTRIBUTE)) {
				String matchAny = attributes.getValue(IQueryTransformerConstants.MATCH_ANY_ATTRIBUTE);
				propertyValue.setMatchAny(Boolean.parseBoolean(matchAny));
			}
			((IRecordConstraint) currentConstraint).addPropertyValue(propertyValue);
			currentPropertyValue = propertyValue;
		} 
		
		// <xpathExpression> element
		else if (IQueryTransformerConstants.XPATH_EXPRESSION_ELEMENT.equals(localName)) 
		{
			xpathExpression = true;
			URI dialectURI = CMDBfServicesUtil.createURI(attributes.getValue(IQueryTransformerConstants.DIALECT_ATTRIBUTE));
			currentConstraint = getArtifactFactory().createXPathExpression(dialectURI);
		} 
		
		// <recordType> element
		else if (IQueryTransformerConstants.RECORD_TYPE_ELEMENT.equals(localName)) 
		{
			IRecordType recordType = getArtifactFactory().createRecordType();
			String namespace = attributes.getValue(ICMDBfServicesConstants.NAMESPACE_ATTRIBUTE);
			String locName = attributes.getValue(ICMDBfServicesConstants.LOCAL_NAME_ATTRIBUTE);
			recordType.setNamespace(CMDBfServicesUtil.createURI(namespace));
			recordType.setLocalName(locName);
			// No additional data or subelements, so we will go ahead and add to template
			((IRecordConstraint) currentConstraint).addRecordType(recordType);
		}  
		
		// <contentSelector> element
		else if (IQueryTransformerConstants.CONTENT_SELECTOR_ELEMENT.equals(localName)) 
		{
			contentSelector = true;
			currentContentSelector = getArtifactFactory().createContentSelector();
			if (shouldRetrieveAttribute(attributes, IQueryTransformerConstants.MATCHED_RECORDS_ATTRIBUTE)) {
				String matchedRecords = attributes.getValue(IQueryTransformerConstants.MATCHED_RECORDS_ATTRIBUTE);
				currentContentSelector.setMatchedRecords(Boolean.parseBoolean(matchedRecords));
			}
		} 
	}
	
	
	/**
	 * Returns true iff 'str' is a valid number
	 * 
	 * @param str The input string
	 * @return true iff 'str' is a number; false otherwise
	 */
	private static boolean isNumber(String str)
	{
		if (str == null)
		{
			return false;
		}
		
		for (int i = 0, charCount = str.length(); i < charCount; i++)
		{
			if (!Character.isDigit(str.charAt(i)))
				return false;
		}
		
		return str.length() > 0;
	}
	
	
	private void addEndpointItems(Attributes attributes, int relationshipType, String localName)
	{
		IRelationshipEndpoint element = getArtifactFactory().createRelationshipEndpoint(relationshipType);
		String referenceId = attributes.getValue(IQueryTransformerConstants.REF_ATTRIBUTE);
		element.setId(referenceId);
		IItemTemplate template = query.getItemTemplate(referenceId);
		if (template != null) {
			element.setRef(template);
		}
		String min = attributes.getValue(IQueryTransformerConstants.MINIMUM_ATTRIBUTE);
		if (min != null) {
			element.setMinimum(Integer.valueOf(min).intValue());
		}
		String max = attributes.getValue(IQueryTransformerConstants.MAXIMUM_ATTRIBUTE);
		if (max != null) {
			element.setMaximum(Integer.valueOf(max).intValue());
		}
		
		if (currentTemplate != null) {
			if (IQueryTransformerConstants.SOURCE_TEMPLATE_ELEMENT.equals(localName)) {
				((IRelationshipTemplate) currentTemplate).setSourceElement(element);
			} else {
				((IRelationshipTemplate) currentTemplate).setTargetElement(element);
			}
		}
	}

	/**
	 * Determines whether an attribute was specified before we actually set it in the artifact.
	 * This allows us to remember whether it was explicitly set in the input XML, so we can regenerate
	 * XML that follows the same pattern.
	 * 
	 * @param attributes
	 * @param attributeName
	 * @return
	 */
	private boolean shouldRetrieveAttribute(Attributes attributes,
			String attributeName) {
		int attributeIndex = attributes.getIndex(attributeName);
		return (attributeIndex != -1) && ((!(attributes instanceof Attributes2)) || ((Attributes2) attributes).isSpecified(attributeIndex));
	}
	
	private void handleInstanceIdElements(String uri, String localName, String name, boolean open)
	{
		if (!open)
		{
			// <mdrId> element
			if (ICMDBfServicesConstants.MDR_ID_ELEMENT.equals(localName)) 
			{
				currentInstanceId.setMdrId(currentElementData);
			} 
			
			// <localId> element
			else if (ICMDBfServicesConstants.LOCAL_ID_ELEMENT.equals(localName)) 
			{
				currentInstanceId.setLocalId(currentElementData);
			} 
		}
	}

	private void handleContentSelector(String uri, String localName, String name, Attributes attributes, boolean open)
	{
		if (open)
		{
			// <selectedRecordType> element
			if (IQueryTransformerConstants.SELECTED_RECORD_TYPE_ELEMENT.equals(localName)) 
			{
				ISelectedRecordType recordType = getArtifactFactory().createSelectedRecordType();
				String namespace = attributes.getValue(ICMDBfServicesConstants.NAMESPACE_ATTRIBUTE);
				String locName = attributes.getValue(ICMDBfServicesConstants.LOCAL_NAME_ATTRIBUTE);
				recordType.setNamespace(CMDBfServicesUtil.createURI(namespace));
				recordType.setLocalName(locName);
				currentSelectedRecordType = recordType;
				currentContentSelector.addSelectedRecordType(recordType);
			}
			// <selectedProperty> element
			else if (IQueryTransformerConstants.SELECTED_PROPERTY_ELEMENT.equals(localName)) 
			{
				IPropertyId property = getArtifactFactory().createPropertyId();
				String namespace = attributes.getValue(ICMDBfServicesConstants.NAMESPACE_ATTRIBUTE);
				String locName = attributes.getValue(ICMDBfServicesConstants.LOCAL_NAME_ATTRIBUTE);
				property.setNamespace(CMDBfServicesUtil.createURI(namespace));
				property.setLocalName(locName);
				currentSelectedRecordType.addSelectedProperty(property);
			}
		}		
	}

	private void handleXPathElements(String uri, String localName, String name, Attributes attributes, boolean open)
	{
		if (open)
		{
			// <prefixMapping> element
			if (IQueryTransformerConstants.PREFIX_MAPPING_ELEMENT.equals(localName)) 
			{
				IPrefixMapping mapping = getArtifactFactory().createPrefixMapping();
				mapping.setPrefix(attributes.getValue(IQueryTransformerConstants.PREFIX_ATTRIBUTE));
				mapping.setNamespace(CMDBfServicesUtil.createURI(attributes.getValue(ICMDBfServicesConstants.NAMESPACE_ATTRIBUTE)));
				((IXPathExpression) currentConstraint).addPrefixMapping(mapping);
			}
		}
		else
		{
			// <xpathExpression> element		
			if (IQueryTransformerConstants.EXPRESSION_ELEMENT.equals(localName)) 
			{
				((IXPathExpression) currentConstraint).setExpression(currentElementData);
			} 
		}
	}

	private void handlePropertyElements(String uri, String localName, String name, Attributes attributes, boolean open)
	{
		if (open)
		{
			// <less> element
			if (IQueryTransformerConstants.LESS_ELEMENT.equals(localName)) 
			{
				createCurrentOperator(attributes, IOperator.LESS_THAN);
			} 
			
			// <lessOrEqual> element
			else if (IQueryTransformerConstants.LESS_OR_EQUAL_ELEMENT.equals(localName)) 
			{
				createCurrentOperator(attributes, IOperator.LESS_THAN_OR_EQUAL);
			} 
			
			// <greater> element
			else if (IQueryTransformerConstants.GREATER_ELEMENT.equals(localName)) 
			{
				createCurrentOperator(attributes, IOperator.GREATER_THAN);
			} 
			
			// <greaterOrEqual> element
			else if (IQueryTransformerConstants.GREATER_OR_EQUAL_ELEMENT.equals(localName)) 
			{
				createCurrentOperator(attributes, IOperator.GREATER_THAN_OR_EQUAL);
			} 
			
			// <equal> element
			else if (IQueryTransformerConstants.EQUAL_ELEMENT.equals(localName)) 
			{
				createCurrentStringOperator(attributes, IOperator.EQUALS);
			} 
			
			// <like> element
			else if (IQueryTransformerConstants.LIKE_ELEMENT.equals(localName)) 
			{
				createCurrentStringOperator(attributes, IOperator.LIKE);
			} 
			
			// <contains> element
			else if (IQueryTransformerConstants.CONTAINS_ELEMENT.equals(localName)) 
			{
				createCurrentStringOperator(attributes, IOperator.CONTAINS);
			} 
			
			// <isNull> element
			else if (IQueryTransformerConstants.IS_NULL_ELEMENT.equals(localName)) 
			{
				currentOperator = getArtifactFactory().createOperator(IOperator.IS_NULL);
				if (shouldRetrieveAttribute(attributes, IQueryTransformerConstants.NEGATE_ATTRIBUTE)) {
					String negateString = attributes.getValue(IQueryTransformerConstants.NEGATE_ATTRIBUTE);
					currentOperator.setNegate(Boolean.parseBoolean(negateString));
				}
			}
		}
		else
		{
			if (isComparisonSelector(localName)) 
			{
				currentOperator.setValue(currentElementData);
				currentPropertyValue.addOperator(currentOperator);
			} 
		}
	}

	private void createCurrentOperator(Attributes attributes, int type) 
	{
		currentOperator = getArtifactFactory().createOperator(type);
		// Only call setNegate if the attribute is explicitly specified in the XML.
		// This way, we can later recreate the XML in a similar format.
		if (shouldRetrieveAttribute(attributes, IQueryTransformerConstants.NEGATE_ATTRIBUTE)) {
			String negateString = attributes.getValue(IQueryTransformerConstants.NEGATE_ATTRIBUTE);
			currentOperator.setNegate(Boolean.parseBoolean(negateString));
		}
	}

	private void createCurrentStringOperator(Attributes attributes, int type) 
	{
		// call the other method to ensure "negate" is handled
		createCurrentOperator(attributes, type);
		// Only call setCaseSensitive if the attribute is explicitly specified in the XML.
		// This way, we can later recreate the XML in a similar format.
		if (shouldRetrieveAttribute(attributes, IQueryTransformerConstants.CASE_SENSITIVE_ATTRIBUTE)) {
			String caseSensitiveString = attributes.getValue(IQueryTransformerConstants.CASE_SENSITIVE_ATTRIBUTE);
			currentOperator.setCaseSensitive(Boolean.parseBoolean(caseSensitiveString));
		}
	}
	
	public void endElement(String uri, String localName, String name) throws SAXException 
	{
		if (!ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE.equals(uri))
		{
			return;
		}
				
		// <itemTemplate> element
		if (IQueryTransformerConstants.ITEM_TEMPLATE_ELEMENT.equals(localName)) 
		{
			query.addItemTemplate((IItemTemplate) currentTemplate);
			currentTemplate = null;
		} 
		
		// <relationshipTemplate> element
		else if (relationshipTemplate && IQueryTransformerConstants.RELATIONSHIP_TEMPLATE_ELEMENT.equals(localName)) 
		{			
			relationshipTemplate = false;
			query.addRelationshipTemplate((IRelationshipTemplate) currentTemplate);
			currentTemplate = null;
		} 
		
		// <instanceId> element
		else if (instanceIdConstraint && IQueryTransformerConstants.INSTANCE_ID_ELEMENT.equals(localName)) 
		{
			currentInstanceId = null;
		} 

		// <instanceIdConstraint> element
		else if (instanceIdConstraint && IQueryTransformerConstants.INSTANCE_ID_CONSTRAINT_ELEMENT.equals(localName)) 
		{
			instanceIdConstraint = false;
			currentTemplate.setInstanceIdConstraint((IInstanceIdConstraint) currentConstraint);
			currentConstraint = null;
		} 
		
		// <xpathExpression> element
		else if (IQueryTransformerConstants.XPATH_EXPRESSION_ELEMENT.equals(localName)) 
		{
			xpathExpression = false;
			currentTemplate.addXpathExpression((IXPathExpression) currentConstraint);
			currentConstraint = null;
		} 
		
		
		// <recordConstraint> element
		else if (IQueryTransformerConstants.RECORD_CONSTRAINT_ELEMENT.equals(localName)) 
		{
			currentTemplate.addRecordConstraint((IRecordConstraint) currentConstraint);
			currentConstraint = null;
		} 

		// <propertyValue> element
		else if (propertyValue && IQueryTransformerConstants.PROPERTY_VALUE_ELEMENT.equals(localName)) 
		{
			propertyValue = false;
			currentPropertyValue = null;
		} 
		
		// <contentSelector> element
		else if (contentSelector && IQueryTransformerConstants.CONTENT_SELECTOR_ELEMENT.equals(localName)) 
		{
			contentSelector = false;
			currentTemplate.setContentSelector(currentContentSelector);
			currentContentSelector = null;
		} 
		
		// <isNull> element
		else if (IQueryTransformerConstants.IS_NULL_ELEMENT.equals(localName)) 
		{
			((IPropertyValue) currentPropertyValue).addOperator(currentOperator);
		}  
		
		else if (instanceIdConstraint)
		{
			handleInstanceIdElements (uri, localName, name, false);
		}
		
		else if (propertyValue)
		{
			handlePropertyElements(uri, localName, name, null, false);
		}
		
		else if (xpathExpression)
		{
			handleXPathElements(uri, localName, name, null, false);
		}
		currentElementData = null;
	}

	private boolean isComparisonSelector(String name) 
	{
		return
			(IQueryTransformerConstants.EQUAL_ELEMENT.equals(name)) ||
			(IQueryTransformerConstants.CONTAINS_ELEMENT.equals(name)) ||
			(IQueryTransformerConstants.LIKE_ELEMENT.equals(name)) ||
			(IQueryTransformerConstants.GREATER_ELEMENT.equals(name)) ||
			(IQueryTransformerConstants.GREATER_OR_EQUAL_ELEMENT.equals(name)) ||
			(IQueryTransformerConstants.LESS_ELEMENT.equals(name)) ||
			(IQueryTransformerConstants.LESS_OR_EQUAL_ELEMENT.equals(name));
	}

	public void characters(char[] ch, int start, int length) throws SAXException 
	{
		String tempData = new String(ch, start, length).trim();
		if (currentElementData == null) {
			currentElementData = tempData;
		} else {
			currentElementData += tempData;
		}
	}

	public IRootElement getResult() 
	{
		return query;
	}

	public void error(SAXParseException e) throws SAXException 
	{
		throw e;
	}


	protected IQueryInputArtifactFactory getArtifactFactory() {
		if (artifactFactory == null) {
			artifactFactory = QueryInputArtifactFactory.getInstance();
		}
		return artifactFactory;
	}
}
