/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.query.transform;

import java.io.InputStream;
import java.io.Reader;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServicesUtil;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.common.IXMLWritable;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IQueryOutputArtifactFactory;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.response.artifacts.IQueryResult;
import org.eclipse.cosmos.dc.cmdbf.services.transform.ITransformerHandler;
import org.eclipse.cosmos.dc.cmdbf.services.transform.TransformerException;
import org.xml.sax.InputSource;

/**
 * Convert input stream containing an XML graph query into a data structure
 * 
 * @provisional
 * @author David Whiteman
 */
public class QueryOutputTransformer {
	/**
	 * Convert an IQuery into the XML syntax for query results described in the CMBDf spec
	 * 
	 * @param query
	 * @return an input stream on the XML containing the <queryResult> structure
	 */
	public static InputStream transform(IQueryResult queryResult) {
		return CMDBfServicesUtil.transformImpl((IXMLWritable) queryResult);
	}

	/**
	 * Convert input stream containing an XML structure for a CMDBf query response into a
	 * Java object graph
	 * 
	 * @param xmlInputStream
	 * @return IQueryResult object structure representing a query result
	 * @throws TransformerException
	 */
	public static IQueryResult transform(InputStream xmlInputStream) throws TransformerException 
	{
		return transform(xmlInputStream, null, null);	
	}

	public static IQueryResult transform(InputStream xmlInputStream, IQueryOutputArtifactFactory artifactFactory) throws TransformerException 
	{
		return transform(xmlInputStream, artifactFactory, null);	
	}
	
	/**
	 * Convert input stream containing an XML structure for a CMDBf query response into a
	 * Java object graph
	 * 
	 * @param reader
	 * @return IQueryResult object structure representing a query result
	 * @throws TransformerException
	 */
	public static IQueryResult transform(Reader reader) throws TransformerException 
	{
		return transform(reader, null, null);	
	}

	public static IQueryResult transform(Reader reader, IQueryOutputArtifactFactory artifactFactory) throws TransformerException 
	{
		return transform(reader, artifactFactory, null);	
	}

	
	/**
	 * Convert input stream containing an XML structure for a CMDBf query response into a
	 * Java object graph.  Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param xmlInputStream
	 * @param transformerOptions options to control transformer behavior
	 * @return IQueryResult object structure representing a query result
	 * @throws TransformerException
	 */
	public static IQueryResult transform(InputStream xmlInputStream, IQueryOutputArtifactFactory artifactFactory, Map<String,Object> transformerOptions) throws TransformerException 
	{
		return transformerOptions == null ?
				(IQueryResult) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new QueryOutputStreamHandler(artifactFactory)) :
				(IQueryResult) CMDBfServicesUtil.transformImpl(new InputSource(xmlInputStream), (ITransformerHandler) new QueryOutputStreamHandler(artifactFactory), transformerOptions);	
	}
	
	
	/**
	 * Convert input stream containing an XML structure for a CMDBf query response into a
	 * Java object graph.  Can specify one of the following options:
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_VALIDATION_OPTION}
	 * {@value ICMDBfServicesConstants#SERVICES_TRANSFORM_DATAMODEL_OPTION}
	 * 
	 * @param reader
	 * @param transformerOptions options to control transformer behavior
	 * @return IQueryResult object structure representing a query result
	 * @throws TransformerException
	 */
	public static IQueryResult transform(Reader reader, IQueryOutputArtifactFactory artifactFactory, Map<String,Object> transformerOptions) throws TransformerException 
	{
		return transformerOptions == null ?
				(IQueryResult) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new QueryOutputStreamHandler(artifactFactory)) :
				(IQueryResult) CMDBfServicesUtil.transformImpl(new InputSource(reader), (ITransformerHandler) new QueryOutputStreamHandler(artifactFactory), transformerOptions);	
	}
}
