/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServiceException;
import org.w3c.dom.Node;


/**
 * Represents a String SML property
 * 
 * @provisional
 * @author Ali Mehregani
 */
public class StringProperty extends BaseProperty
{
	/**
	 * The value of this property
	 */
	private String value;
	
	public StringProperty (Node node, String value)
	{
		super(node.getNamespaceURI(), node.getLocalName());
		this.value = value;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.BaseProperty#contains(java.lang.String, boolean)
	 */
	public boolean contains(String str, boolean caseSensitive)
	{		
		String propertyValue = caseSensitive ? this.value : this.value.toUpperCase();
		String operand = caseSensitive ? str : str.toUpperCase();				
		return propertyValue.indexOf(operand) >= 0;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.BaseProperty#equals(java.lang.Object, boolean)
	 */
	public boolean equals(Object o, boolean caseSensitive) throws CMDBfServiceException
	{		
		if (!(o instanceof String))
		{
			throw propertyTypeMismatch();
		}
		
		return caseSensitive ? value.equals((String)o) : value.equalsIgnoreCase((String)o);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.BaseProperty#greater(java.lang.Object)
	 */
	public boolean greater(Object o) throws CMDBfServiceException
	{
		if (!(o instanceof String))
		{
			throw propertyTypeMismatch();
		}
		
		return value.compareTo((String)o) > 0;
	}


	/* (non-Javadoc)
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.BaseProperty#less(java.lang.Object)
	 */
	public boolean less(Object o) throws CMDBfServiceException
	{		
		if (!(o instanceof String))
		{
			throw propertyTypeMismatch();
		}
		
		return value.compareTo((String)o) < 0;
	}


	/* (non-Javadoc)
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.BaseProperty#like(java.lang.String, boolean)
	 */
	public boolean like(String str, boolean caseSensitive)
	{		
		if (str == null)
		{
			return false;
		}
		
		// Check to make sure that the argument only escapes '%', '_', or '\'
		int currentInx = 0;
		int backslashInx = 0;
		int strLength = str.length();
		while (currentInx < strLength && (backslashInx = str.indexOf('\\', currentInx)) > 0)
		{
			if (backslashInx + 1 >= strLength)
			{
				throw new IllegalArgumentException ("Illegal character is escaped at index " + backslashInx);
			}
			
			char escapedChar = str.charAt(backslashInx + 1);
			if (escapedChar != '%' && escapedChar != '_' && escapedChar != '\\')
			{
				throw new IllegalArgumentException ("Illegal character is escaped at index " + backslashInx);
			}
			
			currentInx = backslashInx + 1;
		}
		
		// Escape all reserved regular expression characters		
		final String[] RESERVED_CHARACTERS = new String[] 
	    {
			"\\[", "\\]", "\\-", "\\^",
			"\\&", "\\.", "\\{", "\\}",
			"\\$", "\\?", "\\*", "\\+",
			"\\|"
		};
		for (int i = 0; i < RESERVED_CHARACTERS.length; i++)
		{
			str = str.replaceAll(RESERVED_CHARACTERS[i], "\\" + RESERVED_CHARACTERS[i]);
		}
				
		// Replace all percent ('%') and underscores ('_') that haven't been
		// escaped with asterisk ('*') and a period ('.') respectively
		currentInx  = 0; 
		StringBuffer normalizedPattern = new StringBuffer();
		strLength = str.length();
		while (currentInx < strLength)
		{
			char currentChar = str.charAt(currentInx);
			switch (currentChar)
			{
				case '%':
					normalizedPattern.append(".*");			
					break;
				case '_':
					normalizedPattern.append('.');
					break;			
				default:
					normalizedPattern.append(currentChar);
			}
			currentInx++;
		}
		
		// Run the regular expression
		String propertyValue = caseSensitive ? value : value.toUpperCase();
		String patternString = caseSensitive ? normalizedPattern.toString() : normalizedPattern.toString().toUpperCase();
				
		Pattern pattern = Pattern.compile(patternString);
		Matcher matcher = pattern.matcher(propertyValue);
		return matcher.matches();
	}
}
