/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.mdr.registration.client;

import java.io.InputStream;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.muse.util.xml.XmlUtils;
import org.apache.muse.ws.addressing.EndpointReference;
import org.apache.muse.ws.addressing.WsaConstants;
import org.eclipse.cosmos.dc.dataManager.exception.DataManagerException;
import org.eclipse.cosmos.dc.mdr.client.MdrClient;
import org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.EndPointReferenceSerializer;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.MDRIdentifierSerializer;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.SelectiveRegistrationInputSerializer;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.EndPointReferenceSerializer.MDRReference;
import org.w3c.dom.Element;

/**
 * A client implementation for a federating CMDB.  This class extends
 * the MDR client to provide registration capabilities.  Clients can use
 * this class as-is or provide an alternative implementation by extending
 * {@link MdrClient} or an equivalent class.
 * 
 * @provisional
 * @author Ali Mehregani
 */
public class FederatingCMDBClient extends MdrClient
{
	/**
	 * The constructor
	 * 
	 * @param destination The end point reference identifying this
	 * federating CMDB
	 */
	public FederatingCMDBClient(EndpointReference destination) 
	{
		super(destination);		
	}
	
	
	/**
	 * Provides the ability to send a registration request to a 
	 * federating CMDB.
	 * 
	 * @param registrationRequest The registration request as it conforms
	 * to the CMDBf specification
	 * 
	 * @return An element representing the CMDBf registration response, conforming
	 * to the CMDBf specification
	 * 
	 * @throws DataManagerException In case of any unexpected error
	 */
	public Element register(Element registrationRequest) throws DataManagerException 
	{
		return invokeRemoteOperation(
					registrationRequest, 
					IFederatingCMDB.REGISTRATION_OP_QNAME, 
					IFederatingCMDB.REGISTRATION_REQUEST_QNAME, 
					IFederatingCMDB.REGISTRATION_OP_URI);
	}
	
	
	/**
	 * Registers all items and relationship of MDRs with the set of IDs  
	 * passed in.
	 *  
	 * @param mdrEPRs The MDR identifiers
	 * 
	 * @return The registration response
	 * 
	 * @throws DataManagerException In case of any unexpected error
	 */
	public Element register (EndpointReference[] mdrEPRs) throws DataManagerException 
	{				
		return invokeRemoteOperation(
				EndPointReferenceSerializer.getInstance().toXML(constructIdentifiers(mdrEPRs)), 
				IFederatingCMDB.REGISTER_MDR_ENTRIES_OP_QNAME, 
				IFederatingCMDB.REGISTER_MDR_ENTRIES_MDR_ID_QNAME, 
				IFederatingCMDB.REGISTER_MDR_ENTRIES_OP_URI);
	}
	
	
	/**
	 * Registers the items/relationships that a result of applying query
	 * to the MDR with the ID mdrIdentifiers[i].
	 *  
	 * @param mdrEPRs The IDs for a set of MDRs
	 * @param query The query to apply to each MDR
	 * 
	 * @return The registration response
	 * 
	 * @throws DataManagerException In case of any unexpected error
	 */
	public Element register (EndpointReference[] mdrEPRs, Element query) throws DataManagerException 
	{
		return invokeRemoteOperation(
				SelectiveRegistrationInputSerializer.getInstance().toXML(constructIdentifiers(mdrEPRs), query), 
				IFederatingCMDB.REGISTER_SELECTIVE_MDR_ENTRIES_OP_QNAME, 
				IFederatingCMDB.REGISTER_SELECTIVE_MDR_ENTRIES_REQUEST_QNAME,
				IFederatingCMDB.REGISTER_SELECTIVE_MDR_ENTRIES_OP_URI);
	}
	
	
	/**
	 * Provides the ability to register all MDRs registered with the brokers 
	 * of a management domain
	 * 
	 * @param managementDominEPR The EPR of the management domain
	 * 
	 * @return The registration response
	 * 
	 * @throws DataManagerException In case of any unexpected error
	 */
	public Element register (String managementDominEPR) throws DataManagerException
	{
		Element argument = XmlUtils.createElement(new QName(IFederatingCMDB.MANAGEMENT_DOMAIN_ELEMENT));
		argument.setTextContent(managementDominEPR);
		return invokeRemoteOperation(
				argument, 
				IFederatingCMDB.REGISTER_ALL_OP_QNAME, 
				IFederatingCMDB.REGISTER_ALL_MGT_DOMAIN_QNAME,
				IFederatingCMDB.REGISTER_ALL_OP_URI);
	}
	
	
	/**
	 * Provides the ability to send a deregistration request to a 
	 * federating CMDB.
	 * 
	 * @param deregistrationRequest The deregistration request as it conforms
	 * to the CMDBf specification
	 * 
	 * @return An element representing the CMDBf deregistration response, conforming
	 * to the CMDBf specification
	 * 
	 * @throws DataManagerException In case of any unexpected error
	 */
	public Element deregister(Element deregistrationRequest) throws DataManagerException 
	{
		return invokeRemoteOperation(
				deregistrationRequest, 
				IFederatingCMDB.DEREGISTER_OP_QNAME, 
				IFederatingCMDB.DEREGISTER_REQUEST_QNAME, 
				IFederatingCMDB.DEREGISTER_OP_URI);
	}
	
	
	/**
	 * Provides the ability to deregister all items/relationship of
	 * an MDR set 
	 * 
	 * @param mdrIds The set of MDR ids
	 * 
	 * @return An element representing the CMDBf deregistration response, conforming
	 * to the CMDBf specification
	 * 
	 * @throws DataManagerException In case of any unexpected error
	 */
	public Element deregister(EndpointReference[] mdrIds) throws DataManagerException 
	{
		return invokeRemoteOperation(
				MDRIdentifierSerializer.getInstance().toXML(mdrIds), 
				IFederatingCMDB.DEREGISTER_MDR_ENTRIES_OP_QNAME, 
				IFederatingCMDB.DEREGISTER_MDR_ENTRIES_MDR_ID_QNAME,
				IFederatingCMDB.DEREGISTER_MDR_ENTRIES_OP_URI);
	}
	
	
	/**
	 * Provides the ability to deregister a set of items/relationships based on 
	 * the result of a query
	 * 
	 * @param query The query that will select the set of items/relationships
	 * to deregister.  
	 * 
	 * @return An element representing the CMDBf deregistration response, conforming
	 * to the CMDBf specification
	 * 
	 * @throws DataManagerException In case of any unexpected error
	 */
	public Element deregister(InputStream query) throws DataManagerException 
	{
		try
		{
			Element queryElement = XmlUtils.getDocumentRoot(DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(query));
			return invokeRemoteOperation(
					queryElement, 
					IFederatingCMDB.DEREGISTER_SELECTIVE_ENTRIES_OP_QNAME, 
					IFederatingCMDB.DEREGISTER_SELECTIVE_ENTRIES_QUERY_QNAME,
					IFederatingCMDB.DEREGISTER_SELECTIVE_ENTRIES_OP_URI);
		} 
		catch (Exception e)
		{
			throw new DataManagerException(e);
		}
	}
	
	
	/**
	 * Provides the ability to deregister all items and relationships of
	 * the federating CMDB
	 * 
	 * @return The deregistration response
	 * 
	 * @throws DataManagerException In case of an unexpected error
	 */
	public Element deregisterAll() throws DataManagerException
	{
		return invokeRemoteOperation(
				IFederatingCMDB.DEREGISTER_ALL_OP_QNAME, 
				IFederatingCMDB.DEREGISTER_ALL_OP_URI);
	}
	
	
	private MDRReference[] constructIdentifiers(EndpointReference[] mdrEPRs)
	{
		MDRReference[] identifiers = new MDRReference[mdrEPRs.length]; 
		for (int i = 0; i < mdrEPRs.length; i++)
		{
			identifiers[i] = new MDRReference(
					mdrEPRs[i].getAddress().toString(), 
					mdrEPRs[i].getParameter(WsaConstants.DEFAULT_RESOURCE_ID_QNAME).getTextContent());			
		}
		
		return identifiers;
	}
}
