/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.mdr.registration.common.internal;

import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;

import org.apache.muse.util.xml.XmlUtils;
import org.apache.muse.ws.addressing.EndpointReference;
import org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Used to serialize/de-serialize a set of end point references
 * to/from an XML representation.  This is a singleton class 
 * that should be access via {@link #getInstance()}
 * 
 * @author Ali Mehregani
 */
public class EndPointReferenceSerializer
{
	/**
	 * The instance of this class
	 */
	private static EndPointReferenceSerializer instance;
	
	
	/**
	 * Restrict the visibility of this class
	 */
	private EndPointReferenceSerializer()
	{
		
	}
	
	
	/**
	 * Retrieves and returns the instance of this singleton
	 * class
	 * 
	 * @return The static instance of this class 
	 */
	public static EndPointReferenceSerializer getInstance()
	{
		if (instance == null)
		{
			instance = new EndPointReferenceSerializer();
		}
		return instance;
	}
	
	
	/**
	 * Return the XML representation of the MDR identifiers
	 * passed in
	 * 
	 * @param mdrIdentifiers The MDR identifiers
	 * @return An XML representation
	 */
	public Element toXML (MDRReference[] identifiers)
	{
		Element mdrIdentifiersElem = XmlUtils.createElement(new QName(IFederatingCMDB.MDR_ADDRESSES_ELEMENT));
		Document document = mdrIdentifiersElem.getOwnerDocument();
		for (int i = 0; i < identifiers.length; i++)
		{
			// Create entry element
			Element mdrId = XmlUtils.createElement(new QName(IFederatingCMDB.ADDRESS_ENTRY_ELEMENT));			
			
			// Add the end point reference attribute
			Attr attr = document.createAttribute(IFederatingCMDB.END_POINT_REFERENCE_ATTRIBUTE);
			attr.setNodeValue(identifiers[i].getEndPointReferenceURI());
			mdrId.setAttributeNode(attr);
			
			// Add the default resource name attribute
			attr = document.createAttribute(IFederatingCMDB.DEFAULT_RESOURCE_ID_ATTRIBUTE);
			attr.setNodeValue(identifiers[i].getDefaultResourceId());
			mdrId.setAttributeNode(attr);
			
			mdrIdentifiersElem.appendChild(mdrId);
		}
		
		return mdrIdentifiersElem;
	}
	
	
	/**
	 * De-serializes the XML representation of the MDR identifiers
	 * passed in.
	 * 
	 * @param mdrIdentifiers The XML representation of the identifiers
	 * @return A string array representing the mdr ids
	 */
	public MDRReference[] fromXML (Element mdrIdentifiers)
	{
		if (!mdrIdentifiers.hasChildNodes())
		{
			return new MDRReference[0];
		}
		
		Node rootNode = SerializerUtil.findNode(mdrIdentifiers, IFederatingCMDB.MDR_ADDRESSES_ELEMENT);			
		if (rootNode == null)
		{
			return new MDRReference[0];
		}
		
		List<MDRReference> mdrIdList = new ArrayList<MDRReference>();
		NodeList children = rootNode.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node node = children.item(i);
			if (!IFederatingCMDB.ADDRESS_ENTRY_ELEMENT.equals(node.getLocalName()))
			{
				continue;
			}
			
			String[] values = retrieveAttributes (
					node.getAttributes(), 
					new String[]{
						IFederatingCMDB.END_POINT_REFERENCE_ATTRIBUTE, 
						IFederatingCMDB.DEFAULT_RESOURCE_ID_ATTRIBUTE});
			if (values == null || values.length != 2)
			{
				continue;
			}
			mdrIdList.add(new MDRReference(values[0], values[1]));
		}
		
		return mdrIdList.toArray(new MDRReference[mdrIdList.size()]);
	}


	private String[] retrieveAttributes(NamedNodeMap attributes, String[] attributeNames)
	{
		String[] value = new String[attributeNames.length];
		for (int i = 0; i < attributeNames.length; i++)
		{
			Node node = attributes.getNamedItem(attributeNames[i]);
			String nodeValue = null; 
			if (node == null || (nodeValue = node.getNodeValue()) == null)
			{
				return new String[0];
			}
			
			value[i] = nodeValue;
		}
		return value;
	}
	
	
	public static class MDRReference
	{
		private String endPointReferenceURI;
		private String defaultResourceId;
		private EndpointReference epr;
		
		public MDRReference (EndpointReference epr)
		{
			this.epr = epr;
		}
		
		public MDRReference (String endPointReferenceURI, String defaultResourceId)
		{
			this.endPointReferenceURI = endPointReferenceURI;
			this.defaultResourceId = defaultResourceId;
		}
		
		public MDRReference()
		{
			
		}
		
		/**
		 * @return the endPointReference
		 */
		public String getEndPointReferenceURI()
		{
			return endPointReferenceURI;
		}
		/**
		 * @param endPointReference the endPointReference to set
		 */
		public void setEndPointReferenceURI(String endPointReference)
		{
			this.endPointReferenceURI = endPointReference;
		}
		/**
		 * @return the defaultResourceName
		 */
		public String getDefaultResourceId()
		{
			return defaultResourceId;
		}
		/**
		 * @param defaultResourceName the defaultResourceName to set
		 */
		public void setDefaultResourceId(String defaultResourceId)
		{
			this.defaultResourceId = defaultResourceId;
		}
		/**
		 * @return the epr
		 */
		public EndpointReference getEpr()
		{
			return epr;
		}
		/**
		 * @param epr the epr to set
		 */
		public void setEpr(EndpointReference epr)
		{
			this.epr = epr;
		}				
	}
}
