/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.mdr.registration.common.internal;

import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;

import org.apache.muse.util.xml.XmlUtils;
import org.apache.muse.ws.addressing.EndpointReference;
import org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Serializes a set of MDR IDs
 * 
 * @author Ali Mehregani
 */
public class MDRIdentifierSerializer
{
	private static MDRIdentifierSerializer instance;
	
	public static MDRIdentifierSerializer getInstance()
	{
		if (instance == null)
		{
			instance = new MDRIdentifierSerializer();
		}
		return instance;
	}
	
	private MDRIdentifierSerializer()
	{
		
	}
	
	/**
	 * Serializes a set of MDR ids into a string
	 * representation
	 * 
	 * @param mdrIds A set of MDR IDs
	 * @return The XML representation of the MDR IDs
	 */
	public Element toXML(EndpointReference[] mdrIds)
	{
		Element rootNode = XmlUtils.createElement(new QName(IFederatingCMDB.MDR_IDS_ELEMENT));
		Document document = rootNode.getOwnerDocument();
		for (int i = 0; i < mdrIds.length; i++)
		{
			Element entry = XmlUtils.createElement(new QName(IFederatingCMDB.MDR_ENTRY_ELEMENT));
			Attr attribute = document.createAttribute(IFederatingCMDB.ID_ATTRIBUTE);
			attribute.setValue(mdrIds[i].getAddress().toString());
			entry.setAttributeNode(attribute);
			rootNode.appendChild(entry);
		}
		
		return rootNode;
	}
	
	
	/**
	 * Deserializes the XML representation of a set
	 * of MDR IDs into an array of string
	 * 
	 * @param mdrIds The XML representation of an MDR ID set
	 * @return A string array representation of the MDR ID set
	 */
	public String[] fromXML(Element mdrIds)
	{
		Element rootNode = SerializerUtil.findNode(mdrIds, IFederatingCMDB.MDR_IDS_ELEMENT);
		if (rootNode == null)
		{
			return new String[0];
		}
		
		NodeList children = rootNode.getChildNodes();
		List<String> mdrIdsList = new ArrayList<String>();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node child = children.item(i);
			if (!IFederatingCMDB.MDR_ENTRY_ELEMENT.equals(child.getLocalName()))
			{
				continue;
			}
			mdrIdsList.add(child.getAttributes().getNamedItem(IFederatingCMDB.ID_ATTRIBUTE).getNodeValue());
		}
		return mdrIdsList.toArray(new String[mdrIdsList.size()]);
	}
}
