/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.mdr.registration.common.internal;

import javax.xml.namespace.QName;

import org.apache.muse.util.xml.XmlUtils;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.IQueryTransformerConstants;
import org.eclipse.cosmos.dc.mdr.registration.common.IFederatingCMDB;
import org.eclipse.cosmos.dc.mdr.registration.common.internal.EndPointReferenceSerializer.MDRReference;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Used to serialize/de-serialize a set of MDR identifiers + query request
 * to/from an XML representation.  This is a singleton class that should 
 * be access via {@link #getInstance()}
 * 
 * @author Ali Mehregani
 */
public class SelectiveRegistrationInputSerializer
{
	/**
	 * The instance of this class
	 */
	private static SelectiveRegistrationInputSerializer instance;
	
	
	/**
	 * Restrict the visibility of this class
	 */
	private SelectiveRegistrationInputSerializer()
	{
		
	}
	
	
	/**
	 * Retrieves and returns the instance of this singleton
	 * class
	 * 
	 * @return The static instance of this class 
	 */
	public static SelectiveRegistrationInputSerializer getInstance()
	{
		if (instance == null)
		{
			instance = new SelectiveRegistrationInputSerializer();
		}
		return instance;
	}
	
	
	/**
	 * Return the XML representation of the MDR identifiers + 
	 * query request passed in
	 * 
	 * @param mdrIdentifiers The MDR identifiers
	 * @param queryRequest The query request
	 * @return An XML representation
	 */
	public Element toXML (MDRReference[] mdrIdentifiers, Element queryRequest)
	{
		Element request = XmlUtils.createElement(new QName(IFederatingCMDB.SELECTIVE_REGISTRATION_INPUT_ELEMENT));
		request.appendChild(EndPointReferenceSerializer.getInstance().toXML(mdrIdentifiers));
		Document document = request.getOwnerDocument();
		Node adoptedNode = document.adoptNode(queryRequest);
		if (adoptedNode == null)
		{
			adoptedNode = document.importNode(queryRequest, true);
		}
		request.appendChild(adoptedNode);
		return request;
	}
	
	
	/**
	 * De-serializes the XML representation of the MDR identifiers
	 * passed in and returns an array of String representing each
	 * ID.
	 * 
	 * @param mdrReferences The XML representation of the identifiers
	 * @return A string array representing the mdr ids
	 */
	public SelectiveRegistrationRequest fromXML (Element request)
	{				
		Element root = SerializerUtil.findNode(request, IFederatingCMDB.SELECTIVE_REGISTRATION_INPUT_ELEMENT);
		if (root == null)
		{
			return null;
		}
		
		NodeList children = root.getChildNodes();
		SelectiveRegistrationRequest requestPOJO = new SelectiveRegistrationRequest();
		
		boolean foundMdrId = false;
		boolean foundEPR = false;
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			Node child = children.item(i);
			
			if (Node.ELEMENT_NODE != child.getNodeType())
			{
				continue;
			}
			
			
			if (IFederatingCMDB.MDR_ADDRESSES_ELEMENT.equals(child.getLocalName()))
			{
				MDRReference[] references = EndPointReferenceSerializer.getInstance().fromXML((Element)child);
				if (references != null && references.length > 0)
				{
					foundMdrId = true;
					requestPOJO.setMDRReferences(references);
				}
			}
			else if (IQueryTransformerConstants.QUERY_ELEMENT.equals(child.getLocalName()))
			{
				foundEPR = true;
				requestPOJO.setQuery((Element)child);
			}
		}
		
		if (!foundMdrId || !foundEPR)
		{
			return null;
		}
		
		return requestPOJO;
	}
	
	
	public static class SelectiveRegistrationRequest
	{
		private MDRReference[] mdrReferences;
		private Element query;
		
		
		/**
		 * @return the mdrIdentifiers
		 */
		public MDRReference[] getMDRReferences()
		{
			return mdrReferences;
		}
		/**
		 * @param mdrReferencees the mdrIdentifiers to set
		 */
		public void setMDRReferences(MDRReference[] mdrReferencees)
		{
			this.mdrReferences = mdrReferencees;
		}
		/**
		 * @return the query
		 */
		public Element getQuery()
		{
			return query;
		}
		/**
		 * @param query the query to set
		 */
		public void setQuery(Element query)
		{
			this.query = query;
		}
		
	}
}
