/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.dc.cmdbf.services.query.service.impl;

import java.util.Hashtable;
import java.util.Map;

import org.eclipse.cosmos.dc.cmdbf.services.cmdbfservice.IQueryHandler;
import org.eclipse.cosmos.dc.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.cmdbf.services.common.ICMDBfServicesConstants;
import org.eclipse.cosmos.dc.cmdbf.services.internal.CMDBfInternalUtility;
import org.eclipse.cosmos.dc.cmdbf.services.internal.CMDBfMessages;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.IItemConstraintHandler;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.IItemTemplateHandler;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.IQueryHandlerFactory;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.IRelationshipConstraintHandler;
import org.eclipse.cosmos.dc.cmdbf.services.query.service.IRelationshipTemplateHandler;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.IQueryTransformerConstants;
import org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IConstraint;

/**
 * An abstract implementation of {@link IQueryHandlerFactory}.  Clients
 * who wish to provide a partial implementation of {@link IQueryHandlerFactory} can
 * extend this class.
 * 
 * @provisional
 * @see IQueryHandlerFactory
 * @author Ali Mehregani
 */
public abstract class AbstractQueryHandlerFactory implements IQueryHandlerFactory
{
	/**
	 * Identifies an item template
	 */
	private static final int ITEM_TEMPLATE = 0x00;
	
	/**
	 * Identifies a relationship template
	 */
	private static final int RELATIONSHIP_TEMPLATE = 0x01;
	
	
	/**
	 * Cached values of handlers
	 */
	private Map<Integer, Map<Integer, IQueryHandler>> cachedHandlers;
	
	
	public AbstractQueryHandlerFactory()
	{
		cachedHandlers = new Hashtable<Integer, Map<Integer, IQueryHandler>>();
	}
	
	/**
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.service.IQueryHandlerFactory#createItemTemplateHandler(org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IItemTemplate)
	 */
	public IItemTemplateHandler createItemTemplateHandler() throws CMDBfServiceException
	{		
		return ((IItemTemplateHandler)createHandler(null, ITEM_TEMPLATE));
	}
	
	
	/**
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.service.IQueryHandlerFactory#createItemConstraintHandler(org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IConstraint)
	 */
	public IItemConstraintHandler createItemConstraintHandler(IConstraint constraint) throws CMDBfServiceException
	{
		return (IItemConstraintHandler)createHandler(constraint, ITEM_TEMPLATE);
	}

	
	/**
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.service.IQueryHandlerFactory#createRelationshipTemplateHandler(org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IRelationshipTemplate)
	 */
	public IRelationshipTemplateHandler createRelationshipTemplateHandler() throws CMDBfServiceException
	{
		return ((IRelationshipTemplateHandler)createHandler(null, RELATIONSHIP_TEMPLATE));
	}
	
	
	/**
	 * @see org.eclipse.cosmos.dc.cmdbf.services.query.service.IQueryHandlerFactory#createRelationshipConstraintHandler(org.eclipse.cosmos.dc.cmdbf.services.query.transform.input.artifacts.IConstraint)
	 */
	public IRelationshipConstraintHandler createRelationshipConstraintHandler(IConstraint constraint) throws CMDBfServiceException
	{
		return (IRelationshipConstraintHandler)createHandler(constraint, RELATIONSHIP_TEMPLATE);
	}
	
	
	private IQueryHandler createHandler (IConstraint constraint, int templateType) throws CMDBfServiceException
	{	
		IQueryHandler handler = null;
		int constraintType = constraint == null ? -1 : constraint.getType();
		
		// Check the cache if caching is enabled
		if (isCachingSupported())
		{
			Map<Integer, IQueryHandler> nestedMap = cachedHandlers.get(templateType);
			handler = nestedMap == null ? null : nestedMap.get(constraintType);
			if (handler != null)
			{
				return handler;
			}				
		}
		
		
		switch (constraintType)
		{
			case IConstraint.INSTANCE_ID_CONSTRAINT:
				handler = templateType == ITEM_TEMPLATE ?  (IQueryHandler)createItemInstanceHandler() :
					createRelationshipInstanceHandler();		
				break;
			case IConstraint.RECORD_TYPE:
				handler = templateType == ITEM_TEMPLATE ?  (IQueryHandler)createItemRecordHandler() :
					createRelationshipRecordHandler();		
				break;
			case IConstraint.PROPERTY_VALUE:
				handler = templateType == ITEM_TEMPLATE ?  (IQueryHandler)createItemPropertyHandler() :
					createRelationshipPropertyHandler();
				break;
			case IConstraint.XPATH_EXPRESSION:
				handler = templateType == ITEM_TEMPLATE ?  (IQueryHandler)createItemXPathHandler() :
					createRelationshipXPathHandler();
				break;	
			case -1:
				handler = templateType == ITEM_TEMPLATE ? (IQueryHandler)createItemHandler() :
					createRelationshipHandler();
				break;
			default:				
				throw new CMDBfServiceException(
						CMDBfServiceException.RECIEVER,
						CMDBfServiceException.UNSUPPORTED_CONSTRAINT,
						CMDBfMessages.faultsQueryUnsupportedConstraint);
		}
		
		if (handler != null && isCachingSupported())
		{
			Map<Integer, IQueryHandler> nestedMap = cachedHandlers.get(templateType);
			if (nestedMap == null)
			{
				nestedMap = new Hashtable<Integer, IQueryHandler>();
				cachedHandlers.put(templateType, nestedMap);
			}
			
			nestedMap.put(constraintType, handler);			
		}
		
		return handler;
	}
	

	/**
	 * Create and return an item instance id handler.  A subclass
	 * is expected to override this method if they intend to support
	 * this constraint type.
	 * 
	 * @return Instance id constraint handler
	 * @throws UnsupportedHandlerException In case this handler type is not supported
	 */
	protected IItemConstraintHandler createItemInstanceHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.INSTANCE_ID_CONSTRAINT_ELEMENT);
	}

	
	/**
	 * Create and return an item property value handler.  A subclass
	 * is expected to override this method if they intend to support
	 * this constraint type.
	 * 
	 * @return Property value constraint handler
	 * @throws UnsupportedHandlerException In case this handler type is not supported
	 */
	protected IItemConstraintHandler createItemPropertyHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.PROPERTY_VALUE_ELEMENT);
	}

	
	/**
	 * Create and return an item record type handler.  A subclass
	 * is expected to override this method if they intend to support
	 * this constraint type.
	 * 
	 * @return Record type constraint handler
	 * @throws UnsupportedHandlerException In case this handler type is not supported
	 */
	protected IItemConstraintHandler createItemRecordHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.RECORD_CONSTRAINT_ELEMENT);
	}

	
	/**
	 * Create and return an item XPath constraint handler.  A subclass
	 * is expected to override this method if they intend to support
	 * this constraint type.
	 * 
	 * @return XPath constraint handler
	 * @throws UnsupportedHandlerException In case this handler type is not supported
	 */
	protected IItemConstraintHandler createItemXPathHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.XPATH_EXPRESSION_ELEMENT);
	}

	
	/**
	 * Create and return an relationship instance id handler.  A subclass
	 * is expected to override this method if they intend to support
	 * this constraint type.
	 * 
	 * @return Instance id constraint handler
	 * @throws UnsupportedHandlerException In case this handler type is not supported
	 */
	protected IRelationshipConstraintHandler createRelationshipInstanceHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.INSTANCE_ID_CONSTRAINT_ELEMENT);
	}

	
	/**
	 * Create and return an relationship property value handler.  A subclass
	 * is expected to override this method if they intend to support
	 * this constraint type.
	 * 
	 * @return Property value constraint handler
	 * @throws UnsupportedHandlerException In case this handler type is not supported
	 */
	protected IRelationshipConstraintHandler createRelationshipPropertyHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.PROPERTY_VALUE_ELEMENT);
	}

	
	/**
	 * Create and return an relationship record type handler.  A subclass
	 * is expected to override this method if they intend to support
	 * this constraint type.
	 * 
	 * @return Record type constraint handler
	 * @throws UnsupportedHandlerException In case this handler type is not supported
	 */
	protected IRelationshipConstraintHandler createRelationshipRecordHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.RECORD_CONSTRAINT_ELEMENT);
	}

	
	/**
	 * Create and return an relationship XPath constraint handler.  A subclass
	 * is expected to override this method if they intend to support
	 * this constraint type.
	 * 
	 * @return XPath constraint handler
	 * @throws UnsupportedHandlerException In case this handler type is not supported
	 */
	protected IRelationshipConstraintHandler createRelationshipXPathHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.XPATH_EXPRESSION_ELEMENT);
	}
	
	
	/**
	 * Subclasses are expected to override this method to provide an implementation
	 * of {@link IItemTemplateHandler}
	 * 
	 * @return The item template handler
	 * @throws CMDBfServiceException In case of an error
	 */
	protected IItemTemplateHandler createItemHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.ITEM_TEMPLATE_ELEMENT);
	}
	
	
	/**
	 * Subclasses are expected to override this method to provide an implementation
	 * of {@link IRelationshipTemplateHandler}
	 * 
	 * @return The relationship template handler
	 * @throws CMDBfServiceException In case of an error
	 */
	protected IRelationshipTemplateHandler createRelationshipHandler() throws CMDBfServiceException
	{
		throw unsupportedHandler(IQueryTransformerConstants.RELATIONSHIP_TEMPLATE_ELEMENT);
	}
	
	
	private CMDBfServiceException unsupportedHandler (String handlerName)
	{
		return new CMDBfServiceException(
				CMDBfServiceException.RECIEVER,
				CMDBfServiceException.UNSUPPORTED_CONSTRAINT,
				CMDBfMessages.faultsQueryUnsupportedConstraint,
				CMDBfInternalUtility.createElement(
						ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE, 
						ICMDBfServicesConstants.CMDBF_PREFIX + IQueryTransformerConstants.CONSTRAINT, 
						new String[][]
						{
							new String[]{ICMDBfServicesConstants.NAMESPACE_ATTRIBUTE, ICMDBfServicesConstants.CMDBF_MODEL_NAMESPACE},
							new String[]{ICMDBfServicesConstants.LOCAL_NAME_ATTRIBUTE, handlerName}
						})
				);
	}
	
	/**
	 * This method enables/disables cache.  Caching of handlers
	 * is enabled by default (i.e. a handler is created once and used multiple
	 * times) but subclasses override this method to disable caching.
	 */
	protected boolean isCachingSupported()
	{
		return true;
	}
}
